% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map.R
\name{mapImage}
\alias{mapImage}
\title{Add an Image to a Map}
\usage{
mapImage(
  longitude,
  latitude,
  z,
  zlim,
  zclip = FALSE,
  breaks,
  col,
  colormap,
  border = NA,
  lwd = par("lwd"),
  lty = par("lty"),
  missingColor = NA,
  filledContour = FALSE,
  gridder = "binMean2D",
  debug = getOption("oceDebug")
)
}
\arguments{
\item{longitude}{numeric vector of longitudes corresponding to \code{z} matrix.}

\item{latitude}{numeric vector of latitudes corresponding to \code{z} matrix.}

\item{z}{numeric matrix to be represented as an image.}

\item{zlim}{limit for z (color).}

\item{zclip}{A logical value, \code{TRUE} indicating that out-of-range
\code{z} values should be painted with \code{missingColor} and \code{FALSE}
indicating that these values should be painted with the nearest
in-range color.  If \code{zlim} is given then its min and max set the
range.  If \code{zlim} is not given but \code{breaks} is given, then
the min and max of \code{breaks} sets the range used for z.  If neither
\code{zlim} nor \code{breaks} is given, clipping is not done, i.e. the
action is as if \code{zclip} were \code{FALSE}.}

\item{breaks}{The z values for breaks in the color scheme.  If this is of
length 1, the value indicates the desired number of breaks, which is
supplied to \code{\link[=pretty]{pretty()}}, in determining clean break points.}

\item{col}{Either a vector of colors corresponding to the breaks, of length
1 plus the number of breaks, or a function specifying colors,
e.g. \code{\link[=oce.colorsViridis]{oce.colorsViridis()}} for the Viridis scheme.}

\item{colormap}{optional colormap, as created by \code{\link[=colormap]{colormap()}}.
If a \code{colormap} is provided, then its properties takes precedence
over \code{breaks}, \code{col}, \code{missingColor}, and \code{zclip}
specified to \code{mapImage}.}

\item{border}{Color used for borders of patches (passed to
\code{\link[=polygon]{polygon()}}); the default \code{NA} means no border.}

\item{lwd}{line width, used if borders are drawn.}

\item{lty}{line type, used if borders are drawn.}

\item{missingColor}{a color to be used to indicate missing data, or
\code{NA} to skip the drawing of such regions (which will retain
whatever material has already been drawn at the regions).}

\item{filledContour}{either a logical value indicating whether to use
filled contours to plot the image, or a numerical value indicating the
resampling rate to be used in interpolating from lon-lat coordinates to
x-y coordinates.  See \dQuote{Details} for how this interacts with
\code{gridder}.}

\item{gridder}{Name of gridding function used if \code{filledContour} is \code{TRUE}.
This can be either \code{"binMean2D"} to select \code{\link[=binMean2D]{binMean2D()}} or \code{"interp"} to
select \code{\link[interp:interp]{interp::interp()}}.  The former produces cruder results, but the
latter can be slow for large datasets.  Note that \code{"akima"} is taken as a
synonym for \code{"interp"} (see \dQuote{Historical Note}).}

\item{debug}{A flag that turns on debugging.  Set to 1 to get a
moderate amount of debugging information, or to 2 to get more.}
}
\description{
Plot an image on an existing map that was created with \code{\link[=mapPlot]{mapPlot()}}.
}
\details{
Image data are on a regular grid in lon-lat space, but not in the projected
x-y space.  This means that \code{\link[=image]{image()}} cannot be used.  Instead,
there are two approaches, depending on the value of \code{filledContour}.

If \code{filledContour} is \code{FALSE}, the image "pixels" are drawn with \code{\link[=polygon]{polygon()}}.
This can be prohibitively slow for fine grids.  However, if \code{filledContour}
is \code{TRUE} or a numerical value, then the "pixels" are remapped into a regular
grid and then displayed with \code{\link[=.filled.contour]{.filled.contour()}}.  The remapping starts by
converting the regular lon-lat grid to an irregular x-y grid using
\code{\link[=lonlat2map]{lonlat2map()}}.  This irregular grid is then interpolated onto a regular x-y
grid  with either \code{\link[=binMean2D]{binMean2D()}} or \code{\link[interp:interp]{interp::interp()}}, as determined by the
value of the \code{gridder} parameter. If \code{filledContour} is \code{TRUE}, the
dimensions of the regular x-y grid is the same as that of the original
lon-lat grid; otherwise, the number of rows and columns are multiplied by the
numerical value of \code{filledContour}, e.g. the value 2 means to make the grid
twice as fine.

Filling contours can produce aesthetically-pleasing results, but the method
involves interpolation, so the data are not represented exactly and
analysts are advised to compare the results from the two methods (and
perhaps various grid refinement values) to guard against misinterpretation.

If a \code{\link[=png]{png()}} device is to be used, it is advised to supply
arguments \code{type="cairo"} and \code{antialias="none"} (see reference 1).
}
\section{Historical Note}{


Until oce 1.7.4, the \code{gridder} argument could be set to \code{"akima"}, which used
the \code{akima} package.  However, that package is not released with a FOSS license,
so CRAN requested a change to \CRANpkg{interp}. Note that \code{drawImage()}
intercepts the errors that sometimes get reported by \code{\link[interp:interp]{interp::interp()}}.
}

\examples{
\dontrun{
library(oce)
data(coastlineWorld)
data(topoWorld)

# Northern polar region, with color-coded bathymetry
par(mfrow=c(1,1), mar=c(2,2,1,1))
cm <- colormap(zlim=c(-5000, 0), col=oceColorsGebco)
drawPalette(colormap=cm)
mapPlot(coastlineWorld, projection="+proj=stere +lat_0=90",
        longitudelim=c(-180,180), latitudelim=c(70,110))
mapImage(topoWorld, colormap=cm)
mapGrid(15, 15, polarCircle=1, col=gray(0.2))
mapPolygon(coastlineWorld[['longitude']], coastlineWorld[['latitude']], col="tan")
}

}
\references{
\enumerate{
\item \verb{https://codedocean.wordpress.com/2014/02/03/anti-aliasing-and-image-plots/}
}
}
\seealso{
A map must first have been created with \code{\link[=mapPlot]{mapPlot()}}.

Other functions related to maps: 
\code{\link{formatPosition}()},
\code{\link{lonlat2map}()},
\code{\link{lonlat2utm}()},
\code{\link{map2lonlat}()},
\code{\link{mapArrows}()},
\code{\link{mapAxis}()},
\code{\link{mapContour}()},
\code{\link{mapCoordinateSystem}()},
\code{\link{mapDirectionField}()},
\code{\link{mapGrid}()},
\code{\link{mapLines}()},
\code{\link{mapLocator}()},
\code{\link{mapLongitudeLatitudeXY}()},
\code{\link{mapPlot}()},
\code{\link{mapPoints}()},
\code{\link{mapPolygon}()},
\code{\link{mapScalebar}()},
\code{\link{mapText}()},
\code{\link{mapTissot}()},
\code{\link{oceCRS}()},
\code{\link{shiftLongitude}()},
\code{\link{usrLonLat}()},
\code{\link{utm2lonlat}()}
}
\author{
Dan Kelley
}
\concept{functions related to maps}
