% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trajPlot.R
\name{trajPlot}
\alias{trajPlot}
\title{Trajectory line plots with conditioning}
\usage{
trajPlot(mydata, lon = "lon", lat = "lat", pollutant = "height",
  type = "default", map = TRUE, group = NA, map.fill = TRUE,
  map.res = "default", map.cols = "grey40", map.alpha = 0.4,
  projection = "lambert", parameters = c(51, 51), orientation = c(90,
  0, 0), grid.col = "deepskyblue", npoints = 12, origin = TRUE, ...)
}
\arguments{
\item{mydata}{Data frame, the result of importing a trajectory file using
\code{importTraj}.}

\item{lon}{Column containing the longitude, as a decimal.}

\item{lat}{Column containing the latitude, as a decimal.}

\item{pollutant}{Pollutant to be plotted. By default the trajectory height
is used.}

\item{type}{\code{type} determines how the data are split i.e. conditioned,
  and then plotted. The default is will produce a single plot using the
  entire data. Type can be one of the built-in types as detailed in
  \code{cutData} e.g. "season", "year", "weekday" and so on. For example,
  \code{type = "season"} will produce four plots --- one for each season.

  It is also possible to choose \code{type} as another variable in the data
  frame. If that variable is numeric, then the data will be split into four
  quantiles (if possible) and labelled accordingly. If type is an existing
  character or factor variable, then those categories/levels will be used
  directly. This offers great flexibility for understanding the variation of
  different variables and how they depend on one another.

  \code{type} can be up length two e.g. \code{type = c("season", "weekday")}
  will produce a 2x2 plot split by season and day of the week. Note, when
  two types are provided the first forms the columns and the second the
  rows.}

\item{map}{Should a base map be drawn? If \code{TRUE} the world base map
from the \code{maps} package is used.}

\item{group}{It is sometimes useful to group and colour trajectories
according to a grouping variable. See example below.}

\item{map.fill}{Should the base map be a filled polygon? Default is to fill
countries.}

\item{map.res}{The resolution of the base map. By default the function uses
the \sQuote{world} map from the \code{maps} package. If \code{map.res =
"hires"} then the (much) more detailed base map \sQuote{worldHires} from
the \code{mapdata} package is used. Use \code{library(mapdata)}. Also
available is a map showing the US states. In this case \code{map.res =
"state"} should be used.}

\item{map.cols}{If \code{map.fill = TRUE} \code{map.cols} controls the fill
colour. Examples include \code{map.fill = "grey40"} and \code{map.fill =
openColours("default", 10)}. The latter colours the countries and can help
differentiate them.}

\item{map.alpha}{The transpency level of the filled map which takes values
from 0 (full transparency) to 1 (full opacity). Setting it below 1 can
help view trajectories, trajectory surfaces etc. \emph{and} a filled base
map.}

\item{projection}{The map projection to be used. Different map projections
are possible through the \code{mapproj} package. See \code{?mapproject}
for extensive details and information on setting other parameters and
orientation (see below).}

\item{parameters}{From the \code{mapproj} package. Optional numeric vector
of parameters for use with the projection argument. This argument is
optional only in the sense that certain projections do not require
additional parameters. If a projection does not require additional
parameters then set to null i.e. \code{parameters = NULL}.}

\item{orientation}{From the \code{mapproj} package. An optional vector
c(latitude, longitude, rotation) which describes where the "North Pole"
should be when computing the projection. Normally this is c(90, 0), which
is appropriate for cylindrical and conic projections. For a planar
projection, you should set it to the desired point of tangency. The third
value is a clockwise rotation (in degrees), which defaults to the midrange
of the longitude coordinates in the map.}

\item{grid.col}{The colour of the map grid to be used. To remove the grid
set \code{grid.col = "transparent"}.}

\item{npoints}{A dot is placed every \code{npoints} along each full
trajectory. For hourly back trajectories points are plotted every
\code{npoint} hours. This helps to understand where the air masses were at
particular times and get a feel for the speed of the air (points closer
togther correspond to slower moving air masses). If \code{npoints = NA}
then no points are added.}

\item{origin}{If true a filled circle dot is shown to mark the receptor
point.}

\item{...}{other arguments are passed to \code{cutData} and
\code{scatterPlot}. This provides access to arguments used in both these
functions and functions that they in turn pass arguments on to. For
example, \code{plotTraj} passes the argument \code{cex} on to
\code{scatterPlot} which in turn passes it on to the \code{lattice}
function \code{xyplot} where it is applied to set the plot symbol size.}
}
\description{
This function plots back trajectories. This function requires that data are
imported using the \code{importTraj} function.
}
\details{
Several types of trajectory plot are available. \code{trajPlot} by default
will plot each lat/lon location showing the origin of each trajectory, if no
\code{pollutant} is supplied.

If a pollutant is given, by merging the trajectory data with concentration
data (see example below), the trajectories are colour-coded by the
concentration of \code{pollutant}. With a long time series there can be lots
of overplotting making it difficult to gauge the overall concentration
pattern. In these cases setting \code{alpha} to a low value e.g. 0.1 can
help.

The user can aslo show points instead of lines by \code{plot.type = "p"}.

Note that \code{trajPlot} will plot only the full length trajectories. This
should be remembered when selecting only part of a year to plot.
}
\examples{

# show a simple case with no pollutant i.e. just the trajectories
# let's check to see where the trajectories were coming from when
# Heathrow Airport was closed due to the Icelandic volcanic eruption
# 15--21 April 2010.
# import trajectories for London and plot
\dontrun{
lond <- importTraj("london", 2010)
# well, HYSPLIT seems to think there certainly were conditions where trajectories
# orginated from Iceland...
trajPlot(selectByDate(lond, start = "15/4/2010", end = "21/4/2010"))}

# plot by day, need a column that makes a date
\dontrun{
lond$day <- as.Date(lond$date)
trajPlot(selectByDate(lond, start = "15/4/2010", end = "21/4/2010"),
type = "day")
}

# or show each day grouped by colour, with some other options set
\dontrun{
 trajPlot(selectByDate(lond, start = "15/4/2010", end = "21/4/2010"),
group = "day", col = "jet", lwd = 2, key.pos = "right", key.col = 1)
}
# more examples to follow linking with concentration measurements...

}
\seealso{
\code{\link{importTraj}} to import trajectory data from the King's
  College server and \code{\link{trajLevel}} for trajectory binning
  functions.
}
\author{
David Carslaw
}
