## Part of the pathviewr package
## Last updated: 2020-09-11 VBB


################################# import_batch #################################

#' Batch import of files for either Motive or Flydra (but not a mix of both)
#'
#' @param file_path_list A list of file paths
#' @param import_method Either "flydra" or "motive"
#' @param file_id Optional
#' @param subject_name For Flydra, the assigned subject name
#' @param frame_rate For Flydra, the assigned frame rate
#' @param simplify_marker_naming default TRUE; for Motive, whether marker naming
#' should be simplified
#' @param import_messaging default FALSE; should this function report each time
#' a file has been imported?
#' @param ... Additional arguments (may remove this if needed)
#'
#' @details Refer to \code{read_motive_csv()} and \code{read_flydra_mat()} for
#'   details of data import methods.
#'
#' @return A list of viewr objects (tibble or data.frame with attribute
#'   \code{pathviewr_steps} that includes \code{"viewr"}).
#'
#' @author Vikram B. Baliga
#'
#' @family data import functions
#' @family batch functions
#'
#' @export
#'
#' @examples
#' ## Since we only have one example file of each type provided
#' ## in pathviewr, we will simply import the same example multiple
#' ## times to simulate batch importing. Replace the contents of
#' ## the following list with your own list of files to be imported.
#'
#' ## Make a list of the same example file 3x
#' import_list <-
#'   c(rep(
#'     system.file("extdata", "pathviewr_motive_example_data.csv",
#'                 package = 'pathviewr'),
#'     3
#'   ))
#'
#' ## Batch import
#' motive_batch_imports <-
#'   import_batch(import_list,
#'                import_method = "motive",
#'                import_messaging = TRUE)
#'
#' ## Batch cleaning of these imported files
#' ## via clean_viewr_batch()
#' motive_batch_cleaned <-
#'   clean_viewr_batch(
#'     file_announce = TRUE,
#'     motive_batch_imports,
#'     desired_percent = 50,
#'     max_frame_gap = "autodetect",
#'     span = 0.95
#'   )
#'
#' ## Alternatively, use import_and_clean_batch() to
#' ## combine import with cleaning on a batch of files
#' motive_batch_import_and_clean <-
#'   import_and_clean_batch(
#'     import_list,
#'     import_method = "motive",
#'     import_messaging = TRUE,
#'     motive_batch_imports,
#'     desired_percent = 50,
#'     max_frame_gap = "autodetect",
#'     span = 0.95
#'   )
#'
#' ## Each of these lists of objects can be bound into
#' ## one viewr object (i.e. one tibble) via
#' ## bind_viewr_objects()
#' motive_bound_one <-
#'   bind_viewr_objects(motive_batch_cleaned)
#'
#' motive_bound_two <-
#'   bind_viewr_objects(motive_batch_import_and_clean)
#'
#' ## Either route results in the same object ultimately:
#' identical(motive_bound_one, motive_bound_two)


import_batch <- function(file_path_list,
                         import_method = c("flydra", "motive"),
                         file_id = NA,
                         subject_name = NULL,
                         frame_rate = NULL,
                         simplify_marker_naming = TRUE,
                         import_messaging = FALSE,
                         ...){

  # if ("list" %in% class(file_path_list)){
  #   stop("file_path_list must be a list of file locations")
  # }

  imp_meth <- match.arg(import_method)

  ## Object to be filled
  obj_list <- NULL

  if (imp_meth == "flydra"){
    for (i in seq_len(length(file_path_list))){
      obj_list[[i]] <- read_flydra_mat(
        mat_file = file_path_list[i],
        subject_name = subject_name[i],
        frame_rate = 100)
      if (import_messaging == TRUE)
        message("File ", i, " imported.")
    }
  }

  if (imp_meth == "motive"){
    for (i in seq_len(length(file_path_list))){
      obj_list[[i]] <- read_motive_csv(
        file_name = file_path_list[i],
        simplify_marker_naming = simplify_marker_naming)
      if (import_messaging == TRUE)
        message("File ", i, " imported.")
    }
  }

  ## Export
  return(obj_list)
}


################################# clean_viewr_batch ############################

#' Batch clean viewr files
#'
#' For a list of viewr objects, run through the pipeline (from relabel axes
#' up through get full trajectories, as desired) via clean_viewr()
#'
#' @param obj_list A list of viewr objects (i.e. a list of tibbles that each
#'   have attribute \code{pathviewr_steps} that includes \code{"viewr"})
#' @param file_announce Should the function report each time a file is
#'   processed? Default FALSE; if TRUE, a message will appear in the console
#'   each time a file has been cleaned successfully.
#' @param ... Arguments to be passed in that specify how this function should
#'   clean files.
#'
#' @details viewr objects should be in a list, e.g. the object generated by
#'   \code{import_batch()}.
#'
#'   See \code{clean_viewr()} for details of how cleaning steps are handled
#'   and/or refer to the corresponding cleaning functions themselves.
#'
#' @return A list of viewr objects (tibble or data.frame with attribute
#'   \code{pathviewr_steps} that includes \code{"viewr"}) that have been passed
#'   through the corresponding cleaning functions.
#'
#' @author Vikram B. Baliga
#'
#' @family batch functions
#'
#' @export
#'
#' @inherit import_batch examples

clean_viewr_batch <- function(obj_list,
                              file_announce = FALSE,
                              ...) {

  if (!"list" %in% class(obj_list)){
    stop("obj_list must be a list of file locations")
  }

  cleaned_list <- NULL
  for (i in seq_len(length(obj_list))){
    cleaned_list[[i]] <-
      obj_list[[i]] %>%
      clean_viewr(...)
    if (file_announce == TRUE){
      message("File ",i," has been cleaned successfully.")
    }
    }

  #cleaned_list <- lapply(obj_list, FUN = clean_viewr, ...)

  ## Export
  return(cleaned_list)
}


############################ import_and_clean_batch ############################

#' Batch import and clean files
#'
#' Like \code{clean_viewr_batch()}, but with import as the first step too
#'
#' @param file_path_list A list of file paths leading to files to be imported.
#' @param import_method Either "flydra" or "motive"
#' @inheritParams read_flydra_mat
#' @inheritParams read_motive_csv
#' @param subject_name For Flydra, the subject name applied to all files
#' @param frame_rate For Flydra, the frame rate applied to all files
#' @param simplify_marker_naming For Motive, if Markers are encountered, should
#'  they be renamed from "Subject:marker" to "marker"? Defaults to TRUE
#' @param import_messaging Should this function report each time a file has been
#' processed?
#' @param ... Additional arguments to specify how data should be cleaned.
#'
#' @inherit clean_viewr_batch details
#'
#' @return A list of viewr objects (tibble or data.frame with attribute
#'   \code{pathviewr_steps} that includes \code{"viewr"}) that have been passed
#'   through the corresponding cleaning functions.
#'
#' @author Vikram B. Baliga
#'
#' @family data import functions
#' @family batch functions
#'
#' @export
#'
#' @inherit import_batch examples

import_and_clean_batch <- function(file_path_list,
                                   import_method = c("flydra", "motive"),
                                   file_id = NA,
                                   subject_name = NULL,
                                   frame_rate = NULL,
                                   simplify_marker_naming = TRUE,
                                   import_messaging = FALSE,
                                   ...){
  ## Revise this:
  # if (!"list" %in% class(file_path_list)){
  #   stop("file_path_list must be a list of file locations")
  # }

  imp_meth <- match.arg(import_method)

  ## Object to be filled
  obj_list <- NULL

  if (imp_meth == "flydra"){
    for (i in seq_len(length(file_path_list))){
      obj_list[[i]] <- read_flydra_mat(
        mat_file = file_path_list[i],
        subject_name = subject_name[i],
        frame_rate = frame_rate)
      if (import_messaging == TRUE)
        message("File ", i, " imported.")
    }
  }

  if (imp_meth == "motive"){
    for (i in seq_len(length(file_path_list))){
      obj_list[[i]] <- read_motive_csv(
        file_name = file_path_list[i],
        simplify_marker_naming = simplify_marker_naming)
      if (import_messaging == TRUE)
        message("File ", i, " imported.")
    }
  }

  ## Now clean
  obj_cleaned <- clean_viewr_batch(obj_list, ...)

  ## Export
  return(obj_cleaned)
}


############################### bind_viewr_objects #############################

#' Bind viewr objects
#'
#' Combine a list of multiple viewr objects into a single viewr object
#'
#' @param obj_list A list of viewr objects
#'
#' @return A single viewr object (tibble or data.frame with attribute
#'   \code{pathviewr_steps} that includes \code{"viewr"}) that combines all the
#'   rows of the source viewr objects in \code{obj_list}. Metadata may not
#'   necessarily be retained and therefore \code{attributes} should be used with
#'   caution.
#'
#' @author Vikram B. Baliga
#'
#' @family batch functions
#'
#' @export
#'
#' @inherit import_batch examples

bind_viewr_objects <- function(obj_list) {

  if (!"list" %in% class(obj_list)){
    stop("obj_list must be a list of viewr objects")
  }

  ## THIS FUNCTION LIKELY NEEDS TO BE EDITED TO ACCOMODATE ATTRIBUTES FROM
  ## EACH OBJECT. Not sure what exactly should be retained from each and how
  ## it all needs to be strung together yet...

  ## Combine the objects
  bound_obj <- dplyr::bind_rows(obj_list)

  ## Attributes
  attr(bound_obj, "pathviewr_steps") <- c("viewr", "bound_viewr_objects")

  ## Export
  return(bound_obj)

}
