% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_factorstructure.R
\name{check_factorstructure}
\alias{check_factorstructure}
\alias{check_kmo}
\alias{check_sphericity_bartlett}
\title{Check suitability of data for Factor Analysis (FA) with Bartlett's Test of Sphericity and KMO}
\usage{
check_factorstructure(x, n = NULL, ...)

check_kmo(x, n = NULL, ...)

check_sphericity_bartlett(x, n = NULL, ...)
}
\arguments{
\item{x}{A dataframe or a correlation matrix. If the latter is passed, \code{n}
must be provided.}

\item{n}{If a correlation matrix was passed, the number of observations must
be specified.}

\item{...}{Arguments passed to or from other methods.}
}
\value{
A list of lists of indices related to sphericity and KMO.
}
\description{
This checks whether the data is appropriate for Factor Analysis (FA) by
running the Bartlett's Test of Sphericity and the Kaiser, Meyer, Olkin (KMO)
Measure of Sampling Adequacy (MSA). See \strong{details} below for more information
about the interpretation and meaning of each test.
}
\details{
\subsection{Bartlett's Test of Sphericity}{

Bartlett's (1951) test of sphericity tests whether a matrix (of correlations)
is significantly different from an identity matrix (filled with 0). It tests
whether the correlation coefficients are all 0. The test computes the
probability that the correlation matrix has significant correlations among at
least some of the variables in a dataset, a prerequisite for factor analysis
to work.

While it is often suggested to check whether Bartlett’s test of sphericity is
significant before starting with factor analysis, one needs to remember that
the test is testing a pretty extreme scenario (that all correlations are non-significant).
As the sample size increases, this test tends to be always significant, which
makes it not particularly useful or informative in well-powered studies.
}

\subsection{Kaiser, Meyer, Olkin (KMO)}{

\emph{(Measure of Sampling Adequacy (MSA) for Factor Analysis.)}

Kaiser (1970) introduced a Measure of Sampling Adequacy (MSA), later modified
by Kaiser and Rice (1974). The Kaiser-Meyer-Olkin (KMO) statistic, which can
vary from 0 to 1, indicates the degree to which each variable in a set is
predicted without error by the other variables.

A value of 0 indicates that the sum of partial correlations is large relative
to the sum correlations, indicating factor analysis is likely to be
inappropriate. A KMO value close to 1 indicates that the sum of partial
correlations is not large relative to the sum of correlations and so factor
analysis should yield distinct and reliable factors. It means that patterns
of correlations are relatively compact, and so factor analysis should yield
distinct and reliable factors. Values smaller than 0.5 suggest that you should
either collect more data or rethink which variables to include.

Kaiser (1974) suggested that KMO > .9 were marvelous, in the .80s,
meritorious, in the .70s, middling, in the .60s, mediocre, in the .50s,
miserable, and less than .5, unacceptable. Hair et al. (2006) suggest
accepting a value > 0.5. Values between 0.5 and 0.7 are mediocre, and values
between 0.7 and 0.8 are good.

Variables with individual KMO values below 0.5 could be considered for
exclusion them from the analysis (note that you would need to re-compute the
KMO indices as they are dependent on the whole dataset).
}
}
\examples{
library(performance)

check_factorstructure(mtcars)

# One can also pass a correlation matrix
r <- cor(mtcars)
check_factorstructure(r, n = nrow(mtcars))

}
\references{
This function is a wrapper around the \code{KMO} and the \code{cortest.bartlett()}
functions in the \strong{psych} package (Revelle, 2016).
\itemize{
\item Revelle, W. (2016). How To: Use the psych package for Factor Analysis
and data reduction.
\item Bartlett, M. S. (1951). The effect of standardization on a Chi-square
approximation in factor analysis. Biometrika, 38(3/4), 337-344.
\item Kaiser, H. F. (1970). A second generation little jiffy.
Psychometrika, 35(4), 401-415.
\item Kaiser, H. F., & Rice, J. (1974). Little jiffy, mark IV. Educational
and psychological measurement, 34(1), 111-117.
\item Kaiser, H. F. (1974). An index of factorial simplicity.
Psychometrika, 39(1), 31-36.
}
}
\seealso{
\code{\link[=check_clusterstructure]{check_clusterstructure()}}.
}
