% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prior_spec.R
\name{prior specification}
\alias{prior specification}
\title{prior distribution}
\description{
Specification of prior distributions.
}
\details{
A prior distribution on parameters is specified by means of the \code{rprior} and/or \code{dprior} arguments to \code{pomp}.
As with the other \link[=basic components]{basic model components}, it is preferable to specify these using C snippets.
In writing a C snippet for the prior sampler (\code{rprior}), keep in mind that:
\enumerate{
  \item Within the context in which the snippet will be evaluated, only the parameters will be defined.
  \item The goal of such a snippet is the replacement of parameters with values drawn from the prior distribution.
  \item Hyperparameters can be included in the ordinary parameter list.
  Obviously, hyperparameters should not be replaced with random draws.
}
In writing a C snippet for the prior density function (\code{dprior}), observe that:
\enumerate{
  \item Within the context in which the snippet will be evaluated, only the parameters and \code{give_log} will be defined.
  \item The goal of such a snippet is computation of the prior probability density, or the log of same, at a given point in parameter space.
  This scalar value should be returned in the variable \code{lik}.
  When \code{give_log == 1}, \code{lik} should contain the log of the prior probability density.
  \item Hyperparameters can be included in the ordinary parameter list.
}
\link[=Csnippet]{General rules for writing C snippets can be found here}.

Alternatively, one can furnish \R functions for one or both of these arguments.
In this case, \code{rprior} must be a function that makes a draw from
the prior distribution of the parameters and returns a named vector
containing all the parameters.
The only required argument of this function is \code{...}.

Similarly, the \code{dprior} function must evaluate the prior probability
density (or log density if \code{log == TRUE}) and return that single
scalar value.
The only required arguments of this function are \code{...} and \code{log}.
}
\section{Default behavior}{

By default, the prior is assumed flat and improper.
In particular, \code{dprior} returns \code{1} (\code{0} if \code{log = TRUE}) for every parameter set.
Since it is impossible to simulate from a flat improper prior, \code{rprocess} returns missing values (\code{NA}s).
}

\section{Note for Windows users}{


Some Windows users report problems when using C snippets in parallel computations.
These appear to arise when the temporary files created during the C snippet compilation process are not handled properly by the operating system.
To circumvent this problem, use the \code{\link[=pomp]{cdir}} and \code{\link[=pomp]{cfile}} options to cause the C snippets to be written to a file of your choice, thus avoiding the use of temporary files altogether.
}

\examples{
\dontrun{
  ## Starting with an existing pomp object

  verhulst() \%>\% window(end=30) -> po
  
  ## we add or change prior distributions using the two
  ## arguments 'rprior' and 'dprior'. Here, we introduce
  ## a Gamma prior on the 'r' parameter.
  ## We construct 'rprior' and 'dprior' using R functions.

  po \%>\%
    bsmc2(
      rprior=function (n_0, K0, K1, sigma, tau, r0, r1, ...) {
        c(
          n_0 = n_0,
          K = rgamma(n=1,shape=K0,scale=K1),
          r = rgamma(n=1,shape=r0,scale=r1),
          sigma = sigma,
          tau = tau
        )
      },
      dprior=function(K, K0, K1, r, r0, r1, ..., log) {
        p <- dgamma(x=c(K,r),shape=c(K0,r0),scale=c(K1,r1),log=log)
        if (log) sum(p) else prod(p)
      },
      params=c(n_0=10000,K=10000,K0=10,K1=1000,
        r=0.9,r0=0.9,r1=1,sigma=0.5,tau=0.3),
      Np=1000
    ) -> B

  ## We can also pass them as C snippets:

  po \%>\%
    bsmc2(
      rprior=Csnippet("
         K = rgamma(K0,K1);
         r = rgamma(r0,r1);"
      ),
      dprior=Csnippet("
         double lik1 = dgamma(K,K0,K1,give_log);
         double lik2 = dgamma(r,r0,r1,give_log);
         lik = (give_log) ? lik1+lik2 : lik1*lik2;"
      ),
      paramnames=c("K","K0","K1","r","r0","r1"),
      params=c(n_0=10000,K=10000,K0=10,K1=1000,
        r=0.9,r0=0.9,r1=1,sigma=0.5,tau=0.3),
      Np=10000
    ) -> B

  ## The prior is plotted in grey; the posterior, in blue.
  plot(B)

  B \%>\%
    pmcmc(Nmcmc=100,Np=1000,proposal=mvn.diag.rw(c(r=0.01,K=10))) -> Bb

  plot(Bb,pars=c("loglik","log.prior","r","K"))

}
}
\seealso{
\code{\link{dprior}} \code{\link{rprior}}

More on implementing POMP models: 
\code{\link{Csnippet}},
\code{\link{accumulator variables}},
\code{\link{basic components}},
\code{\link{covariates}},
\code{\link{distributions}},
\code{\link{dmeasure specification}},
\code{\link{dprocess specification}},
\code{\link{parameter transformations}},
\code{\link{pomp-package}},
\code{\link{pomp}},
\code{\link{rinit specification}},
\code{\link{rmeasure specification}},
\code{\link{rprocess specification}},
\code{\link{skeleton specification}},
\code{\link{transformations}},
\code{\link{userdata}}

More on Bayesian methods:
\code{\link{approximate Bayesian computation}},
\code{\link{bsmc2}()},
\code{\link{dprior}()},
\code{\link{pmcmc}()},
\code{\link{rprior}()}
}
\concept{Bayesian methods}
\concept{implementation information}
