% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tfam_lambda.R
\name{tfam_lambda}
\alias{tfam_lambda}
\title{Transfer function analysis}
\usage{
tfam_lambda(A, elementtype = NULL, Flim = c(-1, 10), Plim = c(-1, 10),
  plength = 100, digits = 1e-10)
}
\arguments{
\item{A}{a square, irreducible, nonnegative numeric matrix of any dimension}

\item{elementtype}{(optional) a character matrix of the same dimension as 
\code{A} describing the structure of \code{A}: \code{"P"} denotes elements 
bounded between 0 and 1, i.e. survival, growth, regression; \code{"F"} 
denotes elements not bounded at 1, i.e. fecundity, fission; \code{NA} 
denotes absent elements (see details).}

\item{Flim, Plim}{the perturbation ranges for \code{"F"} and \code{"P"} 
elements, expressed as a proportion of their magnitude  (see details).}

\item{plength}{the desired length of the perturbation ranges.}

\item{digits}{specifies which values of lambda should be excluded from 
analysis to avoid a computationally singular system (see details).}
}
\value{
A list containing numerical arrays:
\describe{
\item{p}{perturbation magnitudes}
\item{lambda}{dominant eigenvalues of perturbed matrices}
}
The first and second dimensions of the arrays are equivalent to
the first and second dimensions of \code{A}. The third dimension of the
arrays are the vectors returned by \code{tfa_lambda}. e.g. $lambda[3,2,] selects
the lambda values for the transfer function of element [3,2] of the matrix.
}
\description{
Transfer function analysis of the dominant eigenvalue of a population matrix 
projection model for all matrix elements.
}
\details{
\code{tfam_lambda} calculates an array of transfer functions of the dominant
eigenvalue of \code{A}. A separate transfer function for each nonzero 
element of \code{A} is calculated (each element perturbed independently of 
the others). The function is most useful for use with the S3 method 
\code{\link{plot.tfam}} to visualise how perturbations affect the 
life cycle transitions, and easily compare the (nonlinear) effect of 
perturbation to different transitions on the dominant eigenvalue.

The sizes of the perturbations are determined by \code{elementtype}, 
\code{Flim}, \code{Plim} and \code{plength}. \code{elementtype} gives the 
type of each element, specifying whether perturbations should be 
bounded at 1 (\code{elementtype = "P"}) or not (\code{elementtype = "F"}). 
If \code{elementtype} is not directly specified, the function assigns its 
own types, with those in the first row attributed \code{"F"}, and elsewhere 
in the matrix attributed \code{"F"} if the value of the element >1 and 
\code{"P"} if the value of the element is <=1. \code{Flim} and \code{Plim} 
determine the desired perturbation magnitude, expressed as a proportion of 
the magnitude of the elements of \code{A}, whilst plength determines the 
length of the perturbation vector.  For example, if an "F" element is equal 
to 0.5, \code{Flim=c(-1,10)} and \code{plength=100} then the perturbation 
to that element is \code{seq(-1*0.5,10*0.5,100-1)}. The process is the same 
for \code{"P"} elements, except that these are truncated to a maximum value 
of 1 (growth/survival elements cannot be greater than 1). Both \code{"F"} 
and \code{"P"} elements are truncated to a minimum value of 0.

\code{tfam_lambda} uses \code{\link{tfa_lambda}} to calculate transfer 
functions. \code{digits} is passed to \code{tfa_lambda} to prevent the 
problem of singular matrices (see details in \code{\link{tfa_lambda}}).

\code{tfam_lambda} will not work for reducible matrices.
}
\examples{
  # Create a 3x3 matrix
  ( A <- matrix(c(0,1,2,0.5,0.1,0,0,0.6,0.6), byrow=TRUE, ncol=3) )

  # Calculate the matrix of transfer functions using default arguments
  ( tfmat<-tfam_lambda(A) )

  # Plot the result using the S3 method
  plot(tfmat)

  # Plot the transfer function of element [3,2] without using the S3 method
  par(mfrow=c(1,1))
  par(mar=c(5,4,4,2)+0.1)
  plot(tfmat$lambda[3,2,]~tfmat$p[3,2,],xlab="p",ylab="lambda",type="l")

  # Create a new matrix with fission of adults
  B <- A; B[2,3] <- 0.9; B

  # Calculate the matrix of transfer functions using chosen arguments
  # that give the exact structure of the new matrix
  # and perturb a minimum of half the value of an element and
  # a maximum of double the value of an element
  ( etype <- matrix(c(NA, "F", "F", "P", "P", "F", NA, "P", "P"), 
                  ncol=3, byrow=TRUE) )
  ( tfmat2 <- tfam_lambda(B, elementtype=etype, Flim=c(-0.5,2),
                      Plim=c(-0.5,2)) )

  # Plot the new matrix of transfer functions using the S3 method
  plot(tfmat2)
    
}
\references{
\itemize{
 \item Townley & Hodgson (2004) J. Appl. Ecol., 41, 1155-1161.
 \item Hodgson et al. (2006) J. Theor. Biol., 70, 214-224.
}
}
\seealso{
S3 plotting method: \code{\link{plot.tfa}}

Other TransferFunctionAnalyses: \code{\link{tfa_inertia}},
  \code{\link{tfa_lambda}}, \code{\link{tfam_inertia}},
  \code{\link{tfs_inertia}}, \code{\link{tfs_lambda}}

Other PerturbationAnalyses: \code{\link{elas}},
  \code{\link{sens}}, \code{\link{tfa_inertia}},
  \code{\link{tfa_lambda}}, \code{\link{tfam_inertia}},
  \code{\link{tfs_inertia}}, \code{\link{tfs_lambda}}
}
\concept{
transfer function systems control nonlinear perturbation population viability
PVA ecology demography PPM MPM
}
