% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/backtestLeaderboard.R
\name{backtestLeaderboard}
\alias{backtestLeaderboard}
\title{Leaderboard of portfolios from the backtest results.}
\usage{
backtestLeaderboard(bt = NA, weights = list(), summary_fun = median,
  show_benchmark = TRUE)
}
\arguments{
\item{bt}{Backtest results as produced by the function \code{\link{portfolioBacktest}}.}

\item{weights}{List of weights for the different performance measures as obtained 
in \code{\link{backtestSummary}()$performance} (i.e., 
\code{"Sharpe ratio"}, \code{"max drawdown"}, \code{"annual return"}, \code{"annual volatility"}, 
\code{"Sterling ratio"}, \code{"Omega ratio"}, \code{"ROT bps"},
\code{"cpu_time"}, and \code{"failure ratio"}), as well as
\code{"cpu time"} and \code{"failure rate"}.
For example: \code{weights = list("Sharpe ratio" = 8, "max drawdown" = 4)}.}

\item{summary_fun}{Summary function to be employed (e.g., \code{median} or \code{mean}).}

\item{show_benchmark}{Logical value indicating whether to include benchmarks in the summary (default is \code{TRUE}).}
}
\value{
List with the following elements:
\item{\code{leaderboard_scores}}{Matrix with the individual scores for the portfolios (as chosen in \code{weights}) and the final score.}
\item{\code{leaderboard_performance}}{Matrix with all the performance measures for the portfolios.}
\item{\code{error_summary}}{Error messages generated by each portfolio on each dataset. 
                            Useful for debugging and give feedback to the portfolio managers of the 
                            different portfolios.}
}
\description{
Leaderboard of portfolios according to the backtesting results 
and a ranking based on the combination of several performance criteria.
Since the different performance measures hava different ranges and distributions, 
each is first transformed according to its empirical distribution function (along 
the empirical distribution of the portfolios being ranked) to obtain percentile 
scores. After that transformation, each of the measures has an empirical uniform 
distribution in the interval \code{[0, 100]} and can be weighted to obtain the final ranking.
}
\examples{
\donttest{ 
library(portfolioBacktest)
data(dataset10)  # load dataset

# define your own portfolio function
quintile_portfolio <- function(data) {
  X <- diff(log(data$adjusted))[-1]  
  N <- ncol(X)
  ranking <- sort(colMeans(X), decreasing = TRUE, index.return = TRUE)$ix
  w <- rep(0, N)
  w[ranking[1:round(N/5)]] <- 1/round(N/5)
  return(w)
}

# do backtest
bt <- portfolioBacktest(list("Quintile" = quintile_portfolio), dataset10,
                        benchmark = c("uniform", "index"))

# see all performance measures available for the ranking
backtestSummary(bt)$performance

# show leaderboard
leaderboard <- backtestLeaderboard(bt, weights = list("Sharpe ratio"  = 6,
                                                      "max drawdown"  = 1,
                                                      "ROT bps"       = 1,
                                                      "cpu time"      = 1,
                                                      "failure rate"  = 1))
leaderboard$leaderboard_scores
}

}
\author{
Daniel P. Palomar and Rui Zhou
}
