## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)
set.seed(1)
library(rxode2)
setRxThreads(2L)  # limit the number of threads

## ----setup--------------------------------------------------------------------
library(posologyr)

## -----------------------------------------------------------------------------
mod_warfarin_nlmixr <- function() {
    ini({
      #Fixed effects: population estimates 
      THETA_ktr=0.106
      THETA_ka=-0.087
      THETA_cl=-2.03
      THETA_v=2.07
      THETA_emax=3.4
      THETA_ec50=0.00724
      THETA_kout=-2.9
      THETA_e0=4.57
      
      #Random effects: inter-individual variability
      ETA_ktr ~ 1.024695
      ETA_ka ~ 0.9518403
      ETA_cl ~ 0.5300943
      ETA_v ~ 0.4785394
      ETA_emax ~ 0.7134424
      ETA_ec50 ~ 0.7204165
      ETA_kout ~ 0.3563706
      ETA_e0 ~ 0.2660827
      
      #Unexplained residual variability
      cp.sd <- 0.144
      cp.prop.sd <- 0.15
      pca.sd <- 3.91
    })
    model({
      #Individual model and covariates
      ktr <- exp(THETA_ktr + ETA_ktr)
      ka <- exp(THETA_ka + ETA_ka)
      cl <- exp(THETA_cl + ETA_cl)
      v <- exp(THETA_v + ETA_v)
      emax = expit(THETA_emax + ETA_emax)
      ec50 =  exp(THETA_ec50 + ETA_ec50)
      kout = exp(THETA_kout + ETA_kout)
      e0 = exp(THETA_e0 + ETA_e0)
      
      #Structural model defined using ordinary differential equations (ODE)
      DCP = center/v
      PD=1-emax*DCP/(ec50+DCP)
      
      effect(0) = e0
      kin = e0*kout
      
      d/dt(depot) = -ktr * depot
      d/dt(gut) =  ktr * depot -ka * gut
      d/dt(center) =  ka * gut - cl / v * center
      d/dt(effect) = kin*PD -kout*effect
      
      cp = center / v
      pca = effect
      
      #Model for unexplained residual variability
      cp ~ add(cp.sd) + prop(cp.prop.sd)
      pca ~ add(pca.sd)
    })
  }

## -----------------------------------------------------------------------------
warf_01 <- data.frame(ID=1,
                      TIME=c(0.0,1.0,3.0,6.0,24.0,24.0,36.0,36.0,48.0,48.0,72.0,72.0,144.0),
                      DV=c(0.0,1.9,6.6,10.8,5.6,44.0,4.0,27.0,2.7,28.0,0.8,31.0,71.0),
                      DVID=c("cp","cp","cp","cp","cp","pca","cp","pca","cp","pca","cp","pca","pca"),
                      EVID=c(1,0,0,0,0,0,0,0,0,0,0,0,0),
                      AMT=c(100,0,0,0,0,0,0,0,0,0,0,0,0))
warf_01

## -----------------------------------------------------------------------------
map_warf_01  <- poso_estim_map(warf_01,mod_warfarin_nlmixr)
map_warf_01

## -----------------------------------------------------------------------------
plot(map_warf_01$model,"cp")

## -----------------------------------------------------------------------------
plot(map_warf_01$model,"pca")

