% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prais_winsten.R, R/print.prais.R
\name{prais_winsten}
\alias{prais_winsten}
\alias{print.prais}
\title{Prais-Winsten Estimator for AR(1) Serial Correlation}
\usage{
prais_winsten(
  formula,
  data,
  index,
  max_iter = 50L,
  tol = 1e-06,
  twostep = FALSE,
  panelwise = FALSE,
  rhoweight = c("none", "T", "T1"),
  ...
)

\method{print}{prais}(x, digits = max(3L, getOption("digits") - 3L), ...)
}
\arguments{
\item{formula}{an object of class \code{"formula"} (or one that can be coerced to that class):
a symbolic description of the model to be fitted.}

\item{data}{a data frame containing the variables in the model. If panel data is used,
it must also contain the ID and time variables.}

\item{index}{a character vector specifying the ID and time variables. If only one variable
is provided, it is assumed to be the time variable and the data will be reordered
accordingly.}

\item{max_iter}{integer specifying the maximum number of allowed iterations. Default is 50.}

\item{tol}{numeric specifying the maximum absolute difference between the estimator of \eqn{rho}
in the current and the previous iteration that has to be attained to reach convergence.
Default is 1e-6.}

\item{twostep}{logical. If \code{TRUE}, the estimation will stop after the first iteration.}

\item{panelwise}{logical. If \code{TRUE}, \eqn{\rho} will be calculated for each panel separately.
Default is \code{FALSE}. Only used for panel data. See 'Details'.}

\item{rhoweight}{character specifying how \eqn{\rho} should be calculated if \code{panelwise = TRUE}.
See 'Details'.}

\item{...}{arguments passed to \code{\link[stats]{lm}}.}

\item{x}{an object of class "prais", usually, a result of a call to \code{\link{prais_winsten}}.}

\item{digits}{the number of significant digits to use when printing.}
}
\value{
A list of class \code{"prais"} containing the following components:
\item{coefficients}{a named vector of coefficients.}
\item{rho}{the values of the AR(1) coefficient \eqn{\rho} from all iterations.}
\item{residuals}{the residuals, that is the response minus the fitted values.}
\item{fitted.values}{the fitted mean values.}
\item{rank}{the numeric rank of the fitted linear model.}
\item{df.residual}{the residual degrees of freedom.}
\item{call}{the matched call.}
\item{terms}{the terms object used.}
\item{model}{the original model frame, i.e., before the Prais-Winsten transformation.}
\item{index}{a character specifying the ID and time variables.}
}
\description{
The Prais-Winsten estimator takes into account AR(1) serial correlation of the errors
in a linear regression model. The procedure recursively estimates the coefficients
and the error autocorrelation of the specified model until sufficient convergence of
the AR(1) coefficient is reached. All estimates are obtained by OLS.
}
\details{
If \eqn{\rho} takes a value above 1 during the estimation process,
the Prais-Winsten transformation cannot be applied to the first
observations, because \eqn{(1 - \rho^2)^{(1 / 2)}} is not real. These observations
are dropped during the respective iteration and the estimator effectively becomes
the Cochrane-Orcutt estimator.

If \code{panelwise = TRUE}, \code{twostep = FALSE} and \code{rhoweight = "none"},
each individual estimate of \eqn{rho} is re-estimated until convergence is achieved for all coefficients.

If \code{panelwise = TRUE}, the calculation of \eqn{\rho} can be further specified in argument
\code{rhoweight}. If \code{rhoweight = "none"}, \eqn{\rho} is assumed to be panel-specific. If
\code{rhoweight = "T"}, \eqn{\rho} is calculated as a weighted mean of panel-specific estimates, where
the number of available observations per panel, i.e. \eqn{T_i}, is used as weight. If \code{rhoweight = "T1"},
\eqn{\rho} is calculated as a weighted mean of panel-specific estimates, where the number of available
observations per panel minus one, i.e. \eqn{T_i - 1}, is used as weight.
}
\examples{
# Generate an artificial sample
set.seed(1234567)
n <- 100
x <- sample(20:40, n, replace = TRUE)
rho <- .5

# AR(1) errors
u <- rnorm(n, 0, 5)
for (i in 2:n) {
  u[i] <- u[i] + rho * u[i - 1]
}
pw_sample <- data.frame("x" = x, "y" = 10 + 1.5 * x + u, "time" = 1:n)

# Estimate
pw <- prais_winsten(y ~ x, data = pw_sample, index = "time")
summary(pw)

}
\references{
Beck, N. L. and Katz, J. N. (1995): What to do (and not to do) with time-series cross-section data. American Political Science Review 89, 634-647.

Prais, S. J. and Winsten, C. B. (1954): Trend Estimators and Serial Correlation. Cowles Commission Discussion Paper, 383 (Chicago).

Wooldridge, J. M. (2013): Introductory Econometrics. A Modern Approach. 5th ed. Mason, OH: South-Western Cengage Learning Cengage.
}
