% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions_predkmeans.R
\name{predkmeans}
\alias{predkmeans}
\title{Predictive K-means Clustering}
\usage{
predkmeans(X, R, K, mu = NULL, muStart = c("kmeans", "random"),
  sigma2 = 0, sigma2fixed = FALSE, maxitEM = 100, tol = 1e-05,
  convEM = c("both", "mu", "gamma"), nStarts = 1, maxitMlogit = 500,
  verbose = 0, muRestart = 1000, returnAll = FALSE, ...)
}
\arguments{
\item{X}{An \code{n} by \code{p} matrix or data frame of data to be clustered.}

\item{R}{Covariates used for clustering. Required unless doing k-means
clustering (i.e. \code{sigma2=0} and \code{sigma2fixed=TRUE}).}

\item{K}{Number of clusters}

\item{mu}{starting values for cluster centers. If NULL (default), 
then value is chosen according to \code{muStart}.}

\item{muStart}{Character string indicating how initial value
 of mu should be selected. Only used if mu=NULL.  Possible
values are \code{"random"} or \code{"kmeans"} (default).}

\item{sigma2}{starting value of sigma2. If set to \code{0} and
\code{sigma2fixed=TRUE}, the standard k-means is done
instead of predictive k-means.}

\item{sigma2fixed}{Logical indicating whether sigma2
should be held fixed.  If FALSE, then
sigma2 is estimated using Maximum Likelihood.}

\item{maxitEM}{Maximum number of EM iterations for
finding the Mixture of Experts solution. If doing regular
k-means, this is passed as \code{iter.max}.}

\item{tol}{convergence criterion}

\item{convEM}{controls the measure of convergence for the 
EM algorithm.  Should be one of "mu", "gamma", or
"both".  Defaults to "both."  The EM algorithm 
stops when the Frobenius norm of the change in mu,
the change in gamma, or the change in mu and the change in gamma
is less than 'tol'.}

\item{nStarts}{number of times to perform EM algorithm}

\item{maxitMlogit}{Maximum number of iterations in the
mlogit optimization (nested within EM algorithm)}

\item{verbose}{numeric vector indicating how much output to produce}

\item{muRestart}{Gives max number of attempts at picking
   starting values. Only used when muStart='random'.
If selected starting values for mu are constant
within each cluster, then the starting values
are re-selected up to muRestart times.}

\item{returnAll}{A list containing all \code{nStarts} solutions is
included in the output.}

\item{...}{Additional arguments passed to \code{\link{mlogit}}}
}
\value{
An object of class \code{predkmeans}, containing the following elements:
\item{res.best}{A list containing the results from the best-fitting solution to the Mixture of Experts problem: 
 \describe{
  \item{mu}{Maximum-likelihood estimate of intercepts from normal mixture model. These are the cluster centers.}
  \item{gamma}{Maximum-likelihood estimates of the mixture coefficeints.}
  \item{sigma2}{If \code{sigma2fixed=FALSE}, the maximum likelihood estimate of \code{sigma2}}
  \item{conv}{Indicator of covergence.}
  \item{objective}{Value of the log-likelihood.}
  \item{iter}{Number of iterations.}
  \item{mfit}{A subset of output from \code{\link{mlogit}}.}
}}
\item{center}{Matrix of cluster centers}
\item{cluster}{Vector of cluster labels assigned to observations}
\item{K}{Number of clusters}
\item{sigma2}{Final value of sigma^2.}
\item{wSS}{Mean within-cluster sum-of-squares}
\item{sigma2fixed}{Logical indicator of whether sigma2 was held fixed}
}
\description{
Uses a Mixture-of-experts algorithm to find 
cluster centers that are influenced by prediction covariates.
}
\details{
A thorough description of this method is provided in Keller et al. (2017).
		The algorithm for sovling the mixture of Experts model is 
	based upon the approach presented by Jordan and Jacobs (1994). 

	If \code{sigma2} is 0 and \code{sigm2fixed} is TRUE, then standard k-means clustering (using \code{\link{kmeans}}) is done instead.
}
\examples{
n <- 200
r1 <- rnorm(n)
r2 <- rnorm(n)
u1 <- rbinom(n, size=1,prob=0)
cluster <- ifelse(r1<0, ifelse(u1, "A", "B"), ifelse(r2<0, "C", "D"))
mu1 <- c(A=2, B=2, C=-2, D=-2)
mu2 <- c(A=1, B=-1, C=-1, D=-1)
x1 <- rnorm(n, mu1[cluster], 4)
x2 <- rnorm(n, mu2[cluster], 4)
R <- model.matrix(~r1 + r2)
X <- cbind(x1, x2)
pkm <- predkmeans(X=cbind(x1, x2), R=R, K=4)
summary(pkm)
}
\references{
Keller, J.P., Drton, M., Larson, T., Kaufman, J.D., Sandler, D.P., and Szpiro, A.A. (2017). Covariate-adaptive clustering of exposures for air pollution epidemiology cohorts. \emph{Annals of Applied Statistics}, 11(1):93--113.

Jordan M. and Jacobs R. (1994). Hierarchical mixtures of
	experts and the EM algorithm. \emph{Neural computation 6}(2),
	 181-214.
}
\seealso{
\code{\link{predictML.predkmeans}, \link{predkmeansCVest}}
}
\author{
Joshua Keller
}
