#' Tripeptide Composition Descriptor
#'
#' This function calculates the Tripeptide Composition descriptor (dim: 8,000).
#'
#' @param x A character vector, as the input protein sequence.
#'
#' @return A length 8,000 named vector
#'
#' @author Nan Xiao <\url{https://nanx.me}>
#'
#' @seealso See \code{\link{extractAAC}} and \code{\link{extractDC}}
#' for Amino Acid Composition and Dipeptide Composition descriptors.
#'
#' @export extractTC
#'
#' @references
#' M. Bhasin, G. P. S. Raghava.
#' Classification of Nuclear Receptors Based on
#' Amino Acid Composition and Dipeptide Composition.
#' \emph{Journal of Biological Chemistry}, 2004, 279, 23262.
#'
#' @examples
#' x <- readFASTA(system.file("protseq/P00750.fasta", package = "protr"))[[1]]
#' extractTC(x)
extractTC <- function(x) {
  if (protcheck(x) == FALSE) {
    stop("x has unrecognized amino acid type")
  }

  # To achieve maximum performance, here we use dictionary directly
  # TCDict could also be generated with
  # AADict = c('A', 'R', 'N', 'D', 'C', 'E', 'Q', 'G', 'H', 'I',
  #            'L', 'K', 'M', 'F', 'P', 'S', 'T', 'W', 'Y', 'V')
  # DCDict = as.vector((outer(AADict, AADict, paste, sep = '')))
  # TCDict = as.vector((outer(DCDict, AADict, paste, sep = '')))

  TCDict <- c(
    "AAA", "RAA", "NAA", "DAA", "CAA", "EAA", "QAA", "GAA", "HAA",
    "IAA", "LAA", "KAA", "MAA", "FAA", "PAA", "SAA", "TAA", "WAA",
    "YAA", "VAA", "ARA", "RRA", "NRA", "DRA", "CRA", "ERA", "QRA",
    "GRA", "HRA", "IRA", "LRA", "KRA", "MRA", "FRA", "PRA", "SRA",
    "TRA", "WRA", "YRA", "VRA", "ANA", "RNA", "NNA", "DNA", "CNA",
    "ENA", "QNA", "GNA", "HNA", "INA", "LNA", "KNA", "MNA", "FNA",
    "PNA", "SNA", "TNA", "WNA", "YNA", "VNA", "ADA", "RDA", "NDA",
    "DDA", "CDA", "EDA", "QDA", "GDA", "HDA", "IDA", "LDA", "KDA",
    "MDA", "FDA", "PDA", "SDA", "TDA", "WDA", "YDA", "VDA", "ACA",
    "RCA", "NCA", "DCA", "CCA", "ECA", "QCA", "GCA", "HCA", "ICA",
    "LCA", "KCA", "MCA", "FCA", "PCA", "SCA", "TCA", "WCA", "YCA",
    "VCA", "AEA", "REA", "NEA", "DEA", "CEA", "EEA", "QEA", "GEA",
    "HEA", "IEA", "LEA", "KEA", "MEA", "FEA", "PEA", "SEA", "TEA",
    "WEA", "YEA", "VEA", "AQA", "RQA", "NQA", "DQA", "CQA", "EQA",
    "QQA", "GQA", "HQA", "IQA", "LQA", "KQA", "MQA", "FQA", "PQA",
    "SQA", "TQA", "WQA", "YQA", "VQA", "AGA", "RGA", "NGA", "DGA",
    "CGA", "EGA", "QGA", "GGA", "HGA", "IGA", "LGA", "KGA", "MGA",
    "FGA", "PGA", "SGA", "TGA", "WGA", "YGA", "VGA", "AHA", "RHA",
    "NHA", "DHA", "CHA", "EHA", "QHA", "GHA", "HHA", "IHA", "LHA",
    "KHA", "MHA", "FHA", "PHA", "SHA", "THA", "WHA", "YHA", "VHA",
    "AIA", "RIA", "NIA", "DIA", "CIA", "EIA", "QIA", "GIA", "HIA",
    "IIA", "LIA", "KIA", "MIA", "FIA", "PIA", "SIA", "TIA", "WIA",
    "YIA", "VIA", "ALA", "RLA", "NLA", "DLA", "CLA", "ELA", "QLA",
    "GLA", "HLA", "ILA", "LLA", "KLA", "MLA", "FLA", "PLA", "SLA",
    "TLA", "WLA", "YLA", "VLA", "AKA", "RKA", "NKA", "DKA", "CKA",
    "EKA", "QKA", "GKA", "HKA", "IKA", "LKA", "KKA", "MKA", "FKA",
    "PKA", "SKA", "TKA", "WKA", "YKA", "VKA", "AMA", "RMA", "NMA",
    "DMA", "CMA", "EMA", "QMA", "GMA", "HMA", "IMA", "LMA", "KMA",
    "MMA", "FMA", "PMA", "SMA", "TMA", "WMA", "YMA", "VMA", "AFA",
    "RFA", "NFA", "DFA", "CFA", "EFA", "QFA", "GFA", "HFA", "IFA",
    "LFA", "KFA", "MFA", "FFA", "PFA", "SFA", "TFA", "WFA", "YFA",
    "VFA", "APA", "RPA", "NPA", "DPA", "CPA", "EPA", "QPA", "GPA",
    "HPA", "IPA", "LPA", "KPA", "MPA", "FPA", "PPA", "SPA", "TPA",
    "WPA", "YPA", "VPA", "ASA", "RSA", "NSA", "DSA", "CSA", "ESA",
    "QSA", "GSA", "HSA", "ISA", "LSA", "KSA", "MSA", "FSA", "PSA",
    "SSA", "TSA", "WSA", "YSA", "VSA", "ATA", "RTA", "NTA", "DTA",
    "CTA", "ETA", "QTA", "GTA", "HTA", "ITA", "LTA", "KTA", "MTA",
    "FTA", "PTA", "STA", "TTA", "WTA", "YTA", "VTA", "AWA", "RWA",
    "NWA", "DWA", "CWA", "EWA", "QWA", "GWA", "HWA", "IWA", "LWA",
    "KWA", "MWA", "FWA", "PWA", "SWA", "TWA", "WWA", "YWA", "VWA",
    "AYA", "RYA", "NYA", "DYA", "CYA", "EYA", "QYA", "GYA", "HYA",
    "IYA", "LYA", "KYA", "MYA", "FYA", "PYA", "SYA", "TYA", "WYA",
    "YYA", "VYA", "AVA", "RVA", "NVA", "DVA", "CVA", "EVA", "QVA",
    "GVA", "HVA", "IVA", "LVA", "KVA", "MVA", "FVA", "PVA", "SVA",
    "TVA", "WVA", "YVA", "VVA", "AAR", "RAR", "NAR", "DAR", "CAR",
    "EAR", "QAR", "GAR", "HAR", "IAR", "LAR", "KAR", "MAR", "FAR",
    "PAR", "SAR", "TAR", "WAR", "YAR", "VAR", "ARR", "RRR", "NRR",
    "DRR", "CRR", "ERR", "QRR", "GRR", "HRR", "IRR", "LRR", "KRR",
    "MRR", "FRR", "PRR", "SRR", "TRR", "WRR", "YRR", "VRR", "ANR",
    "RNR", "NNR", "DNR", "CNR", "ENR", "QNR", "GNR", "HNR", "INR",
    "LNR", "KNR", "MNR", "FNR", "PNR", "SNR", "TNR", "WNR", "YNR",
    "VNR", "ADR", "RDR", "NDR", "DDR", "CDR", "EDR", "QDR", "GDR",
    "HDR", "IDR", "LDR", "KDR", "MDR", "FDR", "PDR", "SDR", "TDR",
    "WDR", "YDR", "VDR", "ACR", "RCR", "NCR", "DCR", "CCR", "ECR",
    "QCR", "GCR", "HCR", "ICR", "LCR", "KCR", "MCR", "FCR", "PCR",
    "SCR", "TCR", "WCR", "YCR", "VCR", "AER", "RER", "NER", "DER",
    "CER", "EER", "QER", "GER", "HER", "IER", "LER", "KER", "MER",
    "FER", "PER", "SER", "TER", "WER", "YER", "VER", "AQR", "RQR",
    "NQR", "DQR", "CQR", "EQR", "QQR", "GQR", "HQR", "IQR", "LQR",
    "KQR", "MQR", "FQR", "PQR", "SQR", "TQR", "WQR", "YQR", "VQR",
    "AGR", "RGR", "NGR", "DGR", "CGR", "EGR", "QGR", "GGR", "HGR",
    "IGR", "LGR", "KGR", "MGR", "FGR", "PGR", "SGR", "TGR", "WGR",
    "YGR", "VGR", "AHR", "RHR", "NHR", "DHR", "CHR", "EHR", "QHR",
    "GHR", "HHR", "IHR", "LHR", "KHR", "MHR", "FHR", "PHR", "SHR",
    "THR", "WHR", "YHR", "VHR", "AIR", "RIR", "NIR", "DIR", "CIR",
    "EIR", "QIR", "GIR", "HIR", "IIR", "LIR", "KIR", "MIR", "FIR",
    "PIR", "SIR", "TIR", "WIR", "YIR", "VIR", "ALR", "RLR", "NLR",
    "DLR", "CLR", "ELR", "QLR", "GLR", "HLR", "ILR", "LLR", "KLR",
    "MLR", "FLR", "PLR", "SLR", "TLR", "WLR", "YLR", "VLR", "AKR",
    "RKR", "NKR", "DKR", "CKR", "EKR", "QKR", "GKR", "HKR", "IKR",
    "LKR", "KKR", "MKR", "FKR", "PKR", "SKR", "TKR", "WKR", "YKR",
    "VKR", "AMR", "RMR", "NMR", "DMR", "CMR", "EMR", "QMR", "GMR",
    "HMR", "IMR", "LMR", "KMR", "MMR", "FMR", "PMR", "SMR", "TMR",
    "WMR", "YMR", "VMR", "AFR", "RFR", "NFR", "DFR", "CFR", "EFR",
    "QFR", "GFR", "HFR", "IFR", "LFR", "KFR", "MFR", "FFR", "PFR",
    "SFR", "TFR", "WFR", "YFR", "VFR", "APR", "RPR", "NPR", "DPR",
    "CPR", "EPR", "QPR", "GPR", "HPR", "IPR", "LPR", "KPR", "MPR",
    "FPR", "PPR", "SPR", "TPR", "WPR", "YPR", "VPR", "ASR", "RSR",
    "NSR", "DSR", "CSR", "ESR", "QSR", "GSR", "HSR", "ISR", "LSR",
    "KSR", "MSR", "FSR", "PSR", "SSR", "TSR", "WSR", "YSR", "VSR",
    "ATR", "RTR", "NTR", "DTR", "CTR", "ETR", "QTR", "GTR", "HTR",
    "ITR", "LTR", "KTR", "MTR", "FTR", "PTR", "STR", "TTR", "WTR",
    "YTR", "VTR", "AWR", "RWR", "NWR", "DWR", "CWR", "EWR", "QWR",
    "GWR", "HWR", "IWR", "LWR", "KWR", "MWR", "FWR", "PWR", "SWR",
    "TWR", "WWR", "YWR", "VWR", "AYR", "RYR", "NYR", "DYR", "CYR",
    "EYR", "QYR", "GYR", "HYR", "IYR", "LYR", "KYR", "MYR", "FYR",
    "PYR", "SYR", "TYR", "WYR", "YYR", "VYR", "AVR", "RVR", "NVR",
    "DVR", "CVR", "EVR", "QVR", "GVR", "HVR", "IVR", "LVR", "KVR",
    "MVR", "FVR", "PVR", "SVR", "TVR", "WVR", "YVR", "VVR", "AAN",
    "RAN", "NAN", "DAN", "CAN", "EAN", "QAN", "GAN", "HAN", "IAN",
    "LAN", "KAN", "MAN", "FAN", "PAN", "SAN", "TAN", "WAN", "YAN",
    "VAN", "ARN", "RRN", "NRN", "DRN", "CRN", "ERN", "QRN", "GRN",
    "HRN", "IRN", "LRN", "KRN", "MRN", "FRN", "PRN", "SRN", "TRN",
    "WRN", "YRN", "VRN", "ANN", "RNN", "NNN", "DNN", "CNN", "ENN",
    "QNN", "GNN", "HNN", "INN", "LNN", "KNN", "MNN", "FNN", "PNN",
    "SNN", "TNN", "WNN", "YNN", "VNN", "ADN", "RDN", "NDN", "DDN",
    "CDN", "EDN", "QDN", "GDN", "HDN", "IDN", "LDN", "KDN", "MDN",
    "FDN", "PDN", "SDN", "TDN", "WDN", "YDN", "VDN", "ACN", "RCN",
    "NCN", "DCN", "CCN", "ECN", "QCN", "GCN", "HCN", "ICN", "LCN",
    "KCN", "MCN", "FCN", "PCN", "SCN", "TCN", "WCN", "YCN", "VCN",
    "AEN", "REN", "NEN", "DEN", "CEN", "EEN", "QEN", "GEN", "HEN",
    "IEN", "LEN", "KEN", "MEN", "FEN", "PEN", "SEN", "TEN", "WEN",
    "YEN", "VEN", "AQN", "RQN", "NQN", "DQN", "CQN", "EQN", "QQN",
    "GQN", "HQN", "IQN", "LQN", "KQN", "MQN", "FQN", "PQN", "SQN",
    "TQN", "WQN", "YQN", "VQN", "AGN", "RGN", "NGN", "DGN", "CGN",
    "EGN", "QGN", "GGN", "HGN", "IGN", "LGN", "KGN", "MGN", "FGN",
    "PGN", "SGN", "TGN", "WGN", "YGN", "VGN", "AHN", "RHN", "NHN",
    "DHN", "CHN", "EHN", "QHN", "GHN", "HHN", "IHN", "LHN", "KHN",
    "MHN", "FHN", "PHN", "SHN", "THN", "WHN", "YHN", "VHN", "AIN",
    "RIN", "NIN", "DIN", "CIN", "EIN", "QIN", "GIN", "HIN", "IIN",
    "LIN", "KIN", "MIN", "FIN", "PIN", "SIN", "TIN", "WIN", "YIN",
    "VIN", "ALN", "RLN", "NLN", "DLN", "CLN", "ELN", "QLN", "GLN",
    "HLN", "ILN", "LLN", "KLN", "MLN", "FLN", "PLN", "SLN", "TLN",
    "WLN", "YLN", "VLN", "AKN", "RKN", "NKN", "DKN", "CKN", "EKN",
    "QKN", "GKN", "HKN", "IKN", "LKN", "KKN", "MKN", "FKN", "PKN",
    "SKN", "TKN", "WKN", "YKN", "VKN", "AMN", "RMN", "NMN", "DMN",
    "CMN", "EMN", "QMN", "GMN", "HMN", "IMN", "LMN", "KMN", "MMN",
    "FMN", "PMN", "SMN", "TMN", "WMN", "YMN", "VMN", "AFN", "RFN",
    "NFN", "DFN", "CFN", "EFN", "QFN", "GFN", "HFN", "IFN", "LFN",
    "KFN", "MFN", "FFN", "PFN", "SFN", "TFN", "WFN", "YFN", "VFN",
    "APN", "RPN", "NPN", "DPN", "CPN", "EPN", "QPN", "GPN", "HPN",
    "IPN", "LPN", "KPN", "MPN", "FPN", "PPN", "SPN", "TPN", "WPN",
    "YPN", "VPN", "ASN", "RSN", "NSN", "DSN", "CSN", "ESN", "QSN",
    "GSN", "HSN", "ISN", "LSN", "KSN", "MSN", "FSN", "PSN", "SSN",
    "TSN", "WSN", "YSN", "VSN", "ATN", "RTN", "NTN", "DTN", "CTN",
    "ETN", "QTN", "GTN", "HTN", "ITN", "LTN", "KTN", "MTN", "FTN",
    "PTN", "STN", "TTN", "WTN", "YTN", "VTN", "AWN", "RWN", "NWN",
    "DWN", "CWN", "EWN", "QWN", "GWN", "HWN", "IWN", "LWN", "KWN",
    "MWN", "FWN", "PWN", "SWN", "TWN", "WWN", "YWN", "VWN", "AYN",
    "RYN", "NYN", "DYN", "CYN", "EYN", "QYN", "GYN", "HYN", "IYN",
    "LYN", "KYN", "MYN", "FYN", "PYN", "SYN", "TYN", "WYN", "YYN",
    "VYN", "AVN", "RVN", "NVN", "DVN", "CVN", "EVN", "QVN", "GVN",
    "HVN", "IVN", "LVN", "KVN", "MVN", "FVN", "PVN", "SVN", "TVN",
    "WVN", "YVN", "VVN", "AAD", "RAD", "NAD", "DAD", "CAD", "EAD",
    "QAD", "GAD", "HAD", "IAD", "LAD", "KAD", "MAD", "FAD", "PAD",
    "SAD", "TAD", "WAD", "YAD", "VAD", "ARD", "RRD", "NRD", "DRD",
    "CRD", "ERD", "QRD", "GRD", "HRD", "IRD", "LRD", "KRD", "MRD",
    "FRD", "PRD", "SRD", "TRD", "WRD", "YRD", "VRD", "AND", "RND",
    "NND", "DND", "CND", "END", "QND", "GND", "HND", "IND", "LND",
    "KND", "MND", "FND", "PND", "SND", "TND", "WND", "YND", "VND",
    "ADD", "RDD", "NDD", "DDD", "CDD", "EDD", "QDD", "GDD", "HDD",
    "IDD", "LDD", "KDD", "MDD", "FDD", "PDD", "SDD", "TDD", "WDD",
    "YDD", "VDD", "ACD", "RCD", "NCD", "DCD", "CCD", "ECD", "QCD",
    "GCD", "HCD", "ICD", "LCD", "KCD", "MCD", "FCD", "PCD", "SCD",
    "TCD", "WCD", "YCD", "VCD", "AED", "RED", "NED", "DED", "CED",
    "EED", "QED", "GED", "HED", "IED", "LED", "KED", "MED", "FED",
    "PED", "SED", "TED", "WED", "YED", "VED", "AQD", "RQD", "NQD",
    "DQD", "CQD", "EQD", "QQD", "GQD", "HQD", "IQD", "LQD", "KQD",
    "MQD", "FQD", "PQD", "SQD", "TQD", "WQD", "YQD", "VQD", "AGD",
    "RGD", "NGD", "DGD", "CGD", "EGD", "QGD", "GGD", "HGD", "IGD",
    "LGD", "KGD", "MGD", "FGD", "PGD", "SGD", "TGD", "WGD", "YGD",
    "VGD", "AHD", "RHD", "NHD", "DHD", "CHD", "EHD", "QHD", "GHD",
    "HHD", "IHD", "LHD", "KHD", "MHD", "FHD", "PHD", "SHD", "THD",
    "WHD", "YHD", "VHD", "AID", "RID", "NID", "DID", "CID", "EID",
    "QID", "GID", "HID", "IID", "LID", "KID", "MID", "FID", "PID",
    "SID", "TID", "WID", "YID", "VID", "ALD", "RLD", "NLD", "DLD",
    "CLD", "ELD", "QLD", "GLD", "HLD", "ILD", "LLD", "KLD", "MLD",
    "FLD", "PLD", "SLD", "TLD", "WLD", "YLD", "VLD", "AKD", "RKD",
    "NKD", "DKD", "CKD", "EKD", "QKD", "GKD", "HKD", "IKD", "LKD",
    "KKD", "MKD", "FKD", "PKD", "SKD", "TKD", "WKD", "YKD", "VKD",
    "AMD", "RMD", "NMD", "DMD", "CMD", "EMD", "QMD", "GMD", "HMD",
    "IMD", "LMD", "KMD", "MMD", "FMD", "PMD", "SMD", "TMD", "WMD",
    "YMD", "VMD", "AFD", "RFD", "NFD", "DFD", "CFD", "EFD", "QFD",
    "GFD", "HFD", "IFD", "LFD", "KFD", "MFD", "FFD", "PFD", "SFD",
    "TFD", "WFD", "YFD", "VFD", "APD", "RPD", "NPD", "DPD", "CPD",
    "EPD", "QPD", "GPD", "HPD", "IPD", "LPD", "KPD", "MPD", "FPD",
    "PPD", "SPD", "TPD", "WPD", "YPD", "VPD", "ASD", "RSD", "NSD",
    "DSD", "CSD", "ESD", "QSD", "GSD", "HSD", "ISD", "LSD", "KSD",
    "MSD", "FSD", "PSD", "SSD", "TSD", "WSD", "YSD", "VSD", "ATD",
    "RTD", "NTD", "DTD", "CTD", "ETD", "QTD", "GTD", "HTD", "ITD",
    "LTD", "KTD", "MTD", "FTD", "PTD", "STD", "TTD", "WTD", "YTD",
    "VTD", "AWD", "RWD", "NWD", "DWD", "CWD", "EWD", "QWD", "GWD",
    "HWD", "IWD", "LWD", "KWD", "MWD", "FWD", "PWD", "SWD", "TWD",
    "WWD", "YWD", "VWD", "AYD", "RYD", "NYD", "DYD", "CYD", "EYD",
    "QYD", "GYD", "HYD", "IYD", "LYD", "KYD", "MYD", "FYD", "PYD",
    "SYD", "TYD", "WYD", "YYD", "VYD", "AVD", "RVD", "NVD", "DVD",
    "CVD", "EVD", "QVD", "GVD", "HVD", "IVD", "LVD", "KVD", "MVD",
    "FVD", "PVD", "SVD", "TVD", "WVD", "YVD", "VVD", "AAC", "RAC",
    "NAC", "DAC", "CAC", "EAC", "QAC", "GAC", "HAC", "IAC", "LAC",
    "KAC", "MAC", "FAC", "PAC", "SAC", "TAC", "WAC", "YAC", "VAC",
    "ARC", "RRC", "NRC", "DRC", "CRC", "ERC", "QRC", "GRC", "HRC",
    "IRC", "LRC", "KRC", "MRC", "FRC", "PRC", "SRC", "TRC", "WRC",
    "YRC", "VRC", "ANC", "RNC", "NNC", "DNC", "CNC", "ENC", "QNC",
    "GNC", "HNC", "INC", "LNC", "KNC", "MNC", "FNC", "PNC", "SNC",
    "TNC", "WNC", "YNC", "VNC", "ADC", "RDC", "NDC", "DDC", "CDC",
    "EDC", "QDC", "GDC", "HDC", "IDC", "LDC", "KDC", "MDC", "FDC",
    "PDC", "SDC", "TDC", "WDC", "YDC", "VDC", "ACC", "RCC", "NCC",
    "DCC", "CCC", "ECC", "QCC", "GCC", "HCC", "ICC", "LCC", "KCC",
    "MCC", "FCC", "PCC", "SCC", "TCC", "WCC", "YCC", "VCC", "AEC",
    "REC", "NEC", "DEC", "CEC", "EEC", "QEC", "GEC", "HEC", "IEC",
    "LEC", "KEC", "MEC", "FEC", "PEC", "SEC", "TEC", "WEC", "YEC",
    "VEC", "AQC", "RQC", "NQC", "DQC", "CQC", "EQC", "QQC", "GQC",
    "HQC", "IQC", "LQC", "KQC", "MQC", "FQC", "PQC", "SQC", "TQC",
    "WQC", "YQC", "VQC", "AGC", "RGC", "NGC", "DGC", "CGC", "EGC",
    "QGC", "GGC", "HGC", "IGC", "LGC", "KGC", "MGC", "FGC", "PGC",
    "SGC", "TGC", "WGC", "YGC", "VGC", "AHC", "RHC", "NHC", "DHC",
    "CHC", "EHC", "QHC", "GHC", "HHC", "IHC", "LHC", "KHC", "MHC",
    "FHC", "PHC", "SHC", "THC", "WHC", "YHC", "VHC", "AIC", "RIC",
    "NIC", "DIC", "CIC", "EIC", "QIC", "GIC", "HIC", "IIC", "LIC",
    "KIC", "MIC", "FIC", "PIC", "SIC", "TIC", "WIC", "YIC", "VIC",
    "ALC", "RLC", "NLC", "DLC", "CLC", "ELC", "QLC", "GLC", "HLC",
    "ILC", "LLC", "KLC", "MLC", "FLC", "PLC", "SLC", "TLC", "WLC",
    "YLC", "VLC", "AKC", "RKC", "NKC", "DKC", "CKC", "EKC", "QKC",
    "GKC", "HKC", "IKC", "LKC", "KKC", "MKC", "FKC", "PKC", "SKC",
    "TKC", "WKC", "YKC", "VKC", "AMC", "RMC", "NMC", "DMC", "CMC",
    "EMC", "QMC", "GMC", "HMC", "IMC", "LMC", "KMC", "MMC", "FMC",
    "PMC", "SMC", "TMC", "WMC", "YMC", "VMC", "AFC", "RFC", "NFC",
    "DFC", "CFC", "EFC", "QFC", "GFC", "HFC", "IFC", "LFC", "KFC",
    "MFC", "FFC", "PFC", "SFC", "TFC", "WFC", "YFC", "VFC", "APC",
    "RPC", "NPC", "DPC", "CPC", "EPC", "QPC", "GPC", "HPC", "IPC",
    "LPC", "KPC", "MPC", "FPC", "PPC", "SPC", "TPC", "WPC", "YPC",
    "VPC", "ASC", "RSC", "NSC", "DSC", "CSC", "ESC", "QSC", "GSC",
    "HSC", "ISC", "LSC", "KSC", "MSC", "FSC", "PSC", "SSC", "TSC",
    "WSC", "YSC", "VSC", "ATC", "RTC", "NTC", "DTC", "CTC", "ETC",
    "QTC", "GTC", "HTC", "ITC", "LTC", "KTC", "MTC", "FTC", "PTC",
    "STC", "TTC", "WTC", "YTC", "VTC", "AWC", "RWC", "NWC", "DWC",
    "CWC", "EWC", "QWC", "GWC", "HWC", "IWC", "LWC", "KWC", "MWC",
    "FWC", "PWC", "SWC", "TWC", "WWC", "YWC", "VWC", "AYC", "RYC",
    "NYC", "DYC", "CYC", "EYC", "QYC", "GYC", "HYC", "IYC", "LYC",
    "KYC", "MYC", "FYC", "PYC", "SYC", "TYC", "WYC", "YYC", "VYC",
    "AVC", "RVC", "NVC", "DVC", "CVC", "EVC", "QVC", "GVC", "HVC",
    "IVC", "LVC", "KVC", "MVC", "FVC", "PVC", "SVC", "TVC", "WVC",
    "YVC", "VVC", "AAE", "RAE", "NAE", "DAE", "CAE", "EAE", "QAE",
    "GAE", "HAE", "IAE", "LAE", "KAE", "MAE", "FAE", "PAE", "SAE",
    "TAE", "WAE", "YAE", "VAE", "ARE", "RRE", "NRE", "DRE", "CRE",
    "ERE", "QRE", "GRE", "HRE", "IRE", "LRE", "KRE", "MRE", "FRE",
    "PRE", "SRE", "TRE", "WRE", "YRE", "VRE", "ANE", "RNE", "NNE",
    "DNE", "CNE", "ENE", "QNE", "GNE", "HNE", "INE", "LNE", "KNE",
    "MNE", "FNE", "PNE", "SNE", "TNE", "WNE", "YNE", "VNE", "ADE",
    "RDE", "NDE", "DDE", "CDE", "EDE", "QDE", "GDE", "HDE", "IDE",
    "LDE", "KDE", "MDE", "FDE", "PDE", "SDE", "TDE", "WDE", "YDE",
    "VDE", "ACE", "RCE", "NCE", "DCE", "CCE", "ECE", "QCE", "GCE",
    "HCE", "ICE", "LCE", "KCE", "MCE", "FCE", "PCE", "SCE", "TCE",
    "WCE", "YCE", "VCE", "AEE", "REE", "NEE", "DEE", "CEE", "EEE",
    "QEE", "GEE", "HEE", "IEE", "LEE", "KEE", "MEE", "FEE", "PEE",
    "SEE", "TEE", "WEE", "YEE", "VEE", "AQE", "RQE", "NQE", "DQE",
    "CQE", "EQE", "QQE", "GQE", "HQE", "IQE", "LQE", "KQE", "MQE",
    "FQE", "PQE", "SQE", "TQE", "WQE", "YQE", "VQE", "AGE", "RGE",
    "NGE", "DGE", "CGE", "EGE", "QGE", "GGE", "HGE", "IGE", "LGE",
    "KGE", "MGE", "FGE", "PGE", "SGE", "TGE", "WGE", "YGE", "VGE",
    "AHE", "RHE", "NHE", "DHE", "CHE", "EHE", "QHE", "GHE", "HHE",
    "IHE", "LHE", "KHE", "MHE", "FHE", "PHE", "SHE", "THE", "WHE",
    "YHE", "VHE", "AIE", "RIE", "NIE", "DIE", "CIE", "EIE", "QIE",
    "GIE", "HIE", "IIE", "LIE", "KIE", "MIE", "FIE", "PIE", "SIE",
    "TIE", "WIE", "YIE", "VIE", "ALE", "RLE", "NLE", "DLE", "CLE",
    "ELE", "QLE", "GLE", "HLE", "ILE", "LLE", "KLE", "MLE", "FLE",
    "PLE", "SLE", "TLE", "WLE", "YLE", "VLE", "AKE", "RKE", "NKE",
    "DKE", "CKE", "EKE", "QKE", "GKE", "HKE", "IKE", "LKE", "KKE",
    "MKE", "FKE", "PKE", "SKE", "TKE", "WKE", "YKE", "VKE", "AME",
    "RME", "NME", "DME", "CME", "EME", "QME", "GME", "HME", "IME",
    "LME", "KME", "MME", "FME", "PME", "SME", "TME", "WME", "YME",
    "VME", "AFE", "RFE", "NFE", "DFE", "CFE", "EFE", "QFE", "GFE",
    "HFE", "IFE", "LFE", "KFE", "MFE", "FFE", "PFE", "SFE", "TFE",
    "WFE", "YFE", "VFE", "APE", "RPE", "NPE", "DPE", "CPE", "EPE",
    "QPE", "GPE", "HPE", "IPE", "LPE", "KPE", "MPE", "FPE", "PPE",
    "SPE", "TPE", "WPE", "YPE", "VPE", "ASE", "RSE", "NSE", "DSE",
    "CSE", "ESE", "QSE", "GSE", "HSE", "ISE", "LSE", "KSE", "MSE",
    "FSE", "PSE", "SSE", "TSE", "WSE", "YSE", "VSE", "ATE", "RTE",
    "NTE", "DTE", "CTE", "ETE", "QTE", "GTE", "HTE", "ITE", "LTE",
    "KTE", "MTE", "FTE", "PTE", "STE", "TTE", "WTE", "YTE", "VTE",
    "AWE", "RWE", "NWE", "DWE", "CWE", "EWE", "QWE", "GWE", "HWE",
    "IWE", "LWE", "KWE", "MWE", "FWE", "PWE", "SWE", "TWE", "WWE",
    "YWE", "VWE", "AYE", "RYE", "NYE", "DYE", "CYE", "EYE", "QYE",
    "GYE", "HYE", "IYE", "LYE", "KYE", "MYE", "FYE", "PYE", "SYE",
    "TYE", "WYE", "YYE", "VYE", "AVE", "RVE", "NVE", "DVE", "CVE",
    "EVE", "QVE", "GVE", "HVE", "IVE", "LVE", "KVE", "MVE", "FVE",
    "PVE", "SVE", "TVE", "WVE", "YVE", "VVE", "AAQ", "RAQ", "NAQ",
    "DAQ", "CAQ", "EAQ", "QAQ", "GAQ", "HAQ", "IAQ", "LAQ", "KAQ",
    "MAQ", "FAQ", "PAQ", "SAQ", "TAQ", "WAQ", "YAQ", "VAQ", "ARQ",
    "RRQ", "NRQ", "DRQ", "CRQ", "ERQ", "QRQ", "GRQ", "HRQ", "IRQ",
    "LRQ", "KRQ", "MRQ", "FRQ", "PRQ", "SRQ", "TRQ", "WRQ", "YRQ",
    "VRQ", "ANQ", "RNQ", "NNQ", "DNQ", "CNQ", "ENQ", "QNQ", "GNQ",
    "HNQ", "INQ", "LNQ", "KNQ", "MNQ", "FNQ", "PNQ", "SNQ", "TNQ",
    "WNQ", "YNQ", "VNQ", "ADQ", "RDQ", "NDQ", "DDQ", "CDQ", "EDQ",
    "QDQ", "GDQ", "HDQ", "IDQ", "LDQ", "KDQ", "MDQ", "FDQ", "PDQ",
    "SDQ", "TDQ", "WDQ", "YDQ", "VDQ", "ACQ", "RCQ", "NCQ", "DCQ",
    "CCQ", "ECQ", "QCQ", "GCQ", "HCQ", "ICQ", "LCQ", "KCQ", "MCQ",
    "FCQ", "PCQ", "SCQ", "TCQ", "WCQ", "YCQ", "VCQ", "AEQ", "REQ",
    "NEQ", "DEQ", "CEQ", "EEQ", "QEQ", "GEQ", "HEQ", "IEQ", "LEQ",
    "KEQ", "MEQ", "FEQ", "PEQ", "SEQ", "TEQ", "WEQ", "YEQ", "VEQ",
    "AQQ", "RQQ", "NQQ", "DQQ", "CQQ", "EQQ", "QQQ", "GQQ", "HQQ",
    "IQQ", "LQQ", "KQQ", "MQQ", "FQQ", "PQQ", "SQQ", "TQQ", "WQQ",
    "YQQ", "VQQ", "AGQ", "RGQ", "NGQ", "DGQ", "CGQ", "EGQ", "QGQ",
    "GGQ", "HGQ", "IGQ", "LGQ", "KGQ", "MGQ", "FGQ", "PGQ", "SGQ",
    "TGQ", "WGQ", "YGQ", "VGQ", "AHQ", "RHQ", "NHQ", "DHQ", "CHQ",
    "EHQ", "QHQ", "GHQ", "HHQ", "IHQ", "LHQ", "KHQ", "MHQ", "FHQ",
    "PHQ", "SHQ", "THQ", "WHQ", "YHQ", "VHQ", "AIQ", "RIQ", "NIQ",
    "DIQ", "CIQ", "EIQ", "QIQ", "GIQ", "HIQ", "IIQ", "LIQ", "KIQ",
    "MIQ", "FIQ", "PIQ", "SIQ", "TIQ", "WIQ", "YIQ", "VIQ", "ALQ",
    "RLQ", "NLQ", "DLQ", "CLQ", "ELQ", "QLQ", "GLQ", "HLQ", "ILQ",
    "LLQ", "KLQ", "MLQ", "FLQ", "PLQ", "SLQ", "TLQ", "WLQ", "YLQ",
    "VLQ", "AKQ", "RKQ", "NKQ", "DKQ", "CKQ", "EKQ", "QKQ", "GKQ",
    "HKQ", "IKQ", "LKQ", "KKQ", "MKQ", "FKQ", "PKQ", "SKQ", "TKQ",
    "WKQ", "YKQ", "VKQ", "AMQ", "RMQ", "NMQ", "DMQ", "CMQ", "EMQ",
    "QMQ", "GMQ", "HMQ", "IMQ", "LMQ", "KMQ", "MMQ", "FMQ", "PMQ",
    "SMQ", "TMQ", "WMQ", "YMQ", "VMQ", "AFQ", "RFQ", "NFQ", "DFQ",
    "CFQ", "EFQ", "QFQ", "GFQ", "HFQ", "IFQ", "LFQ", "KFQ", "MFQ",
    "FFQ", "PFQ", "SFQ", "TFQ", "WFQ", "YFQ", "VFQ", "APQ", "RPQ",
    "NPQ", "DPQ", "CPQ", "EPQ", "QPQ", "GPQ", "HPQ", "IPQ", "LPQ",
    "KPQ", "MPQ", "FPQ", "PPQ", "SPQ", "TPQ", "WPQ", "YPQ", "VPQ",
    "ASQ", "RSQ", "NSQ", "DSQ", "CSQ", "ESQ", "QSQ", "GSQ", "HSQ",
    "ISQ", "LSQ", "KSQ", "MSQ", "FSQ", "PSQ", "SSQ", "TSQ", "WSQ",
    "YSQ", "VSQ", "ATQ", "RTQ", "NTQ", "DTQ", "CTQ", "ETQ", "QTQ",
    "GTQ", "HTQ", "ITQ", "LTQ", "KTQ", "MTQ", "FTQ", "PTQ", "STQ",
    "TTQ", "WTQ", "YTQ", "VTQ", "AWQ", "RWQ", "NWQ", "DWQ", "CWQ",
    "EWQ", "QWQ", "GWQ", "HWQ", "IWQ", "LWQ", "KWQ", "MWQ", "FWQ",
    "PWQ", "SWQ", "TWQ", "WWQ", "YWQ", "VWQ", "AYQ", "RYQ", "NYQ",
    "DYQ", "CYQ", "EYQ", "QYQ", "GYQ", "HYQ", "IYQ", "LYQ", "KYQ",
    "MYQ", "FYQ", "PYQ", "SYQ", "TYQ", "WYQ", "YYQ", "VYQ", "AVQ",
    "RVQ", "NVQ", "DVQ", "CVQ", "EVQ", "QVQ", "GVQ", "HVQ", "IVQ",
    "LVQ", "KVQ", "MVQ", "FVQ", "PVQ", "SVQ", "TVQ", "WVQ", "YVQ",
    "VVQ", "AAG", "RAG", "NAG", "DAG", "CAG", "EAG", "QAG", "GAG",
    "HAG", "IAG", "LAG", "KAG", "MAG", "FAG", "PAG", "SAG", "TAG",
    "WAG", "YAG", "VAG", "ARG", "RRG", "NRG", "DRG", "CRG", "ERG",
    "QRG", "GRG", "HRG", "IRG", "LRG", "KRG", "MRG", "FRG", "PRG",
    "SRG", "TRG", "WRG", "YRG", "VRG", "ANG", "RNG", "NNG", "DNG",
    "CNG", "ENG", "QNG", "GNG", "HNG", "ING", "LNG", "KNG", "MNG",
    "FNG", "PNG", "SNG", "TNG", "WNG", "YNG", "VNG", "ADG", "RDG",
    "NDG", "DDG", "CDG", "EDG", "QDG", "GDG", "HDG", "IDG", "LDG",
    "KDG", "MDG", "FDG", "PDG", "SDG", "TDG", "WDG", "YDG", "VDG",
    "ACG", "RCG", "NCG", "DCG", "CCG", "ECG", "QCG", "GCG", "HCG",
    "ICG", "LCG", "KCG", "MCG", "FCG", "PCG", "SCG", "TCG", "WCG",
    "YCG", "VCG", "AEG", "REG", "NEG", "DEG", "CEG", "EEG", "QEG",
    "GEG", "HEG", "IEG", "LEG", "KEG", "MEG", "FEG", "PEG", "SEG",
    "TEG", "WEG", "YEG", "VEG", "AQG", "RQG", "NQG", "DQG", "CQG",
    "EQG", "QQG", "GQG", "HQG", "IQG", "LQG", "KQG", "MQG", "FQG",
    "PQG", "SQG", "TQG", "WQG", "YQG", "VQG", "AGG", "RGG", "NGG",
    "DGG", "CGG", "EGG", "QGG", "GGG", "HGG", "IGG", "LGG", "KGG",
    "MGG", "FGG", "PGG", "SGG", "TGG", "WGG", "YGG", "VGG", "AHG",
    "RHG", "NHG", "DHG", "CHG", "EHG", "QHG", "GHG", "HHG", "IHG",
    "LHG", "KHG", "MHG", "FHG", "PHG", "SHG", "THG", "WHG", "YHG",
    "VHG", "AIG", "RIG", "NIG", "DIG", "CIG", "EIG", "QIG", "GIG",
    "HIG", "IIG", "LIG", "KIG", "MIG", "FIG", "PIG", "SIG", "TIG",
    "WIG", "YIG", "VIG", "ALG", "RLG", "NLG", "DLG", "CLG", "ELG",
    "QLG", "GLG", "HLG", "ILG", "LLG", "KLG", "MLG", "FLG", "PLG",
    "SLG", "TLG", "WLG", "YLG", "VLG", "AKG", "RKG", "NKG", "DKG",
    "CKG", "EKG", "QKG", "GKG", "HKG", "IKG", "LKG", "KKG", "MKG",
    "FKG", "PKG", "SKG", "TKG", "WKG", "YKG", "VKG", "AMG", "RMG",
    "NMG", "DMG", "CMG", "EMG", "QMG", "GMG", "HMG", "IMG", "LMG",
    "KMG", "MMG", "FMG", "PMG", "SMG", "TMG", "WMG", "YMG", "VMG",
    "AFG", "RFG", "NFG", "DFG", "CFG", "EFG", "QFG", "GFG", "HFG",
    "IFG", "LFG", "KFG", "MFG", "FFG", "PFG", "SFG", "TFG", "WFG",
    "YFG", "VFG", "APG", "RPG", "NPG", "DPG", "CPG", "EPG", "QPG",
    "GPG", "HPG", "IPG", "LPG", "KPG", "MPG", "FPG", "PPG", "SPG",
    "TPG", "WPG", "YPG", "VPG", "ASG", "RSG", "NSG", "DSG", "CSG",
    "ESG", "QSG", "GSG", "HSG", "ISG", "LSG", "KSG", "MSG", "FSG",
    "PSG", "SSG", "TSG", "WSG", "YSG", "VSG", "ATG", "RTG", "NTG",
    "DTG", "CTG", "ETG", "QTG", "GTG", "HTG", "ITG", "LTG", "KTG",
    "MTG", "FTG", "PTG", "STG", "TTG", "WTG", "YTG", "VTG", "AWG",
    "RWG", "NWG", "DWG", "CWG", "EWG", "QWG", "GWG", "HWG", "IWG",
    "LWG", "KWG", "MWG", "FWG", "PWG", "SWG", "TWG", "WWG", "YWG",
    "VWG", "AYG", "RYG", "NYG", "DYG", "CYG", "EYG", "QYG", "GYG",
    "HYG", "IYG", "LYG", "KYG", "MYG", "FYG", "PYG", "SYG", "TYG",
    "WYG", "YYG", "VYG", "AVG", "RVG", "NVG", "DVG", "CVG", "EVG",
    "QVG", "GVG", "HVG", "IVG", "LVG", "KVG", "MVG", "FVG", "PVG",
    "SVG", "TVG", "WVG", "YVG", "VVG", "AAH", "RAH", "NAH", "DAH",
    "CAH", "EAH", "QAH", "GAH", "HAH", "IAH", "LAH", "KAH", "MAH",
    "FAH", "PAH", "SAH", "TAH", "WAH", "YAH", "VAH", "ARH", "RRH",
    "NRH", "DRH", "CRH", "ERH", "QRH", "GRH", "HRH", "IRH", "LRH",
    "KRH", "MRH", "FRH", "PRH", "SRH", "TRH", "WRH", "YRH", "VRH",
    "ANH", "RNH", "NNH", "DNH", "CNH", "ENH", "QNH", "GNH", "HNH",
    "INH", "LNH", "KNH", "MNH", "FNH", "PNH", "SNH", "TNH", "WNH",
    "YNH", "VNH", "ADH", "RDH", "NDH", "DDH", "CDH", "EDH", "QDH",
    "GDH", "HDH", "IDH", "LDH", "KDH", "MDH", "FDH", "PDH", "SDH",
    "TDH", "WDH", "YDH", "VDH", "ACH", "RCH", "NCH", "DCH", "CCH",
    "ECH", "QCH", "GCH", "HCH", "ICH", "LCH", "KCH", "MCH", "FCH",
    "PCH", "SCH", "TCH", "WCH", "YCH", "VCH", "AEH", "REH", "NEH",
    "DEH", "CEH", "EEH", "QEH", "GEH", "HEH", "IEH", "LEH", "KEH",
    "MEH", "FEH", "PEH", "SEH", "TEH", "WEH", "YEH", "VEH", "AQH",
    "RQH", "NQH", "DQH", "CQH", "EQH", "QQH", "GQH", "HQH", "IQH",
    "LQH", "KQH", "MQH", "FQH", "PQH", "SQH", "TQH", "WQH", "YQH",
    "VQH", "AGH", "RGH", "NGH", "DGH", "CGH", "EGH", "QGH", "GGH",
    "HGH", "IGH", "LGH", "KGH", "MGH", "FGH", "PGH", "SGH", "TGH",
    "WGH", "YGH", "VGH", "AHH", "RHH", "NHH", "DHH", "CHH", "EHH",
    "QHH", "GHH", "HHH", "IHH", "LHH", "KHH", "MHH", "FHH", "PHH",
    "SHH", "THH", "WHH", "YHH", "VHH", "AIH", "RIH", "NIH", "DIH",
    "CIH", "EIH", "QIH", "GIH", "HIH", "IIH", "LIH", "KIH", "MIH",
    "FIH", "PIH", "SIH", "TIH", "WIH", "YIH", "VIH", "ALH", "RLH",
    "NLH", "DLH", "CLH", "ELH", "QLH", "GLH", "HLH", "ILH", "LLH",
    "KLH", "MLH", "FLH", "PLH", "SLH", "TLH", "WLH", "YLH", "VLH",
    "AKH", "RKH", "NKH", "DKH", "CKH", "EKH", "QKH", "GKH", "HKH",
    "IKH", "LKH", "KKH", "MKH", "FKH", "PKH", "SKH", "TKH", "WKH",
    "YKH", "VKH", "AMH", "RMH", "NMH", "DMH", "CMH", "EMH", "QMH",
    "GMH", "HMH", "IMH", "LMH", "KMH", "MMH", "FMH", "PMH", "SMH",
    "TMH", "WMH", "YMH", "VMH", "AFH", "RFH", "NFH", "DFH", "CFH",
    "EFH", "QFH", "GFH", "HFH", "IFH", "LFH", "KFH", "MFH", "FFH",
    "PFH", "SFH", "TFH", "WFH", "YFH", "VFH", "APH", "RPH", "NPH",
    "DPH", "CPH", "EPH", "QPH", "GPH", "HPH", "IPH", "LPH", "KPH",
    "MPH", "FPH", "PPH", "SPH", "TPH", "WPH", "YPH", "VPH", "ASH",
    "RSH", "NSH", "DSH", "CSH", "ESH", "QSH", "GSH", "HSH", "ISH",
    "LSH", "KSH", "MSH", "FSH", "PSH", "SSH", "TSH", "WSH", "YSH",
    "VSH", "ATH", "RTH", "NTH", "DTH", "CTH", "ETH", "QTH", "GTH",
    "HTH", "ITH", "LTH", "KTH", "MTH", "FTH", "PTH", "STH", "TTH",
    "WTH", "YTH", "VTH", "AWH", "RWH", "NWH", "DWH", "CWH", "EWH",
    "QWH", "GWH", "HWH", "IWH", "LWH", "KWH", "MWH", "FWH", "PWH",
    "SWH", "TWH", "WWH", "YWH", "VWH", "AYH", "RYH", "NYH", "DYH",
    "CYH", "EYH", "QYH", "GYH", "HYH", "IYH", "LYH", "KYH", "MYH",
    "FYH", "PYH", "SYH", "TYH", "WYH", "YYH", "VYH", "AVH", "RVH",
    "NVH", "DVH", "CVH", "EVH", "QVH", "GVH", "HVH", "IVH", "LVH",
    "KVH", "MVH", "FVH", "PVH", "SVH", "TVH", "WVH", "YVH", "VVH",
    "AAI", "RAI", "NAI", "DAI", "CAI", "EAI", "QAI", "GAI", "HAI",
    "IAI", "LAI", "KAI", "MAI", "FAI", "PAI", "SAI", "TAI", "WAI",
    "YAI", "VAI", "ARI", "RRI", "NRI", "DRI", "CRI", "ERI", "QRI",
    "GRI", "HRI", "IRI", "LRI", "KRI", "MRI", "FRI", "PRI", "SRI",
    "TRI", "WRI", "YRI", "VRI", "ANI", "RNI", "NNI", "DNI", "CNI",
    "ENI", "QNI", "GNI", "HNI", "INI", "LNI", "KNI", "MNI", "FNI",
    "PNI", "SNI", "TNI", "WNI", "YNI", "VNI", "ADI", "RDI", "NDI",
    "DDI", "CDI", "EDI", "QDI", "GDI", "HDI", "IDI", "LDI", "KDI",
    "MDI", "FDI", "PDI", "SDI", "TDI", "WDI", "YDI", "VDI", "ACI",
    "RCI", "NCI", "DCI", "CCI", "ECI", "QCI", "GCI", "HCI", "ICI",
    "LCI", "KCI", "MCI", "FCI", "PCI", "SCI", "TCI", "WCI", "YCI",
    "VCI", "AEI", "REI", "NEI", "DEI", "CEI", "EEI", "QEI", "GEI",
    "HEI", "IEI", "LEI", "KEI", "MEI", "FEI", "PEI", "SEI", "TEI",
    "WEI", "YEI", "VEI", "AQI", "RQI", "NQI", "DQI", "CQI", "EQI",
    "QQI", "GQI", "HQI", "IQI", "LQI", "KQI", "MQI", "FQI", "PQI",
    "SQI", "TQI", "WQI", "YQI", "VQI", "AGI", "RGI", "NGI", "DGI",
    "CGI", "EGI", "QGI", "GGI", "HGI", "IGI", "LGI", "KGI", "MGI",
    "FGI", "PGI", "SGI", "TGI", "WGI", "YGI", "VGI", "AHI", "RHI",
    "NHI", "DHI", "CHI", "EHI", "QHI", "GHI", "HHI", "IHI", "LHI",
    "KHI", "MHI", "FHI", "PHI", "SHI", "THI", "WHI", "YHI", "VHI",
    "AII", "RII", "NII", "DII", "CII", "EII", "QII", "GII", "HII",
    "III", "LII", "KII", "MII", "FII", "PII", "SII", "TII", "WII",
    "YII", "VII", "ALI", "RLI", "NLI", "DLI", "CLI", "ELI", "QLI",
    "GLI", "HLI", "ILI", "LLI", "KLI", "MLI", "FLI", "PLI", "SLI",
    "TLI", "WLI", "YLI", "VLI", "AKI", "RKI", "NKI", "DKI", "CKI",
    "EKI", "QKI", "GKI", "HKI", "IKI", "LKI", "KKI", "MKI", "FKI",
    "PKI", "SKI", "TKI", "WKI", "YKI", "VKI", "AMI", "RMI", "NMI",
    "DMI", "CMI", "EMI", "QMI", "GMI", "HMI", "IMI", "LMI", "KMI",
    "MMI", "FMI", "PMI", "SMI", "TMI", "WMI", "YMI", "VMI", "AFI",
    "RFI", "NFI", "DFI", "CFI", "EFI", "QFI", "GFI", "HFI", "IFI",
    "LFI", "KFI", "MFI", "FFI", "PFI", "SFI", "TFI", "WFI", "YFI",
    "VFI", "API", "RPI", "NPI", "DPI", "CPI", "EPI", "QPI", "GPI",
    "HPI", "IPI", "LPI", "KPI", "MPI", "FPI", "PPI", "SPI", "TPI",
    "WPI", "YPI", "VPI", "ASI", "RSI", "NSI", "DSI", "CSI", "ESI",
    "QSI", "GSI", "HSI", "ISI", "LSI", "KSI", "MSI", "FSI", "PSI",
    "SSI", "TSI", "WSI", "YSI", "VSI", "ATI", "RTI", "NTI", "DTI",
    "CTI", "ETI", "QTI", "GTI", "HTI", "ITI", "LTI", "KTI", "MTI",
    "FTI", "PTI", "STI", "TTI", "WTI", "YTI", "VTI", "AWI", "RWI",
    "NWI", "DWI", "CWI", "EWI", "QWI", "GWI", "HWI", "IWI", "LWI",
    "KWI", "MWI", "FWI", "PWI", "SWI", "TWI", "WWI", "YWI", "VWI",
    "AYI", "RYI", "NYI", "DYI", "CYI", "EYI", "QYI", "GYI", "HYI",
    "IYI", "LYI", "KYI", "MYI", "FYI", "PYI", "SYI", "TYI", "WYI",
    "YYI", "VYI", "AVI", "RVI", "NVI", "DVI", "CVI", "EVI", "QVI",
    "GVI", "HVI", "IVI", "LVI", "KVI", "MVI", "FVI", "PVI", "SVI",
    "TVI", "WVI", "YVI", "VVI", "AAL", "RAL", "NAL", "DAL", "CAL",
    "EAL", "QAL", "GAL", "HAL", "IAL", "LAL", "KAL", "MAL", "FAL",
    "PAL", "SAL", "TAL", "WAL", "YAL", "VAL", "ARL", "RRL", "NRL",
    "DRL", "CRL", "ERL", "QRL", "GRL", "HRL", "IRL", "LRL", "KRL",
    "MRL", "FRL", "PRL", "SRL", "TRL", "WRL", "YRL", "VRL", "ANL",
    "RNL", "NNL", "DNL", "CNL", "ENL", "QNL", "GNL", "HNL", "INL",
    "LNL", "KNL", "MNL", "FNL", "PNL", "SNL", "TNL", "WNL", "YNL",
    "VNL", "ADL", "RDL", "NDL", "DDL", "CDL", "EDL", "QDL", "GDL",
    "HDL", "IDL", "LDL", "KDL", "MDL", "FDL", "PDL", "SDL", "TDL",
    "WDL", "YDL", "VDL", "ACL", "RCL", "NCL", "DCL", "CCL", "ECL",
    "QCL", "GCL", "HCL", "ICL", "LCL", "KCL", "MCL", "FCL", "PCL",
    "SCL", "TCL", "WCL", "YCL", "VCL", "AEL", "REL", "NEL", "DEL",
    "CEL", "EEL", "QEL", "GEL", "HEL", "IEL", "LEL", "KEL", "MEL",
    "FEL", "PEL", "SEL", "TEL", "WEL", "YEL", "VEL", "AQL", "RQL",
    "NQL", "DQL", "CQL", "EQL", "QQL", "GQL", "HQL", "IQL", "LQL",
    "KQL", "MQL", "FQL", "PQL", "SQL", "TQL", "WQL", "YQL", "VQL",
    "AGL", "RGL", "NGL", "DGL", "CGL", "EGL", "QGL", "GGL", "HGL",
    "IGL", "LGL", "KGL", "MGL", "FGL", "PGL", "SGL", "TGL", "WGL",
    "YGL", "VGL", "AHL", "RHL", "NHL", "DHL", "CHL", "EHL", "QHL",
    "GHL", "HHL", "IHL", "LHL", "KHL", "MHL", "FHL", "PHL", "SHL",
    "THL", "WHL", "YHL", "VHL", "AIL", "RIL", "NIL", "DIL", "CIL",
    "EIL", "QIL", "GIL", "HIL", "IIL", "LIL", "KIL", "MIL", "FIL",
    "PIL", "SIL", "TIL", "WIL", "YIL", "VIL", "ALL", "RLL", "NLL",
    "DLL", "CLL", "ELL", "QLL", "GLL", "HLL", "ILL", "LLL", "KLL",
    "MLL", "FLL", "PLL", "SLL", "TLL", "WLL", "YLL", "VLL", "AKL",
    "RKL", "NKL", "DKL", "CKL", "EKL", "QKL", "GKL", "HKL", "IKL",
    "LKL", "KKL", "MKL", "FKL", "PKL", "SKL", "TKL", "WKL", "YKL",
    "VKL", "AML", "RML", "NML", "DML", "CML", "EML", "QML", "GML",
    "HML", "IML", "LML", "KML", "MML", "FML", "PML", "SML", "TML",
    "WML", "YML", "VML", "AFL", "RFL", "NFL", "DFL", "CFL", "EFL",
    "QFL", "GFL", "HFL", "IFL", "LFL", "KFL", "MFL", "FFL", "PFL",
    "SFL", "TFL", "WFL", "YFL", "VFL", "APL", "RPL", "NPL", "DPL",
    "CPL", "EPL", "QPL", "GPL", "HPL", "IPL", "LPL", "KPL", "MPL",
    "FPL", "PPL", "SPL", "TPL", "WPL", "YPL", "VPL", "ASL", "RSL",
    "NSL", "DSL", "CSL", "ESL", "QSL", "GSL", "HSL", "ISL", "LSL",
    "KSL", "MSL", "FSL", "PSL", "SSL", "TSL", "WSL", "YSL", "VSL",
    "ATL", "RTL", "NTL", "DTL", "CTL", "ETL", "QTL", "GTL", "HTL",
    "ITL", "LTL", "KTL", "MTL", "FTL", "PTL", "STL", "TTL", "WTL",
    "YTL", "VTL", "AWL", "RWL", "NWL", "DWL", "CWL", "EWL", "QWL",
    "GWL", "HWL", "IWL", "LWL", "KWL", "MWL", "FWL", "PWL", "SWL",
    "TWL", "WWL", "YWL", "VWL", "AYL", "RYL", "NYL", "DYL", "CYL",
    "EYL", "QYL", "GYL", "HYL", "IYL", "LYL", "KYL", "MYL", "FYL",
    "PYL", "SYL", "TYL", "WYL", "YYL", "VYL", "AVL", "RVL", "NVL",
    "DVL", "CVL", "EVL", "QVL", "GVL", "HVL", "IVL", "LVL", "KVL",
    "MVL", "FVL", "PVL", "SVL", "TVL", "WVL", "YVL", "VVL", "AAK",
    "RAK", "NAK", "DAK", "CAK", "EAK", "QAK", "GAK", "HAK", "IAK",
    "LAK", "KAK", "MAK", "FAK", "PAK", "SAK", "TAK", "WAK", "YAK",
    "VAK", "ARK", "RRK", "NRK", "DRK", "CRK", "ERK", "QRK", "GRK",
    "HRK", "IRK", "LRK", "KRK", "MRK", "FRK", "PRK", "SRK", "TRK",
    "WRK", "YRK", "VRK", "ANK", "RNK", "NNK", "DNK", "CNK", "ENK",
    "QNK", "GNK", "HNK", "INK", "LNK", "KNK", "MNK", "FNK", "PNK",
    "SNK", "TNK", "WNK", "YNK", "VNK", "ADK", "RDK", "NDK", "DDK",
    "CDK", "EDK", "QDK", "GDK", "HDK", "IDK", "LDK", "KDK", "MDK",
    "FDK", "PDK", "SDK", "TDK", "WDK", "YDK", "VDK", "ACK", "RCK",
    "NCK", "DCK", "CCK", "ECK", "QCK", "GCK", "HCK", "ICK", "LCK",
    "KCK", "MCK", "FCK", "PCK", "SCK", "TCK", "WCK", "YCK", "VCK",
    "AEK", "REK", "NEK", "DEK", "CEK", "EEK", "QEK", "GEK", "HEK",
    "IEK", "LEK", "KEK", "MEK", "FEK", "PEK", "SEK", "TEK", "WEK",
    "YEK", "VEK", "AQK", "RQK", "NQK", "DQK", "CQK", "EQK", "QQK",
    "GQK", "HQK", "IQK", "LQK", "KQK", "MQK", "FQK", "PQK", "SQK",
    "TQK", "WQK", "YQK", "VQK", "AGK", "RGK", "NGK", "DGK", "CGK",
    "EGK", "QGK", "GGK", "HGK", "IGK", "LGK", "KGK", "MGK", "FGK",
    "PGK", "SGK", "TGK", "WGK", "YGK", "VGK", "AHK", "RHK", "NHK",
    "DHK", "CHK", "EHK", "QHK", "GHK", "HHK", "IHK", "LHK", "KHK",
    "MHK", "FHK", "PHK", "SHK", "THK", "WHK", "YHK", "VHK", "AIK",
    "RIK", "NIK", "DIK", "CIK", "EIK", "QIK", "GIK", "HIK", "IIK",
    "LIK", "KIK", "MIK", "FIK", "PIK", "SIK", "TIK", "WIK", "YIK",
    "VIK", "ALK", "RLK", "NLK", "DLK", "CLK", "ELK", "QLK", "GLK",
    "HLK", "ILK", "LLK", "KLK", "MLK", "FLK", "PLK", "SLK", "TLK",
    "WLK", "YLK", "VLK", "AKK", "RKK", "NKK", "DKK", "CKK", "EKK",
    "QKK", "GKK", "HKK", "IKK", "LKK", "KKK", "MKK", "FKK", "PKK",
    "SKK", "TKK", "WKK", "YKK", "VKK", "AMK", "RMK", "NMK", "DMK",
    "CMK", "EMK", "QMK", "GMK", "HMK", "IMK", "LMK", "KMK", "MMK",
    "FMK", "PMK", "SMK", "TMK", "WMK", "YMK", "VMK", "AFK", "RFK",
    "NFK", "DFK", "CFK", "EFK", "QFK", "GFK", "HFK", "IFK", "LFK",
    "KFK", "MFK", "FFK", "PFK", "SFK", "TFK", "WFK", "YFK", "VFK",
    "APK", "RPK", "NPK", "DPK", "CPK", "EPK", "QPK", "GPK", "HPK",
    "IPK", "LPK", "KPK", "MPK", "FPK", "PPK", "SPK", "TPK", "WPK",
    "YPK", "VPK", "ASK", "RSK", "NSK", "DSK", "CSK", "ESK", "QSK",
    "GSK", "HSK", "ISK", "LSK", "KSK", "MSK", "FSK", "PSK", "SSK",
    "TSK", "WSK", "YSK", "VSK", "ATK", "RTK", "NTK", "DTK", "CTK",
    "ETK", "QTK", "GTK", "HTK", "ITK", "LTK", "KTK", "MTK", "FTK",
    "PTK", "STK", "TTK", "WTK", "YTK", "VTK", "AWK", "RWK", "NWK",
    "DWK", "CWK", "EWK", "QWK", "GWK", "HWK", "IWK", "LWK", "KWK",
    "MWK", "FWK", "PWK", "SWK", "TWK", "WWK", "YWK", "VWK", "AYK",
    "RYK", "NYK", "DYK", "CYK", "EYK", "QYK", "GYK", "HYK", "IYK",
    "LYK", "KYK", "MYK", "FYK", "PYK", "SYK", "TYK", "WYK", "YYK",
    "VYK", "AVK", "RVK", "NVK", "DVK", "CVK", "EVK", "QVK", "GVK",
    "HVK", "IVK", "LVK", "KVK", "MVK", "FVK", "PVK", "SVK", "TVK",
    "WVK", "YVK", "VVK", "AAM", "RAM", "NAM", "DAM", "CAM", "EAM",
    "QAM", "GAM", "HAM", "IAM", "LAM", "KAM", "MAM", "FAM", "PAM",
    "SAM", "TAM", "WAM", "YAM", "VAM", "ARM", "RRM", "NRM", "DRM",
    "CRM", "ERM", "QRM", "GRM", "HRM", "IRM", "LRM", "KRM", "MRM",
    "FRM", "PRM", "SRM", "TRM", "WRM", "YRM", "VRM", "ANM", "RNM",
    "NNM", "DNM", "CNM", "ENM", "QNM", "GNM", "HNM", "INM", "LNM",
    "KNM", "MNM", "FNM", "PNM", "SNM", "TNM", "WNM", "YNM", "VNM",
    "ADM", "RDM", "NDM", "DDM", "CDM", "EDM", "QDM", "GDM", "HDM",
    "IDM", "LDM", "KDM", "MDM", "FDM", "PDM", "SDM", "TDM", "WDM",
    "YDM", "VDM", "ACM", "RCM", "NCM", "DCM", "CCM", "ECM", "QCM",
    "GCM", "HCM", "ICM", "LCM", "KCM", "MCM", "FCM", "PCM", "SCM",
    "TCM", "WCM", "YCM", "VCM", "AEM", "REM", "NEM", "DEM", "CEM",
    "EEM", "QEM", "GEM", "HEM", "IEM", "LEM", "KEM", "MEM", "FEM",
    "PEM", "SEM", "TEM", "WEM", "YEM", "VEM", "AQM", "RQM", "NQM",
    "DQM", "CQM", "EQM", "QQM", "GQM", "HQM", "IQM", "LQM", "KQM",
    "MQM", "FQM", "PQM", "SQM", "TQM", "WQM", "YQM", "VQM", "AGM",
    "RGM", "NGM", "DGM", "CGM", "EGM", "QGM", "GGM", "HGM", "IGM",
    "LGM", "KGM", "MGM", "FGM", "PGM", "SGM", "TGM", "WGM", "YGM",
    "VGM", "AHM", "RHM", "NHM", "DHM", "CHM", "EHM", "QHM", "GHM",
    "HHM", "IHM", "LHM", "KHM", "MHM", "FHM", "PHM", "SHM", "THM",
    "WHM", "YHM", "VHM", "AIM", "RIM", "NIM", "DIM", "CIM", "EIM",
    "QIM", "GIM", "HIM", "IIM", "LIM", "KIM", "MIM", "FIM", "PIM",
    "SIM", "TIM", "WIM", "YIM", "VIM", "ALM", "RLM", "NLM", "DLM",
    "CLM", "ELM", "QLM", "GLM", "HLM", "ILM", "LLM", "KLM", "MLM",
    "FLM", "PLM", "SLM", "TLM", "WLM", "YLM", "VLM", "AKM", "RKM",
    "NKM", "DKM", "CKM", "EKM", "QKM", "GKM", "HKM", "IKM", "LKM",
    "KKM", "MKM", "FKM", "PKM", "SKM", "TKM", "WKM", "YKM", "VKM",
    "AMM", "RMM", "NMM", "DMM", "CMM", "EMM", "QMM", "GMM", "HMM",
    "IMM", "LMM", "KMM", "MMM", "FMM", "PMM", "SMM", "TMM", "WMM",
    "YMM", "VMM", "AFM", "RFM", "NFM", "DFM", "CFM", "EFM", "QFM",
    "GFM", "HFM", "IFM", "LFM", "KFM", "MFM", "FFM", "PFM", "SFM",
    "TFM", "WFM", "YFM", "VFM", "APM", "RPM", "NPM", "DPM", "CPM",
    "EPM", "QPM", "GPM", "HPM", "IPM", "LPM", "KPM", "MPM", "FPM",
    "PPM", "SPM", "TPM", "WPM", "YPM", "VPM", "ASM", "RSM", "NSM",
    "DSM", "CSM", "ESM", "QSM", "GSM", "HSM", "ISM", "LSM", "KSM",
    "MSM", "FSM", "PSM", "SSM", "TSM", "WSM", "YSM", "VSM", "ATM",
    "RTM", "NTM", "DTM", "CTM", "ETM", "QTM", "GTM", "HTM", "ITM",
    "LTM", "KTM", "MTM", "FTM", "PTM", "STM", "TTM", "WTM", "YTM",
    "VTM", "AWM", "RWM", "NWM", "DWM", "CWM", "EWM", "QWM", "GWM",
    "HWM", "IWM", "LWM", "KWM", "MWM", "FWM", "PWM", "SWM", "TWM",
    "WWM", "YWM", "VWM", "AYM", "RYM", "NYM", "DYM", "CYM", "EYM",
    "QYM", "GYM", "HYM", "IYM", "LYM", "KYM", "MYM", "FYM", "PYM",
    "SYM", "TYM", "WYM", "YYM", "VYM", "AVM", "RVM", "NVM", "DVM",
    "CVM", "EVM", "QVM", "GVM", "HVM", "IVM", "LVM", "KVM", "MVM",
    "FVM", "PVM", "SVM", "TVM", "WVM", "YVM", "VVM", "AAF", "RAF",
    "NAF", "DAF", "CAF", "EAF", "QAF", "GAF", "HAF", "IAF", "LAF",
    "KAF", "MAF", "FAF", "PAF", "SAF", "TAF", "WAF", "YAF", "VAF",
    "ARF", "RRF", "NRF", "DRF", "CRF", "ERF", "QRF", "GRF", "HRF",
    "IRF", "LRF", "KRF", "MRF", "FRF", "PRF", "SRF", "TRF", "WRF",
    "YRF", "VRF", "ANF", "RNF", "NNF", "DNF", "CNF", "ENF", "QNF",
    "GNF", "HNF", "INF", "LNF", "KNF", "MNF", "FNF", "PNF", "SNF",
    "TNF", "WNF", "YNF", "VNF", "ADF", "RDF", "NDF", "DDF", "CDF",
    "EDF", "QDF", "GDF", "HDF", "IDF", "LDF", "KDF", "MDF", "FDF",
    "PDF", "SDF", "TDF", "WDF", "YDF", "VDF", "ACF", "RCF", "NCF",
    "DCF", "CCF", "ECF", "QCF", "GCF", "HCF", "ICF", "LCF", "KCF",
    "MCF", "FCF", "PCF", "SCF", "TCF", "WCF", "YCF", "VCF", "AEF",
    "REF", "NEF", "DEF", "CEF", "EEF", "QEF", "GEF", "HEF", "IEF",
    "LEF", "KEF", "MEF", "FEF", "PEF", "SEF", "TEF", "WEF", "YEF",
    "VEF", "AQF", "RQF", "NQF", "DQF", "CQF", "EQF", "QQF", "GQF",
    "HQF", "IQF", "LQF", "KQF", "MQF", "FQF", "PQF", "SQF", "TQF",
    "WQF", "YQF", "VQF", "AGF", "RGF", "NGF", "DGF", "CGF", "EGF",
    "QGF", "GGF", "HGF", "IGF", "LGF", "KGF", "MGF", "FGF", "PGF",
    "SGF", "TGF", "WGF", "YGF", "VGF", "AHF", "RHF", "NHF", "DHF",
    "CHF", "EHF", "QHF", "GHF", "HHF", "IHF", "LHF", "KHF", "MHF",
    "FHF", "PHF", "SHF", "THF", "WHF", "YHF", "VHF", "AIF", "RIF",
    "NIF", "DIF", "CIF", "EIF", "QIF", "GIF", "HIF", "IIF", "LIF",
    "KIF", "MIF", "FIF", "PIF", "SIF", "TIF", "WIF", "YIF", "VIF",
    "ALF", "RLF", "NLF", "DLF", "CLF", "ELF", "QLF", "GLF", "HLF",
    "ILF", "LLF", "KLF", "MLF", "FLF", "PLF", "SLF", "TLF", "WLF",
    "YLF", "VLF", "AKF", "RKF", "NKF", "DKF", "CKF", "EKF", "QKF",
    "GKF", "HKF", "IKF", "LKF", "KKF", "MKF", "FKF", "PKF", "SKF",
    "TKF", "WKF", "YKF", "VKF", "AMF", "RMF", "NMF", "DMF", "CMF",
    "EMF", "QMF", "GMF", "HMF", "IMF", "LMF", "KMF", "MMF", "FMF",
    "PMF", "SMF", "TMF", "WMF", "YMF", "VMF", "AFF", "RFF", "NFF",
    "DFF", "CFF", "EFF", "QFF", "GFF", "HFF", "IFF", "LFF", "KFF",
    "MFF", "FFF", "PFF", "SFF", "TFF", "WFF", "YFF", "VFF", "APF",
    "RPF", "NPF", "DPF", "CPF", "EPF", "QPF", "GPF", "HPF", "IPF",
    "LPF", "KPF", "MPF", "FPF", "PPF", "SPF", "TPF", "WPF", "YPF",
    "VPF", "ASF", "RSF", "NSF", "DSF", "CSF", "ESF", "QSF", "GSF",
    "HSF", "ISF", "LSF", "KSF", "MSF", "FSF", "PSF", "SSF", "TSF",
    "WSF", "YSF", "VSF", "ATF", "RTF", "NTF", "DTF", "CTF", "ETF",
    "QTF", "GTF", "HTF", "ITF", "LTF", "KTF", "MTF", "FTF", "PTF",
    "STF", "TTF", "WTF", "YTF", "VTF", "AWF", "RWF", "NWF", "DWF",
    "CWF", "EWF", "QWF", "GWF", "HWF", "IWF", "LWF", "KWF", "MWF",
    "FWF", "PWF", "SWF", "TWF", "WWF", "YWF", "VWF", "AYF", "RYF",
    "NYF", "DYF", "CYF", "EYF", "QYF", "GYF", "HYF", "IYF", "LYF",
    "KYF", "MYF", "FYF", "PYF", "SYF", "TYF", "WYF", "YYF", "VYF",
    "AVF", "RVF", "NVF", "DVF", "CVF", "EVF", "QVF", "GVF", "HVF",
    "IVF", "LVF", "KVF", "MVF", "FVF", "PVF", "SVF", "TVF", "WVF",
    "YVF", "VVF", "AAP", "RAP", "NAP", "DAP", "CAP", "EAP", "QAP",
    "GAP", "HAP", "IAP", "LAP", "KAP", "MAP", "FAP", "PAP", "SAP",
    "TAP", "WAP", "YAP", "VAP", "ARP", "RRP", "NRP", "DRP", "CRP",
    "ERP", "QRP", "GRP", "HRP", "IRP", "LRP", "KRP", "MRP", "FRP",
    "PRP", "SRP", "TRP", "WRP", "YRP", "VRP", "ANP", "RNP", "NNP",
    "DNP", "CNP", "ENP", "QNP", "GNP", "HNP", "INP", "LNP", "KNP",
    "MNP", "FNP", "PNP", "SNP", "TNP", "WNP", "YNP", "VNP", "ADP",
    "RDP", "NDP", "DDP", "CDP", "EDP", "QDP", "GDP", "HDP", "IDP",
    "LDP", "KDP", "MDP", "FDP", "PDP", "SDP", "TDP", "WDP", "YDP",
    "VDP", "ACP", "RCP", "NCP", "DCP", "CCP", "ECP", "QCP", "GCP",
    "HCP", "ICP", "LCP", "KCP", "MCP", "FCP", "PCP", "SCP", "TCP",
    "WCP", "YCP", "VCP", "AEP", "REP", "NEP", "DEP", "CEP", "EEP",
    "QEP", "GEP", "HEP", "IEP", "LEP", "KEP", "MEP", "FEP", "PEP",
    "SEP", "TEP", "WEP", "YEP", "VEP", "AQP", "RQP", "NQP", "DQP",
    "CQP", "EQP", "QQP", "GQP", "HQP", "IQP", "LQP", "KQP", "MQP",
    "FQP", "PQP", "SQP", "TQP", "WQP", "YQP", "VQP", "AGP", "RGP",
    "NGP", "DGP", "CGP", "EGP", "QGP", "GGP", "HGP", "IGP", "LGP",
    "KGP", "MGP", "FGP", "PGP", "SGP", "TGP", "WGP", "YGP", "VGP",
    "AHP", "RHP", "NHP", "DHP", "CHP", "EHP", "QHP", "GHP", "HHP",
    "IHP", "LHP", "KHP", "MHP", "FHP", "PHP", "SHP", "THP", "WHP",
    "YHP", "VHP", "AIP", "RIP", "NIP", "DIP", "CIP", "EIP", "QIP",
    "GIP", "HIP", "IIP", "LIP", "KIP", "MIP", "FIP", "PIP", "SIP",
    "TIP", "WIP", "YIP", "VIP", "ALP", "RLP", "NLP", "DLP", "CLP",
    "ELP", "QLP", "GLP", "HLP", "ILP", "LLP", "KLP", "MLP", "FLP",
    "PLP", "SLP", "TLP", "WLP", "YLP", "VLP", "AKP", "RKP", "NKP",
    "DKP", "CKP", "EKP", "QKP", "GKP", "HKP", "IKP", "LKP", "KKP",
    "MKP", "FKP", "PKP", "SKP", "TKP", "WKP", "YKP", "VKP", "AMP",
    "RMP", "NMP", "DMP", "CMP", "EMP", "QMP", "GMP", "HMP", "IMP",
    "LMP", "KMP", "MMP", "FMP", "PMP", "SMP", "TMP", "WMP", "YMP",
    "VMP", "AFP", "RFP", "NFP", "DFP", "CFP", "EFP", "QFP", "GFP",
    "HFP", "IFP", "LFP", "KFP", "MFP", "FFP", "PFP", "SFP", "TFP",
    "WFP", "YFP", "VFP", "APP", "RPP", "NPP", "DPP", "CPP", "EPP",
    "QPP", "GPP", "HPP", "IPP", "LPP", "KPP", "MPP", "FPP", "PPP",
    "SPP", "TPP", "WPP", "YPP", "VPP", "ASP", "RSP", "NSP", "DSP",
    "CSP", "ESP", "QSP", "GSP", "HSP", "ISP", "LSP", "KSP", "MSP",
    "FSP", "PSP", "SSP", "TSP", "WSP", "YSP", "VSP", "ATP", "RTP",
    "NTP", "DTP", "CTP", "ETP", "QTP", "GTP", "HTP", "ITP", "LTP",
    "KTP", "MTP", "FTP", "PTP", "STP", "TTP", "WTP", "YTP", "VTP",
    "AWP", "RWP", "NWP", "DWP", "CWP", "EWP", "QWP", "GWP", "HWP",
    "IWP", "LWP", "KWP", "MWP", "FWP", "PWP", "SWP", "TWP", "WWP",
    "YWP", "VWP", "AYP", "RYP", "NYP", "DYP", "CYP", "EYP", "QYP",
    "GYP", "HYP", "IYP", "LYP", "KYP", "MYP", "FYP", "PYP", "SYP",
    "TYP", "WYP", "YYP", "VYP", "AVP", "RVP", "NVP", "DVP", "CVP",
    "EVP", "QVP", "GVP", "HVP", "IVP", "LVP", "KVP", "MVP", "FVP",
    "PVP", "SVP", "TVP", "WVP", "YVP", "VVP", "AAS", "RAS", "NAS",
    "DAS", "CAS", "EAS", "QAS", "GAS", "HAS", "IAS", "LAS", "KAS",
    "MAS", "FAS", "PAS", "SAS", "TAS", "WAS", "YAS", "VAS", "ARS",
    "RRS", "NRS", "DRS", "CRS", "ERS", "QRS", "GRS", "HRS", "IRS",
    "LRS", "KRS", "MRS", "FRS", "PRS", "SRS", "TRS", "WRS", "YRS",
    "VRS", "ANS", "RNS", "NNS", "DNS", "CNS", "ENS", "QNS", "GNS",
    "HNS", "INS", "LNS", "KNS", "MNS", "FNS", "PNS", "SNS", "TNS",
    "WNS", "YNS", "VNS", "ADS", "RDS", "NDS", "DDS", "CDS", "EDS",
    "QDS", "GDS", "HDS", "IDS", "LDS", "KDS", "MDS", "FDS", "PDS",
    "SDS", "TDS", "WDS", "YDS", "VDS", "ACS", "RCS", "NCS", "DCS",
    "CCS", "ECS", "QCS", "GCS", "HCS", "ICS", "LCS", "KCS", "MCS",
    "FCS", "PCS", "SCS", "TCS", "WCS", "YCS", "VCS", "AES", "RES",
    "NES", "DES", "CES", "EES", "QES", "GES", "HES", "IES", "LES",
    "KES", "MES", "FES", "PES", "SES", "TES", "WES", "YES", "VES",
    "AQS", "RQS", "NQS", "DQS", "CQS", "EQS", "QQS", "GQS", "HQS",
    "IQS", "LQS", "KQS", "MQS", "FQS", "PQS", "SQS", "TQS", "WQS",
    "YQS", "VQS", "AGS", "RGS", "NGS", "DGS", "CGS", "EGS", "QGS",
    "GGS", "HGS", "IGS", "LGS", "KGS", "MGS", "FGS", "PGS", "SGS",
    "TGS", "WGS", "YGS", "VGS", "AHS", "RHS", "NHS", "DHS", "CHS",
    "EHS", "QHS", "GHS", "HHS", "IHS", "LHS", "KHS", "MHS", "FHS",
    "PHS", "SHS", "THS", "WHS", "YHS", "VHS", "AIS", "RIS", "NIS",
    "DIS", "CIS", "EIS", "QIS", "GIS", "HIS", "IIS", "LIS", "KIS",
    "MIS", "FIS", "PIS", "SIS", "TIS", "WIS", "YIS", "VIS", "ALS",
    "RLS", "NLS", "DLS", "CLS", "ELS", "QLS", "GLS", "HLS", "ILS",
    "LLS", "KLS", "MLS", "FLS", "PLS", "SLS", "TLS", "WLS", "YLS",
    "VLS", "AKS", "RKS", "NKS", "DKS", "CKS", "EKS", "QKS", "GKS",
    "HKS", "IKS", "LKS", "KKS", "MKS", "FKS", "PKS", "SKS", "TKS",
    "WKS", "YKS", "VKS", "AMS", "RMS", "NMS", "DMS", "CMS", "EMS",
    "QMS", "GMS", "HMS", "IMS", "LMS", "KMS", "MMS", "FMS", "PMS",
    "SMS", "TMS", "WMS", "YMS", "VMS", "AFS", "RFS", "NFS", "DFS",
    "CFS", "EFS", "QFS", "GFS", "HFS", "IFS", "LFS", "KFS", "MFS",
    "FFS", "PFS", "SFS", "TFS", "WFS", "YFS", "VFS", "APS", "RPS",
    "NPS", "DPS", "CPS", "EPS", "QPS", "GPS", "HPS", "IPS", "LPS",
    "KPS", "MPS", "FPS", "PPS", "SPS", "TPS", "WPS", "YPS", "VPS",
    "ASS", "RSS", "NSS", "DSS", "CSS", "ESS", "QSS", "GSS", "HSS",
    "ISS", "LSS", "KSS", "MSS", "FSS", "PSS", "SSS", "TSS", "WSS",
    "YSS", "VSS", "ATS", "RTS", "NTS", "DTS", "CTS", "ETS", "QTS",
    "GTS", "HTS", "ITS", "LTS", "KTS", "MTS", "FTS", "PTS", "STS",
    "TTS", "WTS", "YTS", "VTS", "AWS", "RWS", "NWS", "DWS", "CWS",
    "EWS", "QWS", "GWS", "HWS", "IWS", "LWS", "KWS", "MWS", "FWS",
    "PWS", "SWS", "TWS", "WWS", "YWS", "VWS", "AYS", "RYS", "NYS",
    "DYS", "CYS", "EYS", "QYS", "GYS", "HYS", "IYS", "LYS", "KYS",
    "MYS", "FYS", "PYS", "SYS", "TYS", "WYS", "YYS", "VYS", "AVS",
    "RVS", "NVS", "DVS", "CVS", "EVS", "QVS", "GVS", "HVS", "IVS",
    "LVS", "KVS", "MVS", "FVS", "PVS", "SVS", "TVS", "WVS", "YVS",
    "VVS", "AAT", "RAT", "NAT", "DAT", "CAT", "EAT", "QAT", "GAT",
    "HAT", "IAT", "LAT", "KAT", "MAT", "FAT", "PAT", "SAT", "TAT",
    "WAT", "YAT", "VAT", "ART", "RRT", "NRT", "DRT", "CRT", "ERT",
    "QRT", "GRT", "HRT", "IRT", "LRT", "KRT", "MRT", "FRT", "PRT",
    "SRT", "TRT", "WRT", "YRT", "VRT", "ANT", "RNT", "NNT", "DNT",
    "CNT", "ENT", "QNT", "GNT", "HNT", "INT", "LNT", "KNT", "MNT",
    "FNT", "PNT", "SNT", "TNT", "WNT", "YNT", "VNT", "ADT", "RDT",
    "NDT", "DDT", "CDT", "EDT", "QDT", "GDT", "HDT", "IDT", "LDT",
    "KDT", "MDT", "FDT", "PDT", "SDT", "TDT", "WDT", "YDT", "VDT",
    "ACT", "RCT", "NCT", "DCT", "CCT", "ECT", "QCT", "GCT", "HCT",
    "ICT", "LCT", "KCT", "MCT", "FCT", "PCT", "SCT", "TCT", "WCT",
    "YCT", "VCT", "AET", "RET", "NET", "DET", "CET", "EET", "QET",
    "GET", "HET", "IET", "LET", "KET", "MET", "FET", "PET", "SET",
    "TET", "WET", "YET", "VET", "AQT", "RQT", "NQT", "DQT", "CQT",
    "EQT", "QQT", "GQT", "HQT", "IQT", "LQT", "KQT", "MQT", "FQT",
    "PQT", "SQT", "TQT", "WQT", "YQT", "VQT", "AGT", "RGT", "NGT",
    "DGT", "CGT", "EGT", "QGT", "GGT", "HGT", "IGT", "LGT", "KGT",
    "MGT", "FGT", "PGT", "SGT", "TGT", "WGT", "YGT", "VGT", "AHT",
    "RHT", "NHT", "DHT", "CHT", "EHT", "QHT", "GHT", "HHT", "IHT",
    "LHT", "KHT", "MHT", "FHT", "PHT", "SHT", "THT", "WHT", "YHT",
    "VHT", "AIT", "RIT", "NIT", "DIT", "CIT", "EIT", "QIT", "GIT",
    "HIT", "IIT", "LIT", "KIT", "MIT", "FIT", "PIT", "SIT", "TIT",
    "WIT", "YIT", "VIT", "ALT", "RLT", "NLT", "DLT", "CLT", "ELT",
    "QLT", "GLT", "HLT", "ILT", "LLT", "KLT", "MLT", "FLT", "PLT",
    "SLT", "TLT", "WLT", "YLT", "VLT", "AKT", "RKT", "NKT", "DKT",
    "CKT", "EKT", "QKT", "GKT", "HKT", "IKT", "LKT", "KKT", "MKT",
    "FKT", "PKT", "SKT", "TKT", "WKT", "YKT", "VKT", "AMT", "RMT",
    "NMT", "DMT", "CMT", "EMT", "QMT", "GMT", "HMT", "IMT", "LMT",
    "KMT", "MMT", "FMT", "PMT", "SMT", "TMT", "WMT", "YMT", "VMT",
    "AFT", "RFT", "NFT", "DFT", "CFT", "EFT", "QFT", "GFT", "HFT",
    "IFT", "LFT", "KFT", "MFT", "FFT", "PFT", "SFT", "TFT", "WFT",
    "YFT", "VFT", "APT", "RPT", "NPT", "DPT", "CPT", "EPT", "QPT",
    "GPT", "HPT", "IPT", "LPT", "KPT", "MPT", "FPT", "PPT", "SPT",
    "TPT", "WPT", "YPT", "VPT", "AST", "RST", "NST", "DST", "CST",
    "EST", "QST", "GST", "HST", "IST", "LST", "KST", "MST", "FST",
    "PST", "SST", "TST", "WST", "YST", "VST", "ATT", "RTT", "NTT",
    "DTT", "CTT", "ETT", "QTT", "GTT", "HTT", "ITT", "LTT", "KTT",
    "MTT", "FTT", "PTT", "STT", "TTT", "WTT", "YTT", "VTT", "AWT",
    "RWT", "NWT", "DWT", "CWT", "EWT", "QWT", "GWT", "HWT", "IWT",
    "LWT", "KWT", "MWT", "FWT", "PWT", "SWT", "TWT", "WWT", "YWT",
    "VWT", "AYT", "RYT", "NYT", "DYT", "CYT", "EYT", "QYT", "GYT",
    "HYT", "IYT", "LYT", "KYT", "MYT", "FYT", "PYT", "SYT", "TYT",
    "WYT", "YYT", "VYT", "AVT", "RVT", "NVT", "DVT", "CVT", "EVT",
    "QVT", "GVT", "HVT", "IVT", "LVT", "KVT", "MVT", "FVT", "PVT",
    "SVT", "TVT", "WVT", "YVT", "VVT", "AAW", "RAW", "NAW", "DAW",
    "CAW", "EAW", "QAW", "GAW", "HAW", "IAW", "LAW", "KAW", "MAW",
    "FAW", "PAW", "SAW", "TAW", "WAW", "YAW", "VAW", "ARW", "RRW",
    "NRW", "DRW", "CRW", "ERW", "QRW", "GRW", "HRW", "IRW", "LRW",
    "KRW", "MRW", "FRW", "PRW", "SRW", "TRW", "WRW", "YRW", "VRW",
    "ANW", "RNW", "NNW", "DNW", "CNW", "ENW", "QNW", "GNW", "HNW",
    "INW", "LNW", "KNW", "MNW", "FNW", "PNW", "SNW", "TNW", "WNW",
    "YNW", "VNW", "ADW", "RDW", "NDW", "DDW", "CDW", "EDW", "QDW",
    "GDW", "HDW", "IDW", "LDW", "KDW", "MDW", "FDW", "PDW", "SDW",
    "TDW", "WDW", "YDW", "VDW", "ACW", "RCW", "NCW", "DCW", "CCW",
    "ECW", "QCW", "GCW", "HCW", "ICW", "LCW", "KCW", "MCW", "FCW",
    "PCW", "SCW", "TCW", "WCW", "YCW", "VCW", "AEW", "REW", "NEW",
    "DEW", "CEW", "EEW", "QEW", "GEW", "HEW", "IEW", "LEW", "KEW",
    "MEW", "FEW", "PEW", "SEW", "TEW", "WEW", "YEW", "VEW", "AQW",
    "RQW", "NQW", "DQW", "CQW", "EQW", "QQW", "GQW", "HQW", "IQW",
    "LQW", "KQW", "MQW", "FQW", "PQW", "SQW", "TQW", "WQW", "YQW",
    "VQW", "AGW", "RGW", "NGW", "DGW", "CGW", "EGW", "QGW", "GGW",
    "HGW", "IGW", "LGW", "KGW", "MGW", "FGW", "PGW", "SGW", "TGW",
    "WGW", "YGW", "VGW", "AHW", "RHW", "NHW", "DHW", "CHW", "EHW",
    "QHW", "GHW", "HHW", "IHW", "LHW", "KHW", "MHW", "FHW", "PHW",
    "SHW", "THW", "WHW", "YHW", "VHW", "AIW", "RIW", "NIW", "DIW",
    "CIW", "EIW", "QIW", "GIW", "HIW", "IIW", "LIW", "KIW", "MIW",
    "FIW", "PIW", "SIW", "TIW", "WIW", "YIW", "VIW", "ALW", "RLW",
    "NLW", "DLW", "CLW", "ELW", "QLW", "GLW", "HLW", "ILW", "LLW",
    "KLW", "MLW", "FLW", "PLW", "SLW", "TLW", "WLW", "YLW", "VLW",
    "AKW", "RKW", "NKW", "DKW", "CKW", "EKW", "QKW", "GKW", "HKW",
    "IKW", "LKW", "KKW", "MKW", "FKW", "PKW", "SKW", "TKW", "WKW",
    "YKW", "VKW", "AMW", "RMW", "NMW", "DMW", "CMW", "EMW", "QMW",
    "GMW", "HMW", "IMW", "LMW", "KMW", "MMW", "FMW", "PMW", "SMW",
    "TMW", "WMW", "YMW", "VMW", "AFW", "RFW", "NFW", "DFW", "CFW",
    "EFW", "QFW", "GFW", "HFW", "IFW", "LFW", "KFW", "MFW", "FFW",
    "PFW", "SFW", "TFW", "WFW", "YFW", "VFW", "APW", "RPW", "NPW",
    "DPW", "CPW", "EPW", "QPW", "GPW", "HPW", "IPW", "LPW", "KPW",
    "MPW", "FPW", "PPW", "SPW", "TPW", "WPW", "YPW", "VPW", "ASW",
    "RSW", "NSW", "DSW", "CSW", "ESW", "QSW", "GSW", "HSW", "ISW",
    "LSW", "KSW", "MSW", "FSW", "PSW", "SSW", "TSW", "WSW", "YSW",
    "VSW", "ATW", "RTW", "NTW", "DTW", "CTW", "ETW", "QTW", "GTW",
    "HTW", "ITW", "LTW", "KTW", "MTW", "FTW", "PTW", "STW", "TTW",
    "WTW", "YTW", "VTW", "AWW", "RWW", "NWW", "DWW", "CWW", "EWW",
    "QWW", "GWW", "HWW", "IWW", "LWW", "KWW", "MWW", "FWW", "PWW",
    "SWW", "TWW", "WWW", "YWW", "VWW", "AYW", "RYW", "NYW", "DYW",
    "CYW", "EYW", "QYW", "GYW", "HYW", "IYW", "LYW", "KYW", "MYW",
    "FYW", "PYW", "SYW", "TYW", "WYW", "YYW", "VYW", "AVW", "RVW",
    "NVW", "DVW", "CVW", "EVW", "QVW", "GVW", "HVW", "IVW", "LVW",
    "KVW", "MVW", "FVW", "PVW", "SVW", "TVW", "WVW", "YVW", "VVW",
    "AAY", "RAY", "NAY", "DAY", "CAY", "EAY", "QAY", "GAY", "HAY",
    "IAY", "LAY", "KAY", "MAY", "FAY", "PAY", "SAY", "TAY", "WAY",
    "YAY", "VAY", "ARY", "RRY", "NRY", "DRY", "CRY", "ERY", "QRY",
    "GRY", "HRY", "IRY", "LRY", "KRY", "MRY", "FRY", "PRY", "SRY",
    "TRY", "WRY", "YRY", "VRY", "ANY", "RNY", "NNY", "DNY", "CNY",
    "ENY", "QNY", "GNY", "HNY", "INY", "LNY", "KNY", "MNY", "FNY",
    "PNY", "SNY", "TNY", "WNY", "YNY", "VNY", "ADY", "RDY", "NDY",
    "DDY", "CDY", "EDY", "QDY", "GDY", "HDY", "IDY", "LDY", "KDY",
    "MDY", "FDY", "PDY", "SDY", "TDY", "WDY", "YDY", "VDY", "ACY",
    "RCY", "NCY", "DCY", "CCY", "ECY", "QCY", "GCY", "HCY", "ICY",
    "LCY", "KCY", "MCY", "FCY", "PCY", "SCY", "TCY", "WCY", "YCY",
    "VCY", "AEY", "REY", "NEY", "DEY", "CEY", "EEY", "QEY", "GEY",
    "HEY", "IEY", "LEY", "KEY", "MEY", "FEY", "PEY", "SEY", "TEY",
    "WEY", "YEY", "VEY", "AQY", "RQY", "NQY", "DQY", "CQY", "EQY",
    "QQY", "GQY", "HQY", "IQY", "LQY", "KQY", "MQY", "FQY", "PQY",
    "SQY", "TQY", "WQY", "YQY", "VQY", "AGY", "RGY", "NGY", "DGY",
    "CGY", "EGY", "QGY", "GGY", "HGY", "IGY", "LGY", "KGY", "MGY",
    "FGY", "PGY", "SGY", "TGY", "WGY", "YGY", "VGY", "AHY", "RHY",
    "NHY", "DHY", "CHY", "EHY", "QHY", "GHY", "HHY", "IHY", "LHY",
    "KHY", "MHY", "FHY", "PHY", "SHY", "THY", "WHY", "YHY", "VHY",
    "AIY", "RIY", "NIY", "DIY", "CIY", "EIY", "QIY", "GIY", "HIY",
    "IIY", "LIY", "KIY", "MIY", "FIY", "PIY", "SIY", "TIY", "WIY",
    "YIY", "VIY", "ALY", "RLY", "NLY", "DLY", "CLY", "ELY", "QLY",
    "GLY", "HLY", "ILY", "LLY", "KLY", "MLY", "FLY", "PLY", "SLY",
    "TLY", "WLY", "YLY", "VLY", "AKY", "RKY", "NKY", "DKY", "CKY",
    "EKY", "QKY", "GKY", "HKY", "IKY", "LKY", "KKY", "MKY", "FKY",
    "PKY", "SKY", "TKY", "WKY", "YKY", "VKY", "AMY", "RMY", "NMY",
    "DMY", "CMY", "EMY", "QMY", "GMY", "HMY", "IMY", "LMY", "KMY",
    "MMY", "FMY", "PMY", "SMY", "TMY", "WMY", "YMY", "VMY", "AFY",
    "RFY", "NFY", "DFY", "CFY", "EFY", "QFY", "GFY", "HFY", "IFY",
    "LFY", "KFY", "MFY", "FFY", "PFY", "SFY", "TFY", "WFY", "YFY",
    "VFY", "APY", "RPY", "NPY", "DPY", "CPY", "EPY", "QPY", "GPY",
    "HPY", "IPY", "LPY", "KPY", "MPY", "FPY", "PPY", "SPY", "TPY",
    "WPY", "YPY", "VPY", "ASY", "RSY", "NSY", "DSY", "CSY", "ESY",
    "QSY", "GSY", "HSY", "ISY", "LSY", "KSY", "MSY", "FSY", "PSY",
    "SSY", "TSY", "WSY", "YSY", "VSY", "ATY", "RTY", "NTY", "DTY",
    "CTY", "ETY", "QTY", "GTY", "HTY", "ITY", "LTY", "KTY", "MTY",
    "FTY", "PTY", "STY", "TTY", "WTY", "YTY", "VTY", "AWY", "RWY",
    "NWY", "DWY", "CWY", "EWY", "QWY", "GWY", "HWY", "IWY", "LWY",
    "KWY", "MWY", "FWY", "PWY", "SWY", "TWY", "WWY", "YWY", "VWY",
    "AYY", "RYY", "NYY", "DYY", "CYY", "EYY", "QYY", "GYY", "HYY",
    "IYY", "LYY", "KYY", "MYY", "FYY", "PYY", "SYY", "TYY", "WYY",
    "YYY", "VYY", "AVY", "RVY", "NVY", "DVY", "CVY", "EVY", "QVY",
    "GVY", "HVY", "IVY", "LVY", "KVY", "MVY", "FVY", "PVY", "SVY",
    "TVY", "WVY", "YVY", "VVY", "AAV", "RAV", "NAV", "DAV", "CAV",
    "EAV", "QAV", "GAV", "HAV", "IAV", "LAV", "KAV", "MAV", "FAV",
    "PAV", "SAV", "TAV", "WAV", "YAV", "VAV", "ARV", "RRV", "NRV",
    "DRV", "CRV", "ERV", "QRV", "GRV", "HRV", "IRV", "LRV", "KRV",
    "MRV", "FRV", "PRV", "SRV", "TRV", "WRV", "YRV", "VRV", "ANV",
    "RNV", "NNV", "DNV", "CNV", "ENV", "QNV", "GNV", "HNV", "INV",
    "LNV", "KNV", "MNV", "FNV", "PNV", "SNV", "TNV", "WNV", "YNV",
    "VNV", "ADV", "RDV", "NDV", "DDV", "CDV", "EDV", "QDV", "GDV",
    "HDV", "IDV", "LDV", "KDV", "MDV", "FDV", "PDV", "SDV", "TDV",
    "WDV", "YDV", "VDV", "ACV", "RCV", "NCV", "DCV", "CCV", "ECV",
    "QCV", "GCV", "HCV", "ICV", "LCV", "KCV", "MCV", "FCV", "PCV",
    "SCV", "TCV", "WCV", "YCV", "VCV", "AEV", "REV", "NEV", "DEV",
    "CEV", "EEV", "QEV", "GEV", "HEV", "IEV", "LEV", "KEV", "MEV",
    "FEV", "PEV", "SEV", "TEV", "WEV", "YEV", "VEV", "AQV", "RQV",
    "NQV", "DQV", "CQV", "EQV", "QQV", "GQV", "HQV", "IQV", "LQV",
    "KQV", "MQV", "FQV", "PQV", "SQV", "TQV", "WQV", "YQV", "VQV",
    "AGV", "RGV", "NGV", "DGV", "CGV", "EGV", "QGV", "GGV", "HGV",
    "IGV", "LGV", "KGV", "MGV", "FGV", "PGV", "SGV", "TGV", "WGV",
    "YGV", "VGV", "AHV", "RHV", "NHV", "DHV", "CHV", "EHV", "QHV",
    "GHV", "HHV", "IHV", "LHV", "KHV", "MHV", "FHV", "PHV", "SHV",
    "THV", "WHV", "YHV", "VHV", "AIV", "RIV", "NIV", "DIV", "CIV",
    "EIV", "QIV", "GIV", "HIV", "IIV", "LIV", "KIV", "MIV", "FIV",
    "PIV", "SIV", "TIV", "WIV", "YIV", "VIV", "ALV", "RLV", "NLV",
    "DLV", "CLV", "ELV", "QLV", "GLV", "HLV", "ILV", "LLV", "KLV",
    "MLV", "FLV", "PLV", "SLV", "TLV", "WLV", "YLV", "VLV", "AKV",
    "RKV", "NKV", "DKV", "CKV", "EKV", "QKV", "GKV", "HKV", "IKV",
    "LKV", "KKV", "MKV", "FKV", "PKV", "SKV", "TKV", "WKV", "YKV",
    "VKV", "AMV", "RMV", "NMV", "DMV", "CMV", "EMV", "QMV", "GMV",
    "HMV", "IMV", "LMV", "KMV", "MMV", "FMV", "PMV", "SMV", "TMV",
    "WMV", "YMV", "VMV", "AFV", "RFV", "NFV", "DFV", "CFV", "EFV",
    "QFV", "GFV", "HFV", "IFV", "LFV", "KFV", "MFV", "FFV", "PFV",
    "SFV", "TFV", "WFV", "YFV", "VFV", "APV", "RPV", "NPV", "DPV",
    "CPV", "EPV", "QPV", "GPV", "HPV", "IPV", "LPV", "KPV", "MPV",
    "FPV", "PPV", "SPV", "TPV", "WPV", "YPV", "VPV", "ASV", "RSV",
    "NSV", "DSV", "CSV", "ESV", "QSV", "GSV", "HSV", "ISV", "LSV",
    "KSV", "MSV", "FSV", "PSV", "SSV", "TSV", "WSV", "YSV", "VSV",
    "ATV", "RTV", "NTV", "DTV", "CTV", "ETV", "QTV", "GTV", "HTV",
    "ITV", "LTV", "KTV", "MTV", "FTV", "PTV", "STV", "TTV", "WTV",
    "YTV", "VTV", "AWV", "RWV", "NWV", "DWV", "CWV", "EWV", "QWV",
    "GWV", "HWV", "IWV", "LWV", "KWV", "MWV", "FWV", "PWV", "SWV",
    "TWV", "WWV", "YWV", "VWV", "AYV", "RYV", "NYV", "DYV", "CYV",
    "EYV", "QYV", "GYV", "HYV", "IYV", "LYV", "KYV", "MYV", "FYV",
    "PYV", "SYV", "TYV", "WYV", "YYV", "VYV", "AVV", "RVV", "NVV",
    "DVV", "CVV", "EVV", "QVV", "GVV", "HVV", "IVV", "LVV", "KVV",
    "MVV", "FVV", "PVV", "SVV", "TVV", "WVV", "YVV", "VVV"
  )

  xSplitted <- strsplit(x, split = "")[[1]]
  n <- nchar(x)
  TC <- summary(factor(
    paste(paste(xSplitted[-c(n, n - 1)], xSplitted[-c(1, n)], sep = ""),
          xSplitted[-c(1, 2)],
          sep = ""
    ),
    levels = TCDict
  ), maxsum = 8001) / (n - 2)

  TC
}
