% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scan1snps.R
\name{scan1snps}
\alias{scan1snps}
\title{Single-QTL genome scan at imputed SNPs}
\usage{
scan1snps(
  genoprobs,
  map,
  pheno,
  kinship = NULL,
  addcovar = NULL,
  Xcovar = NULL,
  intcovar = NULL,
  weights = NULL,
  reml = TRUE,
  model = c("normal", "binary"),
  query_func = NULL,
  chr = NULL,
  start = NULL,
  end = NULL,
  snpinfo = NULL,
  batch_length = 20,
  keep_all_snps = FALSE,
  cores = 1,
  ...
)
}
\arguments{
\item{genoprobs}{Genotype probabilities as calculated by
\code{\link[=calc_genoprob]{calc_genoprob()}}.}

\item{map}{Physical map for the positions in the \code{genoprobs}
object: A list of numeric vectors; each vector gives marker
positions for a single chromosome.}

\item{pheno}{A numeric matrix of phenotypes, individuals x phenotypes.}

\item{kinship}{Optional kinship matrix, or a list of kinship matrices (one
per chromosome), in order to use the LOCO (leave one chromosome
out) method.}

\item{addcovar}{An optional numeric matrix of additive covariates.}

\item{Xcovar}{An optional numeric matrix with additional additive covariates used for
null hypothesis when scanning the X chromosome.}

\item{intcovar}{An optional numeric matrix of interactive covariates.}

\item{weights}{An optional numeric vector of positive weights for the
individuals. As with the other inputs, it must have \code{names}
for individual identifiers.}

\item{reml}{If \code{kinship} provided: if \code{reml=TRUE}, use
REML; otherwise maximum likelihood.}

\item{model}{Indicates whether to use a normal model (least
squares) or binary model (logistic regression) for the phenotype.
If \code{model="binary"}, the phenotypes must have values in \eqn{[0, 1]}.}

\item{query_func}{Function for querying SNP information; see
\code{\link[=create_variant_query_func]{create_variant_query_func()}}). Takes arguments
\code{chr}, \code{start}, \code{end}, (with \code{start} and \code{end} in the units in
\code{map}, generally Mbp), and returns a data frame containing
the columns \code{snp}, \code{chr}, \code{pos}, and \code{sdp}. (See \code{snpinfo} below.)}

\item{chr}{Chromosome or chromosomes to scan}

\item{start}{Position defining the start of an interval to scan.
Should be a single number, and if provided, \code{chr} should also
have length 1.}

\item{end}{Position defining the end of an interval to scan.
Should be a single number, and if provided, \code{chr} should also
have length 1.}

\item{snpinfo}{Optional data frame of SNPs to scan; if provided,
\code{query_func}, \code{chr}, \code{start}, and \code{end} are ignored. Should
contain the following columns:
\itemize{
\item \code{chr} - Character string or factor with chromosome
\item \code{pos} - Position (in same units as in the \code{"map"}).
\item \code{sdp} - Strain distribution pattern: an integer, between
1 and \eqn{2^n - 2} where \eqn{n} is the number of strains, whose
binary encoding indicates the founder genotypes
\item \code{snp} - Character string with SNP identifier (if
missing, the rownames are used).
}}

\item{batch_length}{Interval length (in units of \code{map}, generally
Mbp) to scan at one time.}

\item{keep_all_snps}{SNPs are grouped into equivalence classes based
on position and founder genotypes; if \code{keep_all_snps=FALSE},
the return value will contain information only on the indexed
SNPs (one per equivalence class).}

\item{cores}{Number of CPU cores to use, for parallel calculations.
(If \code{0}, use \code{\link[parallel:detectCores]{parallel::detectCores()}}.)
Alternatively, this can be links to a set of cluster sockets, as
produced by \code{\link[parallel:makeCluster]{parallel::makeCluster()}}.}

\item{...}{Additional control parameters passed to \code{\link[=scan1]{scan1()}}}
}
\value{
A list with two components: \code{lod} (matrix of LOD scores)
and \code{snpinfo} (a data frame of SNPs that were scanned,
including columns \code{index} which indicates groups of equivalent
SNPs)
}
\description{
Perform a single-QTL scan across the genome or a defined region at
SNPs genotyped in the founders, by Haley-Knott regression or a
liear mixed model, with possible allowance for covariates.
}
\details{
The analysis proceeds as follows:
\itemize{
\item Call \code{query_func()} to grab all SNPs over a region.
\item Use \code{\link[=index_snps]{index_snps()}} to group SNPs into equivalence classes.
\item Use \code{\link[=genoprob_to_snpprob]{genoprob_to_snpprob()}} to convert \code{genoprobs} to SNP probabilities.
\item Use \code{\link[=scan1]{scan1()}} to do a single-QTL scan at the SNPs.
}
}
\examples{
\dontrun{
# load example data and calculate genotype probabilities
file <- paste0("https://raw.githubusercontent.com/rqtl/",
               "qtl2data/master/DOex/DOex.zip")
DOex <- read_cross2(file)
probs <- calc_genoprob(DOex, error_prob=0.002)

snpdb_file <- system.file("extdata", "cc_variants_small.sqlite", package="qtl2")
queryf <- create_variant_query_func(snpdb_file)

out <- scan1snps(probs, DOex$pmap, DOex$pheno, query_func=queryf, chr=2, start=97, end=98)
}

}
\seealso{
\code{\link[=scan1]{scan1()}}, \code{\link[=genoprob_to_snpprob]{genoprob_to_snpprob()}}, \code{\link[=index_snps]{index_snps()}}, \code{\link[=create_variant_query_func]{create_variant_query_func()}}, \code{\link[=plot_snpasso]{plot_snpasso()}}
}
