% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/find_peaks.R
\name{find_peaks}
\alias{find_peaks}
\title{Find peaks in a set of LOD curves}
\usage{
find_peaks(
  scan1_output,
  map,
  threshold = 3,
  peakdrop = Inf,
  drop = NULL,
  prob = NULL,
  thresholdX = NULL,
  peakdropX = NULL,
  dropX = NULL,
  probX = NULL,
  expand2markers = TRUE,
  sort_by = c("column", "pos", "lod"),
  cores = 1
)
}
\arguments{
\item{scan1_output}{An object of class \code{"scan1"} as returned by
\code{\link[=scan1]{scan1()}}.}

\item{map}{A list of vectors of marker positions, as produced by
\code{\link[=insert_pseudomarkers]{insert_pseudomarkers()}}. Can also be an indexed SNP info table,
as from \code{\link[=index_snps]{index_snps()}} or \code{\link[=scan1snps]{scan1snps()}}.}

\item{threshold}{Minimum LOD score for a peak (can be a vector with
separate thresholds for each lod score column in
\code{scan1_output})}

\item{peakdrop}{Amount that the LOD score must drop between peaks,
if multiple peaks are to be defined on a chromosome. (Can be a vector with
separate values for each lod score column in
\code{scan1_output}.)}

\item{drop}{If provided, LOD support intervals are included in the
results, and this indicates the amount to drop in the support
interval. (Can be a vector with
separate values for each lod score column in
\code{scan1_output}.) Must be \eqn{\le} \code{peakdrop}}

\item{prob}{If provided, Bayes credible intervals are included in the
results, and this indicates the nominal coverage.
(Can be a vector with
separate values for each lod score column in
\code{scan1_output}.) Provide just one of \code{drop} and \code{prob}.}

\item{thresholdX}{Separate threshold for the X chromosome; if
unspecified, the same threshold is used for both autosomes and the
X chromosome. (Like \code{threshold}, this can be a vector with
separate thresholds for each lod score column.)}

\item{peakdropX}{Like \code{peakdrop}, but for the X chromosome; if
unspecified, the same value is used for both autosomes and the X
chromosome.  (Can be a vector with separate values for each lod
score column in \code{scan1_output}.)}

\item{dropX}{Amount to drop for LOD support intervals on the X
chromosome.  Ignored if \code{drop} is not provided. (Can be a
vector with separate values for each lod score column in
\code{scan1_output}.)}

\item{probX}{Nominal coverage for Bayes intervals on the X
chromosome.  Ignored if \code{prob} is not provided. (Can be a
vector with separate values for each lod score column in
\code{scan1_output}.)}

\item{expand2markers}{If TRUE (and if \code{drop} or \code{prob} is
provided, so that QTL intervals are calculated), QTL intervals are
expanded so that their endpoints are at genetic markers.}

\item{sort_by}{Indicates whether to sort the rows by lod column,
genomic position, or LOD score.}

\item{cores}{Number of CPU cores to use, for parallel calculations.
(If \code{0}, use \code{\link[parallel:detectCores]{parallel::detectCores()}}.)
Alternatively, this can be links to a set of cluster sockets, as
produced by \code{\link[parallel:makeCluster]{parallel::makeCluster()}}.}
}
\value{
A data frame with each row being a single peak on a single
chromosome for a single LOD score column, and with columns
\itemize{
\item \code{lodindex} - lod column index
\item \code{lodcolumn} - lod column name
\item \code{chr} - chromosome ID
\item \code{pos} - peak position
\item \code{lod} - lod score at peak
}

If \code{drop} or \code{prob} is provided, the results will include
two additional columns: \code{ci_lo} and \code{ci_hi}, with the
endpoints of the LOD support intervals or Bayes credible wintervals.
}
\description{
Find peaks in a set of LOD curves (output from \code{\link[=scan1]{scan1()}}
}
\details{
For each lod score column on each chromosome, we return a
set of peaks defined as local maxima that exceed the specified
\code{threshold}, with the requirement that the LOD score must have
dropped by at least \code{peakdrop} below the lowest of any two
adjacent peaks.

At a given peak, if there are ties, with multiple positions jointly
achieving the maximum LOD score, we take the average of these
positions as the location of the peak.
}
\examples{
# read data
iron <- read_cross2(system.file("extdata", "iron.zip", package="qtl2"))
\dontshow{iron <- iron[,c(1,2,7,8,9,13,15,16,19)]}

# insert pseudomarkers into map
map <- insert_pseudomarkers(iron$gmap, step=1)

# calculate genotype probabilities
probs <- calc_genoprob(iron, map, error_prob=0.002)

# grab phenotypes and covariates; ensure that covariates have names attribute
pheno <- iron$pheno
covar <- match(iron$covar$sex, c("f", "m")) # make numeric
names(covar) <- rownames(iron$covar)
Xcovar <- get_x_covar(iron)

# perform genome scan
out <- scan1(probs, pheno, addcovar=covar, Xcovar=Xcovar)

# find just the highest peak on each chromosome
find_peaks(out, map, threshold=3)

# possibly multiple peaks per chromosome
find_peaks(out, map, threshold=3, peakdrop=1)

# possibly multiple peaks, also getting 1-LOD support intervals
find_peaks(out, map, threshold=3, peakdrop=1, drop=1)

# possibly multiple peaks, also getting 90\% Bayes intervals
find_peaks(out, map, threshold=3, peakdrop=1, prob=0.9)
}
\seealso{
\code{\link[=scan1]{scan1()}}, \code{\link[=lod_int]{lod_int()}}, \code{\link[=bayes_int]{bayes_int()}}
}
