% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/buildTree.R
\name{buildTree}
\alias{buildTree}
\title{Build and check tree data for subsequent use in
BDAT Fortran subroutines}
\usage{
buildTree(tree, check = NULL, vars = NULL, mapping = NULL)
}
\arguments{
\item{tree}{either a data.frame or a list containing the variables needed,
i.e. spp, D1, H and optionally H1, D2, H2. See details for more
information and parameter \code{mapping} for mapping of variable names.}

\item{check}{character vector which indicates the type of required output and
determines the checks to be done}

\item{vars}{named list with additional variables for the specific
BDAT-functions; see \code{\link{getDiameter}}, \code{\link{getHeight}},
\code{\link{getVolume}}, \code{\link{getBiomass}}, \code{\link{getBark}},
\code{\link{getForm}} and \code{\link{getAssortment}}.
These variables might be included to \code{tree} as well, see details.}

\item{mapping}{mapping of variable names in case a data.frame is given into
parameter \code{tree} and \code{vars} between final \code{colnames(tree)} and
required parameter names. See details.}
}
\value{
a data.frame of class datBDAT.<check> having all variables needed in
specific functions. If \code{check} is NULL, only a basic tree-data.frame of
class "datBDAT" is returned.
}
\description{
this functions takes the data provided and builds a data.frame
to be used in other BDAT get*-functions. It discriminates between different
type of required output via the \code{check}-parameter. Checks are done on
the type and range of the variables given to make sure calls to the
Fortran routines do not suffer from type-mismatch (with potential freezing
of R).
}
\details{
Parameter \code{tree} is able to take either a data.frame with
correct variables names or arbitrary names if \code{mapping} is provided to
map the data.frame names to the required names by
\code{c("df-colname" = "var-name")} or to take a named list. If same-named
variables are present in both \code{tree} and \code{vars}, priority is put
on the ones in \code{vars} since explicitly given.

Possible variables are (*=required, depending on function):
\itemize{
  \item spp*: numeric, 1 <= spp <= 36, see \code{\link{getSpeciesCode}}
  \item D1*: numeric, first measured diameter [cm], usually at 1.3m
  \item H1: numeric, height of first measured diameter [m], if zero,
  internally transformed to 1.3m
  \item D2: numeric, second measured diameter [cm], or form parameter: latter
  is defined in conjunction with \code{H2}:
  \itemize{
    \item D2=0 and H2=0 => taper form of volume tables according to
    Grundner & Schwappach (1906-1938), the default
    \item D2=0 and 0 < H2 < 100 => german NFI1-taper form, with H2 given as
    percentile of the NFI1-\eqn{q_{0.30}}{q0.30}-distribution; H2=50
    corresponds to mean NFI1 taper form, H2<50 to slenderly and H2>50 to
    thicker trees; see \code{\link{getForm}} for more information about
    \eqn{q_{0.30}}{q0.30}
    \item D2=0 and H2>100 => mean NFI1 taper form
    \item D2>0 and H2=0 => D2 is a diameter and H2 is assumed to be 7m
    \item D2>0 and H2>0 => D2 and H2 are given as diameter and height
    \item -1<D2<0 => abs(D2) is interpreted as \eqn{q_{0.30}}{q0.30}
    \item -1>D2 => mean NFI1 taper form
  }
  \item H2: numeric, height of second measured diameter [m], or in
  conjunction with \code{D2}, see there.
  \item H*: numeric, tree height [m]
  \item A*: numeric, lower diameter [cm] or height [m] of section for which
  volume should be calculated, interpretation depends on \code{iAB}, see
  \code{\link{getVolume}}
  \item B*: numeric, upper diameter [cm] or height [m] of section for which
  volume should be calculated, interpretation depends on \code{iAB}, see
  \code{\link{getVolume}}
  \item sl: numeric, length of section over which should be integrated,
  defaults to 2.0m
  \item Dx*: diameter for which height or bark thickness is required
  \item Hx*: height for which diameter is required
  \item inv: inventory for which mean q03 is required, defaults to 1, see
  \code{\link{getForm}}
}

For deriving assortments, the following variables are optional (if not given,
default values are used):
\itemize{
  \item lX: length of unusable wood at stem foot [m], defaults to 0 (X-Holz)
  \item Hkz: indicator for tree top, 0 - normal (default), 1 - Wipfelbruch, 2 - Gipfelbruch
  \itemize{
    \item 0 => H=H
    \item 1 => H=H+2
    \item 2 => DBH < 30 => H=DBH; dbh > 30 => H = 30 + (DBH-30) * 0.3
  }
  \item Skz: indicator for stem type, defaults to 0
  \itemize{
    \item 0 => conifer trees => no restriction; deciduous trees => no assortments
    \item 1 => monopodial deciduous trees => Hsh = 0.7*H
    \item 2 => branching between dbh and 7m => Hsh = 5m
    \item 3 => crown base < 3m => Hsh=0.1
    \item 4 => dead or broken stem => Az = H*0.7
    \item 5 => dead tree => non-usable wood
  }
  \item Hsh: usable stem height, defaults to 0, i.e. 0.7*H
  \item Az: minimum cutting diameter over bark [cm], defaults to 0,
  using an exponential function given DBH
  \item Zsh: minimum cutting diameter under bark for stem wood [cm], defaults
  to 0, using parameter \code{Az} if estimated length < maximum length (i.e. 20m)
  \item Zab: minimum cutting diameter under bark for top segment [cm], defaults
  to 0, i.e. 14cm under bark
  \item Sokz: type assortment calculation, 0 - no assortment,
  1 - Mid diameter (Mittenstärke), 2 - Heilbronner Sortierung, defaults to 1
  \item fixN: number of fixed length assortments at stem foot, defaults to 0
  (no fixed length assortments, irrespective of other fix* parameters)
  \item fixZ: mininum diameter under bark for fixed length assortment at
  stem foot, defaults to 0
  \item fixL: length of fixed length assortment at stem foot, defaults to 0
  \item fixA: fixed length assortement add-on in [cm], defaults to 0
  \item fixR: fixed length assortement add-on in [\%], defaults to 0
}

If parameter \code{tree} is used to hand over all tree data in form of
a data.frame, at least the parameter spp, D1, H must be provided, eventually
mapped via \code{mapping}.
Parameter \code{Hx} and \code{Dx}, which specify height and diameter for
which a diameter or height is requested, respectively, can either be included
to the definition of the tree data or alternatively given separately using
the vars parameter. In that case, vars is used in priority to a identically
named variable in \code{tree}. Additionally, \code{tree} and \code{vars} are
merged via a full outer join.
The add-on in fixed length assortments can be given in absolute and relative
units at the same time, but the higher value will be used.
}
\examples{
## example for only tree data
tree <- list(spp = c(1, 1), D1 = c(30, 25), H = c(25, 30))
res <- buildTree(tree = tree)
head(res)
class(res)

tree <- list(species = c(1, 1), dbh = c(30, 25), h = c(25, 30))
mapping <- c("species" = "spp", "dbh" = "D1", "h" = "H")
res <- buildTree(tree = tree, mapping = mapping)
head(res)
class(res)

## example for diameter calculation
tree <- list(spp = c(1, 1), D1 = c(30, 25), H = c(25, 30))
vars <- list(Hx = c(1.3, 1.3))
mapping <- NULL
res <- buildTree(tree = tree, check = "diameter", vars = vars)
head(res)
class(res)
tree <- list(Art = c(1, 1), Bhd = c(30, 25), H = c(25, 30))
vars <- list(X = c(1.3, 1.3))
mapping <- c("Art" = "spp", "Bhd" = "D1", "X" = "Hx")
res <- buildTree(tree = tree, check = "diameter", vars = vars, mapping = mapping)
head(res)
class(res)

## example with many diameters for one tree
tree <- list(spp = c(1), D1 = c(30), H = c(25))
vars <- list(Hx = seq(0, 25, 0.1))
mapping <- NULL
res <- buildTree(tree = tree, check = "diameter", vars = vars)

tree <- data.frame(s = 1, d = 30, h = 25, hx = 1.3)
mapping <- c("s" = "spp", "d" = "D1", "h" = "H", "hx" = "Hx")
res <- buildTree(tree, check = "diameter", mapping = mapping)
head(res)
class(res)

## example for height calculation
tree <- list(spp = c(1, 1), D1 = c(30, 25), H = c(25, 30))
vars <- list(Dx = c(30, 25))
res <- buildTree(tree = tree, check = "height", vars = vars)
head(res)
class(res)

## example for volume calculation
tree <- list(spp = c(1, 1), D1 = c(30, 25), H = c(25, 30))
check <- "volume"
vars <- list(A = c(30, 25), B = c(7, 7), sl = 0.1)
mapping <- NULL
res <- buildTree(tree = tree, check = "volume", vars = vars)
head(res)
class(res)

## example for bark calculation
tree <- list(spp = c(1, 1), D1 = c(30, 25), H = c(25, 30))
vars <- list(Hx = c(1.3, 1.3))
res <- buildTree(tree = tree, check = "bark", vars = vars)
head(res)
class(res)

## example for assortment calculation
tree <- list(spp = c(1, 1), D1 = c(30, 25), H = c(25, 30))
vars <- list(fixN = 1, fixZ = 10, fixL = 5, fixA = 10, fixR = 0.1)
res <- buildTree(tree = tree, check = "assortment", vars = vars)
head(res)
class(res)

## for cases where 'vars' could be a vector (i.e. getBark, getDiameter and
## getHeight), the following is also possible
tree <- list(spp = c(1, 1), D1 = c(30, 25), H = c(25, 30))
vars <- c(1.3, 1.3)
res <- buildTree(tree = tree, check = "bark", vars = vars)
head(res)
class(res)

res <- buildTree(tree = tree, check = "height", vars = vars)
head(res)
class(res)

## but it is not possible in case of getVolume or getAssortment
## instead, use a named list to achieve a cross join / cartesian product
vars <- list(A = rep(1, 3), B = 5:7)
res <- buildTree(tree = tree, check = "volume", vars = vars)
head(res)
class(res)

## example for 'biomass' calculation
tree <- list(spp = c(1, 1), D1 = c(30, 25), H = c(25, 30))
res <- buildTree(tree = tree, check = "biomass")
head(res)
class(res)

## example with H1 != 1.3m
tree <- list(
  spp = c(1, 1), D1 = c(30, 25), H1 = c(2, 2), H = c(25, 30)
)
res <- buildTree(tree = tree, check = "biomass")
head(res)
class(res)
getBiomass(res)
}
