% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/swirl_2d.R
\name{swirl_2d}
\alias{swirl_2d}
\title{Swirl the values around an origin in 2 dimensions}
\usage{
swirl_2d(
  data,
  radius,
  x_col = NULL,
  y_col = NULL,
  suffix = "_swirled",
  origin = NULL,
  origin_fn = NULL,
  scale_fn = identity,
  keep_original = TRUE,
  degrees_col_name = ".degrees",
  radius_col_name = ".radius",
  origin_col_name = ".origin",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{radius}{Radius of the most-inner swirl on the x-axis in the \emph{simplest} case.
A negative number changes the direction to clockwise rotation.
Can be a \code{vector} with multiple radiuses.

Note: With a custom \code{`scaling_fn`}, this might not be the actual swirl radius anymore. Think of
it more as a width setting where a larger number leads to fewer full rotations.}

\item{x_col}{Name of x column in \code{`data`}. If \code{NULL} and \code{`data`} is a \code{vector},
the index of \code{`data`} is used. If \code{`data`} is a \code{data.frame}, it must be specified.}

\item{y_col}{Name of y column in \code{`data`}. If \code{`data`} is a \code{data.frame}, it must be specified.}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{origin}{Coordinates of the origin to swirl around.
\code{Vector} with 2 elements (i.e. origin_x, origin_y).
Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{scale_fn}{Function for scaling the distances before calculating the degrees.

\strong{Input}: A \code{numeric vector} (the distances).

\strong{Output}: A \code{numeric vector} (the scaled distances) of the same length.

E.g.:

\code{function(d)\{}

\verb{  }\code{d ^ 1.5}

\code{\}}}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{degrees_col_name}{Name of new column with the degrees. If \code{NULL}, no column is added.}

\item{radius_col_name}{Name of new column with the radius. If \code{NULL}, no column is added.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with three new columns containing
the swirled x- and y-values, the degrees, the radius, and the origin coordinates.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

The values are swirled counterclockwise around a specified origin.
The swirling is done by rotating around the origin with the degrees based
on the distances to the origin as so: \deqn{degrees = scale_fn(distances) / (2 * radius) * 360}

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and swirling around e.g. the centroid
of each group.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(ggplot2)

# Set seed
set.seed(4)

# Create a data frame
df <- data.frame(
  "x" = 1:50,
  "y" = 1,
  "r1" = runif(50),
  "r2" = runif(50) * 35,
  "g" = rep(1:5, each = 10)
)

# Swirl values around (0, 0)
swirl_2d(
  data = df,
  radius = 45,
  x_col = "x",
  y_col = "y",
  origin = c(0, 0)
)

\donttest{
# Swirl around the centroid
# with 6 different radius settings
# Scale the distances with custom function
df_swirled <- swirl_2d(
  data = df,
  radius = c(95, 96, 97, 98, 99, 100),
  x_col = "x",
  y_col = "y",
  origin_fn = centroid,
  scale_fn = function(d) {
    d^1.6
  }
)

df_swirled

# Plot swirls
df_swirled \%>\%
  ggplot(aes(x = x_swirled, y = y_swirled, color = factor(.radius))) +
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = ".radius")
}

#
# Swirl random data
# The trick lies in finding the right radius
#

# Swirl the random columns
df_swirled <- swirl_2d(
  data = df,
  radius = 5,
  x_col = "r1",
  y_col = "r2",
  origin_fn = centroid
)

# Plot swirls
df_swirled \%>\%
  ggplot(aes(x = r1_swirled, y = r2_swirled)) +
  geom_point() +
  theme_minimal() +
  labs(x = "r1", y = "r2")
}
\seealso{
Other mutate functions: 
\code{\link{apply_transformation_matrix}()},
\code{\link{cluster_groups}()},
\code{\link{dim_values}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{flip_values}()},
\code{\link{roll_values}()},
\code{\link{rotate_2d}()},
\code{\link{rotate_3d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_3d}()}

Other rotation functions: 
\code{\link{rotate_2d}()},
\code{\link{rotate_3d}()},
\code{\link{swirl_3d}()}

Other distance functions: 
\code{\link{closest_to}()},
\code{\link{dim_values}()},
\code{\link{distance}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{furthest_from}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{distance functions}
\concept{mutate functions}
\concept{rotation functions}
