% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/by_distance.R
\name{closest_to}
\alias{closest_to}
\alias{closest_to_vec}
\title{Orders values by shortest distance to an origin}
\usage{
closest_to(
  data,
  cols = NULL,
  origin = NULL,
  origin_fn = NULL,
  shuffle_ties = FALSE,
  origin_col_name = ".origin",
  distance_col_name = ".distance",
  overwrite = FALSE
)

closest_to_vec(data, origin = NULL, origin_fn = NULL, shuffle_ties = FALSE)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{cols}{Column(s) to create sorting factor by. When \code{NULL} and \code{`data`} is a \code{data.frame},
the row numbers are used.}

\item{origin}{Coordinates of the origin to calculate distances to.
A scalar to use in all dimensions
or a \code{vector} with one scalar per dimension.

\strong{N.B.} Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{shuffle_ties}{Whether to shuffle elements with the same distance to the origin. (Logical)}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{distance_col_name}{Name of new column with the distances to the origin. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
The sorted \code{data.frame} (\code{tibble}) / \code{vector}.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Values are ordered by how close they are to the origin.

In 1d (when
\code{`cols`} has length \code{1}), the origin can be thought of as a target value.
In \emph{n} dimensions, the origin can be thought of as coordinates.

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and ordering the rows by
their distance to the centroid of each group.

The \code{*_vec()} version takes and returns a \code{vector}.

\strong{Example}:

The column values:

\code{c(1, 2, 3, 4, 5)}

and \code{origin = 2}

are \strong{ordered as}:

\code{c(}\strong{\code{2}}\code{, 1, 3, 4, 5)}
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "index" = 1:10,
  "A" = sample(1:10),
  "B" = runif(10),
  "G" = c(
    1, 1, 1, 2, 2,
    2, 3, 3, 3, 3
  ),
  stringsAsFactors = FALSE
)

# Closest to 3 in a vector
closest_to_vec(1:10, origin = 3)

# Closest to the third row (index of data.frame)
closest_to(df, origin = 3)$index

# By each of the columns
closest_to(df, cols = "A", origin = 3)$A
closest_to(df, cols = "A", origin_fn = most_centered)$A
closest_to(df, cols = "B", origin = 0.5)$B
closest_to(df, cols = "B", origin_fn = centroid)$B

# Shuffle the elements with the same distance to the origin
closest_to(df,
  cols = "A",
  origin_fn = create_origin_fn(median),
  shuffle_ties = TRUE
)$A

# Grouped by G
df \%>\%
  dplyr::select(G, A) \%>\% # For clarity
  dplyr::group_by(G) \%>\%
  closest_to(
    cols = "A",
    origin_fn = create_origin_fn(median)
  )

# Plot the rearranged values
plot(
  x = 1:10,
  y = closest_to(df,
    cols = "B",
    origin_fn = create_origin_fn(median)
  )$B,
  xlab = "Position",
  ylab = "B"
)
plot(
  x = 1:10,
  y = closest_to(df,
    cols = "A",
    origin_fn = create_origin_fn(median),
    shuffle_ties = TRUE
  )$A,
  xlab = "Position",
  ylab = "A"
)

# In multiple dimensions
df \%>\%
  closest_to(cols = c("A", "B"), origin_fn = most_centered)
}
\seealso{
Other rearrange functions: 
\code{\link{center_max}()},
\code{\link{center_min}()},
\code{\link{furthest_from}()},
\code{\link{pair_extremes}()},
\code{\link{position_max}()},
\code{\link{position_min}()},
\code{\link{rev_windows}()},
\code{\link{roll_elements}()},
\code{\link{shuffle_hierarchy}()}

Other distance functions: 
\code{\link{dim_values}()},
\code{\link{distance}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{furthest_from}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{distance functions}
\concept{rearrange functions}
