#' @name genericApiCall
#' @title Generic Interface the REDCap API.
#'   
#' @description Permits users to make generic calls to the REDCap
#'   API. This allows use of API methods that do not yet have
#'   dedicated support.
#'   
#' @param rcon A REDCap connection object as generated by 
#'   \code{redcapConnection}.
#' @param content \code{character(1)} The content argument for the
#'   API call.
#' @param make_data_frame \code{logical(1)}. When \code{TRUE}, an
#'   attempt is made to coerce the output to a data frame with
#'   \code{read.csv}.  Otherwise, it is returned as a character
#'   vector.
#' @param colClasses A named list of column names and
#'   classes to apply via \code{read.csv}
#' @param returnFormat \code{character(1)} The format for the return.
#'   Defaults to \code{"csv"}.
#' @param ... Additional named arguments giving arguments to the
#'   API method. 
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#'   
#' @export

genericApiCall <- function(rcon,
                           content,
                           make_data_frame = TRUE,
                           colClasses = NA,
                           returnFormat = "csv",
                           ...)
{
  UseMethod("genericApiCall")
}

#' @rdname genericApiCall
#' @export

genericApiCall.redcapApiConnection <- function(rcon,
                                               content,
                                               make_data_frame = TRUE,
                                               colClasses = NA,
                                               returnFormat = "csv",
                                               ...,
                                               error_handling = getOption("redcap_error_handling")){
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  checkmate::assert_character(x = content,
                              len = 1,
                              add = coll)
  
  checkmate::assert_logical(x = make_data_frame,
                            len = 1,
                            add = coll)
  
  checkmate::assert_character(x = returnFormat,
                              len = 1,
                              add = coll)
  
  checkmate::reportAssertions(coll)
  
  body <- list(...)
  body[["token"]] <- rcon[["token"]]
  body[["content"]] <- content
  body[["returnFormat"]] <- returnFormat
  
  x <- httr::POST(url = rcon$url, 
                  body = body, 
                  config = rcon$config)
  
  if (x$status_code != 200) return(redcap_error(x, error_handling))
  
  x <- as.character(x)
  
  if (make_data_frame){
    tryCatch(utils::read.csv(text = x,
                             colClasses = NA,
                             stringsAsFactors = FALSE),
             error = function(cond) 
               stop("An error occurred while coercing the result to a data frame: ",
                    cond))
  }
  else
  {
    x
  }
}
