% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nice_table.R
\name{nice_table}
\alias{nice_table}
\title{Easily make nice APA tables}
\usage{
nice_table(
  data,
  highlight = FALSE,
  stars = TRUE,
  italics,
  col.format.p,
  col.format.r,
  col.format.ci,
  format.custom,
  col.format.custom,
  width = NULL,
  broom = NULL,
  report = NULL,
  short = FALSE,
  title,
  note,
  separate.header
)
}
\arguments{
\item{data}{The data frame, to be converted to a flextable.
The data frame cannot have duplicate column names.}

\item{highlight}{Highlight rows with statistically significant
results? Requires a column named "p" containing p-values.
Can either accept logical (TRUE/FALSE) OR a numeric value for
a custom critical p-value threshold (e.g., 0.10 or 0.001).}

\item{stars}{Logical. Whether to add asterisks for significant p values.}

\item{italics}{Which columns headers should be italic? Useful
for column names that should be italic but that are not picked
up automatically by the function. Select with numerical range, e.g., 1:3.}

\item{col.format.p}{Applies p-value formatting to columns
that cannot be named "p" (for example for a data frame full
of p-values, also because it is not possible to have more
than one column named "p"). Select with numerical range, e.g., 1:3.}

\item{col.format.r}{Applies r-value formatting to columns
that cannot be named "r" (for example for a data frame full
of r-values, also because it is not possible to have more
than one column named "r"). Select with numerical range, e.g., 1:3.}

\item{col.format.ci}{Applies 95\% confidence interval formatting
to selected columns (e.g., when reporting more than one interval).}

\item{format.custom}{Applies custom formatting to columns
selected via the \code{col.format.custom} argument. This is useful
if one wants custom formatting other than for p- or r-values.
It can also be used to transform (e.g., multiply) certain values
or print a specific symbol along the values for instance.}

\item{col.format.custom}{Which columns to apply the custom
function to. Select with numerical range, e.g., 1:3.}

\item{width}{Width of the table, in percentage of the
total width, when exported e.g., to Word. For full width,
use \code{width = 1}.}

\item{broom}{If providing a tidy table produced with the
\code{broom} package, which model type to use if one wants
automatic formatting (options are "t.test", "lm", "cor.test",
and "wilcox.test").}

\item{report}{If providing an object produced with the
\code{report} package, which model type to use if one wants
automatic formatting (options are "t.test", "lm", and "cor.test").}

\item{short}{Logical. Whether to return an abbreviated
version of the tables made by the \code{report} package.}

\item{title}{Optional, to add a table header, if desired.}

\item{note}{Optional, to add one or more table footnote (APA note),
if desired.}

\item{separate.header}{Logical, whether to separate headers based
on name delimiters (i.e., periods ".").}
}
\value{
An APA-formatted table of class "flextable" (and "nice_table").
}
\description{
Make nice APA tables easily through a wrapper
around the \code{flextable} package with sensical defaults and
automatic formatting features.
}
\details{
The resulting \code{flextable} objects can be opened in
Word with \code{print(table, preview ="docx")}, or saved to
Word with the \code{flextable::save_as_docx()} function.
}
\examples{
\dontshow{if (requireNamespace("flextable", quietly = TRUE) && requireNamespace("methods", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# Make the basic table
my_table <- nice_table(
  mtcars[1:3, ],
  title = c("Table 1", "Motor Trend Car Road Tests"),
  note = c(
    "The data was extracted from the 1974 Motor Trend US magazine.",
    "* p < .05, ** p < .01, *** p < .001"
  )
)
my_table

\donttest{
# Save table to word
mypath <- tempfile(fileext = ".docx")
flextable::save_as_docx(my_table, path = mypath)
}

# Publication-ready tables
mtcars.std <- lapply(mtcars, scale)
model <- lm(mpg ~ cyl + wt * hp, mtcars.std)
stats.table <- as.data.frame(summary(model)$coefficients)
CI <- confint(model)
stats.table <- cbind(
  row.names(stats.table),
  stats.table, CI
)
names(stats.table) <- c(
  "Term", "B", "SE", "t", "p",
  "CI_lower", "CI_upper"
)
nice_table(stats.table, highlight = TRUE)

# Test different column names
test <- head(mtcars)
names(test) <- c(
  "dR", "N", "M", "SD", "b", "np2",
  "ges", "p", "r", "R2", "sr2"
)
test[, 10:11] <- test[, 10:11] / 10
nice_table(test)

# Custom cell formatting (such as p or r)
nice_table(test[8:11], col.format.p = 2:4, highlight = .001)

nice_table(test[8:11], col.format.r = 1:4)

# Apply custom functions to cells
fun <- function(x) {
  x + 11.1
}
nice_table(test[8:11], col.format.custom = 2:4, format.custom = "fun")

fun <- function(x) {
  paste("x", x)
}
nice_table(test[8:11], col.format.custom = 2:4, format.custom = "fun")

# Separate headers based on periods
header.data <- structure(
  list(
    Variable = c(
      "Sepal.Length",
      "Sepal.Width", "Petal.Length"
    ), setosa.M = c(
      5.01, 3.43,
      1.46
    ), setosa.SD = c(0.35, 0.38, 0.17), versicolor.M =
      c(5.94, 2.77, 4.26), versicolor.SD = c(0.52, 0.31, 0.47)
  ),
  row.names = c(NA, -3L), class = "data.frame"
)
nice_table(header.data,
  separate.header = TRUE,
  italics = 2:4
)
\dontshow{\}) # examplesIf}
}
\seealso{
Tutorial: \url{https://rempsyc.remi-theriault.com/articles/table}
}
\keyword{APA}
\keyword{style}
\keyword{table}
