% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table_spec.R
\name{spanning_header}
\alias{spanning_header}
\title{Defines a spanning header}
\usage{
spanning_header(
  x,
  from,
  to,
  label = "",
  label_align = "center",
  level = 1,
  n = NULL
)
}
\arguments{
\item{x}{The table object to add spanning headers to.}

\item{from}{The starting column to span.  Spanning columns are defined as
range of columns 'from' and 'to'. The columns may be identified by position, 
or by quoted or unquoted variable names.
The \code{from} parameter is required.}

\item{to}{The ending column to span.  Spanning columns are defined as
range of columns 'from' and 'to'. The columns may be identified by position,
or by quoted or unquoted variable names.
The \code{to} parameter is required.}

\item{label}{The label to apply to the spanning header.}

\item{label_align}{The alignment to use for the label. Valid values are 
"left", "right", "center", and "centre".  The default for spanning columns
is "center".}

\item{level}{The level to use for the spanning header.  The lowest
spanning level is level 1, the next level above is level 2, and so on.  
By default, the level is set to 1.}

\item{n}{The population count to use for the "N=" label on the spanning 
header. The "N=" label will be formatted according to the \code{n_format}
parameter on the \code{\link{create_table}} function.}
}
\value{
The modified table spec.
}
\description{
Create a header that spans multiple columns.  Spanning headers
are used to group related columns.  Such groupings are a common 
feature of statistical reports.
}
\details{
A spanning header is a label and underline that spans one or more 
columns.  A spanning header is defined minimally by identifying the
column range to be spanned, and a label.  A label alignment and "N="
value may also be supplied.

The spanning column range is defined by the \code{from} and \code{to} 
parameters.  The range identifies a contiguous set of variables on the data.
Variables can be identified by position, a quoted variable name, or an 
unquoted variable name.
}
\examples{
library(reporter)
library(magrittr)

# Create a temporary file
tmp <- file.path(tempdir(), "iris.txt")

# Prepare data
dat <- iris[sample(1:150, 15), c(5, 1, 2, 3, 4)]
dat <- dat[order(dat$Species), ]

# Define table
tbl <- create_table(dat) \%>\% 
  titles("Table 3.2", "IRIS Sample Report") \%>\% 
  spanning_header(2, 3, label = "Sepal") \%>\% 
  spanning_header(4, 5, label = "Petal") \%>\% 
  column_defaults(2:5, format = "\%.1f") \%>\% 
  define(Species, align = "left", dedupe = TRUE, blank_after = TRUE) \%>\% 
  define(Sepal.Length, label = "Length") \%>\% 
  define(Sepal.Width, label = "Width") \%>\% 
  define(Petal.Length, label = "Length") \%>\% 
  define(Petal.Width, label = "Width") \%>\% 
  footnotes("* From Fisher's Iris Dataset")
       
# Define report
rpt <- create_report(tmp, orientation="portrait") \%>\%
  options_fixed(blank_margins = TRUE) \%>\% 
  set_margins(top = 1, bottom =1) \%>\% 
  add_content(tbl, align = "left") 

# Write the report
write_report(rpt)

writeLines(readLines(tmp, encoding = "UTF-8"))

#
#
#
#
#                      Table 3.2
#                  IRIS Sample Report
#
#                       Sepal        Petal
#                   ------------ ------------
#       Species     Length Width Length Width
#       -------------------------------------
#       setosa         5.0   3.0    1.6   0.2
#                      4.6   3.4    1.4   0.3
#                      5.0   3.4    1.6   0.4
#                      5.7   3.8    1.7   0.3
#
#       versicolor     5.7   2.8    4.1   1.3
#                      6.2   2.9    4.3   1.3
#                      7.0   3.2    4.7   1.4
#                      6.6   2.9    4.6   1.3
#
#       virginica      6.2   3.4    5.4   2.3
#                      7.2   3.0    5.8   1.6
#                      6.9   3.1    5.1   2.3
#                      5.6   2.8    4.9   2.0
#                      7.7   2.6    6.9   2.3
#                      6.3   2.8    5.1   1.5
#                      7.7   2.8    6.7   2.0
#
#
#       * From Fisher's Iris Dataset
}
\seealso{
Other table: 
\code{\link{column_defaults}()},
\code{\link{create_table}()},
\code{\link{define}()},
\code{\link{print.table_spec}()},
\code{\link{stub}()}
}
\concept{table}
