% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/request.R
\docType{data}
\name{Request}
\alias{Request}
\alias{as.Request}
\alias{is.Request}
\title{HTTP Request Handling}
\usage{
as.Request(x, ...)

is.Request(x)
}
\arguments{
\item{x}{An object coercible to a \code{Request}.}

\item{...}{Parameters passed on to \code{Request$new()}}
}
\value{
A \code{Request} object (for \code{as.Request()}) or a logical indicating whether
the object is a \code{Request} (for \code{is.Request()})
}
\description{
This class wraps all functionality related to extracting information from a
http request. Much of the functionality is inspired by the Request class in
Express.js, so \href{https://expressjs.com/en/4x/api.html#req}{the documentation}
for this will complement this document. As \code{reqres} is build on top of the
\href{https://github.com/jeffreyhorner/Rook/blob/a5e45f751/README.md}{Rook specifications}
the \code{Request} object is initialized from a Rook-compliant object. This will
often be the request object provided by the \code{httpuv} framework. While it
shouldn't be needed, the original Rook object is always accessible and can be
modified, though any modifications will not propagate to derived values in
the \code{Request} object (e.g. changing the \code{HTTP_HOST} element of the Rook
object will not change the \code{host} field of the \code{Request} object). Because of
this, direct manipulation of the Rook object is generally discouraged.
}
\section{Initialization}{

A new 'Request'-object is initialized using the \code{new()} method on the
generator:

\strong{Usage}
\tabular{l}{
\code{req <- Request$new(rook, trust = FALSE)}
}

\strong{Arguments}
\tabular{lll}{
\code{rook} \tab  \tab The rook request that the new object should wrap\cr
\code{trust} \tab  \tab Is this request trusted blindly. If \code{TRUE} \code{X-Forwarded-*} headers will be returned when querying host, ip, and protocol
}
}

\section{Fields}{

The following fields are accessible in a \code{Request} object:

\describe{
\item{\code{trust}}{A logical indicating whether the request is trusted. \emph{Mutable}}
\item{\code{method}}{A string indicating the request method (in lower case, e.g.
'get', 'put', etc.). \emph{Immutable}}
\item{\code{body}}{An object holding the body of the request. This is an empty
string by default and needs to be populated using the \code{set_body()} method
(this is often done using a body parser that accesses the Rook$input
stream). \emph{Immutable}}
\item{\code{cookies}}{Access a named list of all cookies in the request. These
have been URI decoded. \emph{Immutable}}
\item{\code{headers}}{Access a named list of all headers in the request. In order
to follow R variable naming standards \code{-} have been substituted with \code{_}.
Use the \code{get_header()} method to lookup based on the correct header name.
\emph{Immutable}}
\item{\code{host}}{Return the domain of the server given by the "Host" header if
\code{trust == FALSE}. If \code{trust == true} returns the \code{X-Forwarded-Host} instead.}
\item{\code{ip}}{Returns the remote address of the request if \code{trust == FALSE}.
if \code{trust == TRUE} it will instead return the first value of the
\code{X-Forwarded-For} header. \emph{Immutable}}
\item{\code{ips}}{If \code{trust == TRUE} it will return the full list of ips in the
\code{X-Forwarded-For} header. If \code{trust == FALSE} it will return an empty
vector. \emph{Immutable}}
\item{\code{protocol}}{Returns the protocol (e.g. 'http') used for the request.
If \code{trust == TRUE} it will use the value of the \code{X-Forwarded-Proto} header.
\emph{Immutable}}
\item{\code{root}}{The mount point of the application receiving this request. Can
be empty if the application is mounted on the server root. \emph{Immutable}}
\item{\code{path}}{The part of the url following the root. Defines the local
target of the request (independent of where it is mounted). \emph{Immutable}}
\item{\code{url}}{The full URL of the request. \emph{Immutable}}
\item{\code{query}}{The query string of the request (anything following "?" in
the URL) parsed into a named list. The query has been url decoded and "+"
has been substituted with space. Multiple queries are expected to be
separated by either "&" or "|". \emph{Immutable}}
\item{\code{querystring}}{The unparsed query string of the request, including
"?". If no query string exists it will be \code{""} rather than \code{"?"}}
\item{\code{xhr}}{A logical indicating whether the \code{X-Requested-With} header
equals \code{XMLHttpRequest} thus indicating that the request was performed using
a JavaScript library such as jQuery. \emph{Immutable}}
\item{\code{secure}}{A logical indicating whether the request was performed using
a secure connection, i.e. \code{protocol == 'https'}. \emph{Immutable}}
\item{\code{origin}}{The original object used to create the \code{Request} object. As
\code{reqres} currently only works with rook this will always return the original
rook object. \emph{Immutable}, though the content of the rook object itself might
be manipulated as it is an environment.}
\item{\code{response}}{If a \code{Response} object has been created for this request
it is accessible through this field. \emph{Immutable}}
}
}

\section{Methods}{

The following methods are available in a \code{Request} object:

\describe{
\item{\code{set_body(content)}}{Sets the content of the request body. This method
should mainly be used in concert with a body parser that reads the
\code{rook$input} stream}
\item{\code{set_cookies(cookies)}}{Sets the cookies of the request. The cookies
are automatically parsed and populated, so this method is mainly available
to facilitate cookie signing and encryption}
\item{\code{get_header(name)}}{Get the header of the specified name.}
\item{\code{accepts(types)}}{Given a vector of response content types it returns
the preferred one based on the \code{Accept} header.}
\item{\code{accepts_charsets(charsets)}}{Given a vector of possible character
encodings it returns the preferred one based on the \code{Accept-Charset}
header.}
\item{\code{accepts_encoding(encoding)}}{Given a vector of possible content
encodings (usually compression algorithms) it selects the preferred one
based on the \code{Accept-Encoding} header. If there is no match it will return
\code{"identity"} signaling no compression.}
\item{\code{accepts_language(language)}}{Given a vector of possible content
languages it selects the best one based on the \code{Accept-Language} header.}
\item{\code{is(type)}}{Queries whether the body of the request is in a given
format by looking at the \code{Content-Type} header. Used for selecting the best
parsing method.}
\item{\code{respond()}}{Creates a new \code{Response} object from the request}
\item{\code{parse(..., autofail = TRUE)}}{Based on provided parsers it selects
the appropriate one by looking at the \code{Content-Type} header and assigns the
result to the request body. A parser is a function accepting a raw vector,
and a named list of additional directives,
and returns an R object of any kind (if the parser knows the input to be
plain text, simply wrap it in \code{\link[=rawToChar]{rawToChar()}}). If the body is compressed, it
will be decompressed based on the \code{Content-Encoding} header prior to passing
it on to the parser. See \link{parsers} for a list of pre-supplied parsers.
Parsers are either supplied in a named list or as named arguments to the
parse method. The names should correspond to mime types or known file
extensions. If \code{autofail = TRUE} the response will be set with the correct
error code if parsing fails. \code{parse()} returns \code{TRUE} if parsing was
successful and \code{FALSE} if not}
\item{\code{parse_raw(autofail = TRUE)}}{This is a simpler version of the
\code{parse()} method. It will attempt to decompress the body and set the \code{body}
field to the resulting raw vector. It is then up to the server to decide how
to handle the payload. It returns \code{TRUE} if successful and \code{FALSE}
otherwise.}
\item{\code{as_message()}}{Prints a HTTP representation of the request to the
output stream.}
}
}

\examples{
fake_rook <- fiery::fake_request(
  'http://example.com/test?id=34632&question=who+is+hadley',
  content = 'This is an elaborate ruse',
  headers = list(
    Accept = 'application/json; text/*',
    Content_Type = 'text/plain'
  )
)

req <- Request$new(fake_rook)

# Get full URL
req$url

# Get list of query parameters
req$query

# Test if content is text
req$is('txt')

# Perform content negotiation for the response
req$accepts(c('html', 'json', 'txt'))

# Cleaning up connections
rm(fake_rook, req)
gc()

}
\seealso{
\code{\link{Response}} for handling http responses
}
\keyword{datasets}
