% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inspect.ft.R
\name{inspect.ft}
\alias{inspect.ft}
\title{Explore and visualise flowthrough respirometry data and check for errors}
\usage{
inspect.ft(
  x,
  time = NULL,
  out.oxy = NULL,
  in.oxy = NULL,
  in.oxy.value = NULL,
  delta.oxy = NULL,
  plot = TRUE,
  add.data = NULL,
  ...
)
}
\arguments{
\item{x}{\code{data.frame} containing columns of \code{time} and \code{out.oxy} or
\code{delta.oxy} concentrations, and optionally \code{in.oxy}.}

\item{time}{integer. Defaults to 1. Specifies the column number of the time
data.}

\item{out.oxy}{integer(s). Defaults to \code{NULL}. Specifies the column number(s)
of outflow oxygen data.}

\item{in.oxy}{integer(s). Defaults to \code{NULL}. Specifies the column number(s)
of inflow oxygen data.}

\item{in.oxy.value}{numeric value. Defaults to \code{NULL}. If there is no
continuous \code{in.oxy} data, this specifies a fixed value of oxygen
concentration for inflowing water in same units as \code{out.oxy}, and is used
with \code{out.oxy} to calculate a \code{delta.oxy}.}

\item{delta.oxy}{integer(s). Defaults to all non-time columns if no other
inputs given. Specifies the column number(s) of delta oxygen data, for when
the user has already calculated the difference between outflow and inflow
oxygen (should be negative values for oxygen uptake). If this is used,
\code{out.oxy} and \code{in.oxy} should be NULL.}

\item{plot}{logical. Defaults to TRUE. Plots the data. See Details.}

\item{add.data}{integer. Defaults to \code{NULL}. Specifies the column number of
an optional additional data source that will be plotted in blue alongside
the full oxygen timeseries.}

\item{...}{Allows additional plotting controls to be passed, such as \code{legend = FALSE}, \code{quiet = TRUE}, \code{rate.rev = FALSE} and \code{pos}.}
}
\value{
Output is a \code{list} object of class \code{inspect.ft} containing input
parameters and data, data check summaries, and metadata, which can be
passed to \code{\link[=calc_rate.ft]{calc_rate.ft()}} to determine rates. If there are failed checks
or warnings, the row locations of the potentially problematic data can be
found in \verb{$locs}.
}
\description{
\code{inspect.ft} is a data exploration and preparation function that visualises
flowthrough respirometry data, checks it for common issues, and prepares it
for use in later functions in \code{respR}, such as \code{\link[=calc_rate.ft]{calc_rate.ft()}}.
}
\details{
\code{inspect.ft} is intended to be specific to \emph{flowthrough} respirometry data.
In flowthrough respirometry (also known as 'open flow' or 'continuous flow'
respirometry) rather than calculating a rate from a changing oxygen
concentration recording in a sealed chamber, instead the difference (i.e.
'oxygen delta') between the inflowing and outflowing oxygen concentrations of
a respirometer receiving water at a constant flow rate is used to calculate
an oxygen consumption or production rate, typically after it has reached a
steady state. Therefore, in general, regions of stable oxygen delta values
(difference between outflow and inflow oxygen) are of interest. \code{inspect.ft}
visualises and prepares the data for use in \code{\link[=calc_rate.ft]{calc_rate.ft()}}. By specifying
data types in this function and saving the output, they do not need to be
specified in later functions.
\subsection{Inputs}{

\code{inspect.ft} requires at least two data inputs; a single column of numeric
\code{time} data, with \emph{either} a column of paired \code{out.oxy} concentrations (i.e.
the exhalent or 'downstream' concentrations), \emph{or} a column of already
calculated \code{delta.oxy} values, that is the difference between outflow and
inflow concentrations, or the outflow concentration corrected by a background
recording from a 'blank' or empty chamber.

\strong{out.oxy input option}: If an \code{out.oxy} column has been specified, in order
to calculate the oxygen delta (and therefore a rate in \code{\link[=calc_rate.ft]{calc_rate.ft()}})
there must also be an inflow oxygen concentration input (i.e. the inhalent or
'upstream' concentration). This will generally be a column of paired \code{in.oxy}
concentrations, in which case the paired values of \code{out.oxy} and \code{in.oxy} are
used to calculate the oxygen \code{delta.oxy}, which is saved in the output and
used to determine a rate in \code{\link[=calc_rate.ft]{calc_rate.ft()}}. Alternatively, if the inflow
oxygen concentration is a known, generally unvarying value (such as fully
air-saturated water from a header tank) this can be entered as a single value
via \code{in.oxy.value} and this is used to calculate the \code{delta.oxy}.

\strong{delta.oxy input option}: If delta oxygen values have already been
calculated, these can be entered via the \code{delta.oxy} input, and these are
prepared and saved for rate calculations in \code{calc_rate.ft}.

Given an input data frame \code{x}, the function scans the columns specified via
the \code{time}, \code{out.oxy}, \code{in.oxy} or \code{delta.oxy} inputs. If no columns are
specified, by default the function assumes the first column is \code{time}, and
all others are \code{delta.oxy} oxygen data.  However, best practice is to use the
inputs to specify particular columns.
}

\subsection{Check for numeric data}{

\code{respR} requires data be in the form of paired values of numeric time and
oxygen. All columns are checked that they contain numeric data before any
other checks are performed. If any of the inspected columns do not contain
numeric data the remaining checks for that column are skipped, and the
function exits returning \code{NULL}, printing the summary of the checks. No plot
is produced. Only when all inspected columns pass this numeric check can the
resulting output object be saved and passed to other \code{respR} functions.
}

\subsection{Other checks}{

The \code{time} column is checked for missing (\code{NA/NaN}) values, infinite values
both positive and negative (\code{Inf/-Inf}), that values are sequential, that
there are no duplicate times, and that it is numerically evenly-spaced.
Oxygen columns are checked for missing (\code{NA/NaN}) and infinite values
(\code{Inf/-Inf}). See \strong{Failed Checks} section for what it means for analyses if
these checks result in warnings. If the output is assigned, the specified
columns are saved to a \code{list} object for use in later functions such as
\code{\link[=calc_rate.ft]{calc_rate.ft()}}. A plot is also produced.
}

\subsection{Plot}{

If \code{plot = TRUE}, entered data is plotted against both time (bottom, blue
axis) and row index (top, red axis), depending on the inputs:
\itemize{
\item a single \code{out.oxy} column with either a paired \code{in.oxy} column or
\code{in.oxy.value}: a two panel plot. The top plot is both outflow (green points)
and inflow (turquoise points) oxygen. The bottom plot is the oxygen delta
(black points) between outflow and inflow oxygen, essentially a unitless
oxygen uptake or production rate.
\item a single \code{delta.oxy} column: a one panel plot of oxygen delta values.
\item multiple \code{out.oxy} or \code{delta.oxy} columns: a grid plot of all \code{delta.oxy}
data (either as entered or calculated from \code{out.oxy} and \code{in.oxy}). Specific
delta plots can be examined individually by using the \code{pos} input (e.g.
\code{plot(x, pos = 2)}). Y-axes are not equal.
\item unspecified columns: all columns are plotted assuming \code{time} is in column
1, and all others are oxygen \code{delta.oxy} data. Y-axes are not equal.
}

In delta plots, that is those plotting \code{delta.oxy} values, either directly
entered or calculated, consistent oxygen uptake or production rates will be
represented by flat or level regions. The \code{width} input may help with
selecting regions from which to extract rates, and can be passed in the main
function call or using \code{plot()} on the output object. This smooths delta
oxygen values by calculating a rolling mean across the data. See \strong{Additional
plotting options} below.

\emph{\strong{Note:}} Since \code{respR} is primarily used to examine oxygen consumption,
the delta oxygen and rate plots are by default plotted on a reverse y-axis.
In \code{respR} oxygen uptake rates are negative since they represent a negative
slope of oxygen against time. In these plots the axis is reversed so that
higher uptake rates (i.e. more negative) will be higher on these plots. If
you are interested instead in oxygen production rates, which are positive,
the \code{rate.rev = FALSE} input can be passed in either the \code{inspect.ft} call,
or when using \code{plot()} on the output object. In this case, the delta and rate
values will be plotted numerically, and higher oxygen \emph{production} rates will
be higher on the plot.
}

\subsection{Plot an additional data source}{

Using the \code{add.data} input an additional data source, for example
temperature, can be plotted alongside the oxygen timeseries. This input
should be an integer indicating a column in the input \code{x} data frame sharing
the same time data. None of the data checks are performed on this column; it
is simply to give a basic visual aid in the plot to, for example, help decide
if regions of the data should be used or not used because this parameter was
variable. It is saved in the output as a vector under \verb{$add.data}. It is
plotted in blue on a separate y-axis on the main timeseries plot. It is \emph{not}
plotted if multiple oxygen columns are inspected. See examples.
}

\subsection{Additional plotting options}{

The \code{width} input may help with selecting regions from which to extract
rates. This smooths delta oxygen values by calculating a rolling mean across
the data, and should be a value between 0 and 1 representing a proportion of
the total data width. If left as the default \code{NULL} no smoothing is
performed. This is a visual aid which only affects plotted values and does
not alter output delta oxygen values.

If the legend or labels obscure part of the plot, they can be suppressed via
\code{legend = FALSE} in either the \code{inspect.ft} call, or when using \code{plot()} on
the output object. Suppress console output messages with \code{quiet = TRUE}. If
multiple columns have been inspected, the \code{pos} input can be used to examine
each \code{out.oxy}~\code{in.oxy}~\code{del.oxy} dataset. If axis labels (particularly
y-axis) are difficult to read, \code{las = 2} can be passed to make axis labels
horizontal. In addition, \code{oma} (outer margins, default \code{oma = c(0.4, 1, 1.5, 0.4)}), and \code{mai} (inner margins, default \code{mai = c(0.3, 0.15, 0.35, 0.15)})
can be used to adjust plot margins.
}

\subsection{Multiple data columns}{

For a quick overview of larger experiments, multiple columns of \code{out.oxy},
\code{in.oxy} and \code{delta.oxy} can be inspected, but must share the same numeric
time data column specified by the \code{time} input. Note, multiple column
inspection is chiefly intended to be exploratory functionality to provide a
quick overview of larger datasets. While the output will contain all data
columns in \verb{$dataframe} and \verb{$data}, subsequent functions such as
\code{\link[=calc_rate.ft]{calc_rate.ft()}} will use only the first \code{delta.oxy} column for calculating
rates. Best practice is to inspect and assign each individual experiment or
column pair as separate \code{inspect.ft} objects. See Examples.

If multiple \code{out.oxy} columns are specified, \code{in.oxy} can be a single column
(if for example all chambers are supplied from the same header tank), in
which case it is used to calculate an oxygen delta for all \code{out.oxy} columns.
A single \code{in.oxy.value} in the same units as \code{out.oxy} can also be specified.
There can also be multiple \code{in.oxy} columns, in which case it is assumed each
\code{out.oxy} column is paired with each \code{in.oxy} at the same position, and used
to calculate the oxygen \code{delta.oxy}. In this case, \code{out.oxy} and \code{in.oxy}
must have equal numbers of columns.
}

\subsection{Failed Checks}{

The most important data check in \code{inspect.ft} is that all data columns are
numeric. If any column fails this check, the function skips the remaining
checks for that column, the function exits returning \code{NULL}, and no output
object or plot is produced.

The other failed check that requires action is the check for infinite values
(\code{Inf/-Inf}). Some oxygen sensing systems add these in error when
interference or data dropouts occur. Infinite values will cause problems when
it comes to calculating rates, so need to be removed. If found, locations of
these are printed and can be found in the output object under \verb{$locs}. Note,
these values are not plotted, so special note should be taken of the warnings
and console printout.

The remaining data checks in \code{inspect.ft} are mainly exploratory and help
diagnose and flag potential issues with the data that might affect rate
calculations. For instance, long experiments may have had sensor dropouts the
user is unaware of. Some might not be major issues. For instance, an uneven
time warning can result from using decimalised minutes, which is a completely
valid time metric, but happens to be numerically unevenly spaced. As an
additional check, if uneven time is found, the minimum and maximum intervals
in the time data are in the console output, so a user can see immediately if
there are large gaps in the data.

If some of these checks produce warnings, it should \emph{generally} not hinder
analysis of the data. \code{respR} has been coded to rely on linear regressions on
exact data values, and not make assumptions about data spacing or order.
Therefore issues such as missing or NA/NaN values, duplicate or
non-sequential time values, or uneven time spacing should not cause any
erroneous results, as long as they do not occur over large regions of the
data. \code{inspect.ft} however outputs locations (row numbers) of where these
issues occur (located in the \verb{$locs} element of the output), allowing users
to amend them before analysis. We would recommend that to be completely
confident in any results from analysis of such data, and avoid obscure
errors, these issues be addressed before proceeding.
}

\subsection{Background control or "blank" experiments}{

For experiments in which the specimen data is to be background corrected by a
concurrently-run control experiment, \code{inspect.ft} can be used by specifying
the specimen experiment as \code{out.oxy}, and the "blank" as the \code{in.oxy} input.
In this way, any variations in oxygen in the specimen data due to background
microbial activity, or for any other reason such as fluctuations in inflow
oxygen, are accounted for in the delta oxygen calculations, and therefore in
the rate calculated in \code{\link[=calc_rate.ft]{calc_rate.ft()}}. See the vignettes on the website
for examples.

If the background recordings are experiments with their own outflow and
inflow recordings, which show a generally consistent oxygen delta due to
microbial activity, this can be saved as a separate \code{inspect.ft} object, a
background rate calculated in \code{\link[=calc_rate.ft]{calc_rate.ft()}}, and this used in
\code{\link[=adjust_rate.ft]{adjust_rate.ft()}} as the \code{by} input to perform background adjustments to
specimen rates.

\strong{Note}: All background calculations should be from experiments done at the
\emph{same flow rate} as the specimen experiments to be corrected.
}

\subsection{S3 Generic Functions}{

Saved output objects can be used in the generic S3 functions \code{plot()},
\code{print()} and \code{summary()}.
\itemize{
\item \code{plot()}: plots the result.
\item \code{print()}: prints a summary of the checks performed on the data. If issues
are found, locations (row numbers) are printed (up to first 20 occurrences).
\item \code{summary()}: simple wrapper for \code{print()} function. See above.
}
}
}
\examples{

# Inspect outflow and inflow oxygen data
x <- inspect.ft(flowthrough.rd, time = 1, out.oxy = 2,
                in.oxy = 3)
print(x)
plot(x)

# Inspect outflow oxygen data with inflow oxygen as a known value in
# the same units
x <- inspect.ft(flowthrough.rd, time = 1, out.oxy = 2,
                in.oxy.value = 8.90)

# Inspect already calculated delta oxygen data
inspect.ft(flowthrough.rd, time = 1, delta.oxy = 4)

# inspect multiple columns for a quick overview
inspect.ft(flowthrough_mult.rd, time = 1, delta.oxy = 10:12)

# Inspect outflow and use a blank control chamber as background
# correction
#
# This experiment has increasing background respiration over time.
# Inspecting outflow oxygen with inflow header tank concentrations
# suggests specimen rates (bottom delta.oxy plot) are increasing.
inspect.ft(flowthrough_sim.rd, time = 1,
           out.oxy = 2, in.oxy = 4)

# However, inspecting with recordings from a concurrent blank
# control accounts for this and shows specimen rates are level
# when background is taken into account.
inspect.ft(flowthrough_sim.rd, time = 1,
           out.oxy = 2, in.oxy = 3)

# Inspect and plot an additional data type

}
