% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ee_utils.R
\name{ee_utils_gif_creator}
\alias{ee_utils_gif_creator}
\title{Create a GIF from an Earth Engine ImageCollection}
\usage{
ee_utils_gif_creator(ic, parameters, quiet = FALSE, ...)
}
\arguments{
\item{ic}{An ee$ImageCollection.}

\item{parameters}{List of parameters for visualization and animation. See details.}

\item{quiet}{Logical. Suppress info message.}

\item{...}{parameter(s) passed on to \link[utils:download.file]{download.file}}
}
\value{
A magick-image object of the specified ImageCollection.
}
\description{
Create an GIF (as a magick-image object) from a EE
ImageCollection. Note: Animations can only be created when ImageCollections
is composed by RGB or RGBA image. This can be done by mapping
a visualization function onto an ImageCollection (e.g. \code{ic$map(function(img) img$visualize(...))})
or specifying three bands in parameters argument (See examples). \link{ee_utils_gif_creator} is a
wrapper around \strong{\code{ee$ImageCollection$getVideoThumbURL}}.
}
\details{
The parameters argument is identical to visParams (See \code{rgee::Map$addLayer}),
plus, optionally:
\itemize{
\item \strong{dimensions}: A number or pair of numbers in format c(WIDTH,HEIGHT).
Max dimensions of the thumbnail to render, in pixels. If only one number is
passed, it is used as the maximum, and the other dimension is computed by
proportional scaling.
\item \strong{crs}: A CRS string specifying the projection of the output.
\item \strong{crs_transform}: The affine transform to use for the output
pixel grid.
\item \strong{scale}: A scale to determine the output pixel grid; ignored if
both crs and crs_transform are specified.
\item \strong{region}: ee$Geometry$Polygon, GeoJSON or c(E,S,W,N). Geospatial
region of the result. By default, the whole image.
\item \strong{format}: String. The output format (only 'gif' is currently
supported).
\item \strong{framesPerSecond}: String. Animation speed.
}
}
\examples{
\dontrun{
library(rgee)

ee_Initialize()

col <- ee$ImageCollection("JRC/GSW1_1/YearlyHistory")$map(function(img) {
  year <- img$date()$get("year")
  yearImg <- img$gte(2)$multiply(year)
  despeckle <- yearImg$connectedPixelCount(15, TRUE)$eq(15)
  yearImg$updateMask(despeckle)$selfMask()$set("year", year)
})

appendReverse <- function(col) col$merge(col$sort('year', FALSE))

# -----------------------------------
# 1 Basic Animation - Ucayali Peru
# -----------------------------------

bgColor = "FFFFFF" # Assign white to background pixels.
riverColor = "0D0887" # Assign blue to river pixels.

## 1.1 Create the dataset
annualCol = col$map(function(img) {
  img$unmask(0)$
    visualize(min = 0, max = 1, palette = c(bgColor, riverColor))$
    set("year", img$get("year"))
})
basicAnimation <- appendReverse(annualCol)


## 1.2 Set video arguments
aoi <- ee$Geometry$Rectangle(-74.327, -10.087, -73.931, -9.327)
videoArgs = list(
  dimensions = 600, # Max dimension (pixels), min dimension is proportionally scaled.
  region = aoi,
  framesPerSecond = 10
)

## 1.2 Download, display and save the GIF!
animation <- ee_utils_gif_creator(basicAnimation, videoArgs, mode = "wb")
get_years <- basicAnimation$aggregate_array("year")$getInfo()
animation \%>\%
  ee_utils_gif_annotate("Ucayali, Peru") \%>\%
  ee_utils_gif_annotate(get_years, size = 15, location = "+90+40",
                        boxcolor = "#FFFFFF") \%>\%
  ee_utils_gif_annotate("created using {magick} + {rgee}",
                        size = 15, font = "sans",location = "+70+20") ->
  animation_wtxt
gc(reset = TRUE)
ee_utils_gif_save(animation_wtxt, path = paste0(tempfile(), ".gif"))
}
}
\seealso{
Other GIF functions: 
\code{\link{ee_utils_gif_annotate}()},
\code{\link{ee_utils_gif_save}()}
}
\author{
Jeroen Ooms
}
\concept{GIF functions}
