% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ridge-scores.R
\name{ridge_scores}
\alias{ridge_scores}
\alias{max_score_2}
\title{Scores and scales for Fourier-fitted ridge curves}
\usage{
ridge_scores(
  x,
  mu = c(0, 0),
  coefs = list(cos_a = c(0, 0), sin_b = 0),
  ind_var = 1,
  N = 500,
  scale = TRUE,
  at2 = TRUE
)

max_score_2(
  mu = c(0, 0),
  coefs = list(cos_a = c(0, 0), sin_b = 0),
  ind_var = 1,
  L = 25,
  f = 2,
  at2 = TRUE
)
}
\arguments{
\item{x}{a matrix of size \code{c(nx, 2)} with angular coordinates.}

\item{mu}{a vector of size \code{2} giving \eqn{(\mu_1, \mu_2)}. Defaults
to \code{c(0, 0)}.}

\item{coefs}{list of coefficients \code{cos_a} (\eqn{a_k}) and
\code{sin_b} (\eqn{b_k} giving the Fourier fit of the ridge curve.
Defaults to \code{list(cos_a = c(0, 0), sin_b = 0)}. See examples.}

\item{ind_var}{index \eqn{j} of the variable that parametrizes the ridge.
Defaults to \code{1}.}

\item{N}{number of discretization points for approximating curve lengths.
Defaults to \code{5e2}.}

\item{scale}{scale the resulting scores to \eqn{[-\pi, \pi)^2}? Defaults
to \code{TRUE}.}

\item{at2}{do the \code{atan2} fit instead of the sine fit (only using
\eqn{S_m})? Defaults to \code{TRUE}. \code{at2 = FALSE} is not
recommended to use.}

\item{L}{grid along he variable \code{ind_var} used for searching the
maximum allowed second score. Defaults to \code{25}.}

\item{f}{factor for shrinking the grid on the variable that is different to
\code{ind_var}. Defaults to \code{2}.}
}
\value{
\code{ridge_scores} returns a list with:
\item{scores}{a matrix of size \code{c(nx, 2)} with the ridge scores.}
\item{scales}{a vector of length 2 with the scale limits for the axes.}
\code{max_score_2} computes the maximum allowed second score to rescale if
\code{scale = TRUE}.
}
\description{
Computation of PCA scores for \link[=ridge_curve]{
Fourier-fitted ridge curves}. The scores are defined as follows:
\itemize{
  \item First scores: signed distances along the ridge curve of the data
  projections to \eqn{\mu}.
  \item Second scores: signed toroidal distances from the data points to
  their ridge projections.
}
The scores can be scaled to \eqn{(-\pi, \pi)} or remain as
\eqn{(l / 2, m_2)}, where \eqn{l} is the length of the curve and \eqn{m_2}
is the maximal absolute second score.
}
\details{
The mean \eqn{\mu} corresponds to the first score being null.
}
\examples{
mu <- c(-0.5, 1.65)
th <- seq(-pi, pi, l = 200)
K <- 5
coefs <- list(cos_a = 1 / (1:(K + 1))^3, sin_b = 1 / (1:K)^3)
n <- 10
col <- rainbow(n)

set.seed(13213)
old_par <- par(no.readonly = TRUE)
par(mfrow = c(2, 2))
for (j in 1:2) {

  # Simulate synthetic data close to the ridge curve
  rid <- ridge_curve(theta = th, mu = mu, coefs = coefs, ind_var = j)
  ind <- sort(sample(length(th), size = n))
  eps <- 0.25 * matrix(runif(2 * n, -1, 1), nrow = n, ncol = 2)
  x <- sdetorus::toPiInt(rid[ind, ] + eps)

  # Plot ridge and synthetic data, with signs from the second scores
  s <- ridge_scores(x, mu = mu, coefs = coefs, ind_var = j)$scores
  plot(x, xlim = c(-pi, pi), ylim = c(-pi, pi), axes = FALSE,
       xlab = expression(theta[1]), ylab = expression(theta[2]), col = col,
       pch = ifelse(sign(s[, 2]) == 1, "+", "-"), cex = 1.25)
  sdetorus::linesTorus(rid[, 1], rid[, 2], lwd = 2)
  abline(v = mu[1], lty = 3)
  abline(h = mu[2], lty = 3)
  points(mu[1], mu[2], pch = "*", cex = 3)
  sdetorus::torusAxis()

  # Projections
  theta_projs <- proj_ridge_curve(x = x, mu = mu, coefs = coefs,
                                  ind_var = j, ridge_curve_grid = rid,
                                  )$theta_proj
  projs <- ridge_curve(theta = theta_projs, mu = mu, coefs = coefs,
                       ind_var = j)
  for (i in 1:n) {

    sdetorus::linesTorus(x = c(x[i, 1], projs[i, 1]),
                         y = c(x[i, 2], projs[i, 2]),
                         col = col[i], lty = 3)

  }

  # Scores plot
  plot(s, xlim = c(-pi, pi), ylim = c(-pi, pi), axes = FALSE,
       xlab = "Score 1", ylab = "Score 2", col = col,
       pch = ifelse(sign(s[, 2]) == 1, "+", "-"))
  sdetorus::torusAxis()
  abline(v = 0, lty = 3)
  abline(h = 0, lty = 3)
  points(0, 0, pch = "*", cex = 3)

}
par(old_par)
}
