% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generic-keops_kernel.R
\name{keops_kernel}
\alias{keops_kernel}
\title{Defines a new operators}
\usage{
keops_kernel(formula, args)
}
\arguments{
\item{formula}{text string, an operator formula (see Details).}

\item{args}{vector of text string, formula arguments (see Details).}
}
\value{
a function that can be used to compute the value of the formula
on actual data. This function takes as input a list of data corresponding
to the formula arguments and return the computed values (generally a
vector or a matrix depending on the reduction). It has an additional integer
input parameter \code{inner_dim} indicating if the inner dimension
(c.f. \code{browseVignettes("rkeops")}) corresponds to columns, i.e.
\code{inner_dim=1} (default), or rows, i.e. \code{inner_dim=0}, in the data.
}
\description{
This function is the core of the KeOps library, it allows you to create
new operators based on kernel operation and matrix reduction discribed as a
mathematic formula.
}
\details{
The use of the function \code{keops_kernel} is detailled in the vignettes,
especially how to write formulae, specified input arguments, how to format
data to apply the created operators, etc. Run \code{browseVignettes("rkeops")}
to access the vignettes.

KeOps operators are defined thanks to formula, i.e. a text string describing
the mathematical operations that you want to apply to your data, and a
list defining the input arguments of your formula.

The function \code{keops_kernel} compiles and imports a new operator that
implements the formala given in input, it returns a function that can be
used to compute the result of the formula on actual data.

The returned function expects a list of arguments, as data matrices, whose
order corresponds to the order given in \code{args} to \code{keops_kernel}.
We use a list to avoid useless copies of data.

\strong{Note:} Data are input as a list, because list are references and since
argument passing is done by copy in R, it is better to copy a list of
reference than the actual input data, especially for big matrices.

You should be careful with the input dimension of your data, to correspond
to the input dimension specified in \code{args} (see inner ou outer dimension in
\code{browseVignettes("rkeops")}.

It is possible to compute partial derivatives of user defined operators
with the function \code{\link[=keops_grad]{keops_grad()}}.
}
\examples{
\dontrun{
set_rkeops_options()

## Example 1
# Defining a function that computes for each j the sum over i
# of the scalar products between `x_i` and `y_j` (both 3d vectors), 
# i.e. the sum over the rows of the result of the matrix product `X * t(Y)`
# where `x_i` and `y_j` are the respective rows of the matrices `X` and `Y`.
op <- keops_kernel(formula = "Sum_Reduction((x|y), 1)",
                   args = c("x=Vi(3)", "y=Vj(3)"))
# data
nx <- 10
ny <- 15
# x_i = rows of the matrix X
X <- matrix(runif(nx*3), nrow=nx, ncol=3)
# y_j = rows of the matrix Y
Y <- matrix(runif(ny*3), nrow=ny, ncol=3)
# compute the result (here, by default `inner_dim=1` and columns corresponds 
# to the inner dimension)
res <- op(list(X,Y))

## Example 1 bis
# In example 1, the inner dimension (i.e. the commun dimension of vectors 
# `x_i` and `y_j` corresponds to columns of the matrices `X` and `Y` resp.).
# We know consider the inner dimension to be the rows of the matrices `X` 
# and `Y`.

# data
nx <- 10
ny <- 15
# x_i = columns of the matrix X
X <- matrix(runif(nx*3), nrow=3, ncol=nx)
# y_j = columns of the matrix Y
Y <- matrix(runif(ny*3), nrow=3, ncol=ny)
# compute the result (we specify `inner_dim=0` to indicate that the rows 
# corresponds to the inner dimension)
res <- op(list(X,Y), inner_dim=0)

## Example 2
# Defining a function that computes the convolution with a Gaussian kernel 
# i.e. the sum over i of `e^(lambda * ||x_i - y_j||^2) * beta_j` where `x_i`, 
# `y_j` and `beta_j` are 3d vectors, and `lambda` is a scalar parameter.
op = keops_kernel(formula = "Sum_Reduction(Exp(lambda*SqNorm2(x-y))*beta, 1)",
                 args = c("x=Vi(3)", "y=Vj(3)", 
                          "beta=Vj(3)", "lambda=Pm(1)"))

# data
nx <- 10
ny <- 15
# x_i = rows of the matrix X
X <- matrix(runif(nx*3), nrow=nx, ncol=3)
# y _j = rows of the matrix Y
Y <- matrix(runif(ny*3), nrow=ny, ncol=3)
# beta_j = rows of the matrix beta
beta <- matrix(runif(ny*3), nrow=ny, ncol=3)
# !! important !! y and beta should have the same dimension

# parameter
lambda <- 0.25

# compute the result
res <- op(list(X, Y, beta, lambda))
}
}
\seealso{
\code{\link[=keops_grad]{keops_grad()}}
}
\author{
Ghislain Durif
}
