% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rnndescent.R
\name{prepare_search_graph}
\alias{prepare_search_graph}
\title{Convert a nearest neighbor graph into a search graph}
\usage{
prepare_search_graph(
  data,
  graph,
  metric = "euclidean",
  diversify_prob = 1,
  pruning_degree_multiplier = 1.5,
  n_threads = 0,
  verbose = FALSE,
  obs = "R"
)
}
\arguments{
\item{data}{Matrix of \code{n} items, with observations in the rows and features
in the columns. Optionally, input can be passed with observations in the
columns, by setting \code{obs = "C"}, which should be more efficient. Possible
formats are \code{\link[base:data.frame]{base::data.frame()}}, \code{\link[base:matrix]{base::matrix()}} or
\code{\link[Matrix:sparseMatrix]{Matrix::sparseMatrix()}}. Sparse matrices should be in \code{dgCMatrix} format.
Dataframes will be converted to \code{numerical} matrix format internally, so if
your data columns are \code{logical} and intended to be used with the
specialized binary \code{metric}s, you should convert it to a logical matrix
first (otherwise you will get the slower dense numerical version).}

\item{graph}{neighbor graph for \code{data}, a list containing:
\itemize{
\item \code{idx} an \code{n} by \code{k} matrix containing the nearest neighbor indices of
the data in \code{data}.
\item \code{dist} an \code{n} by \code{k} matrix containing the nearest neighbor distances.
}}

\item{metric}{Type of distance calculation to use. One of:
\itemize{
\item \code{"braycurtis"}
\item \code{"canberra"}
\item \code{"chebyshev"}
\item \code{"correlation"} (1 minus the Pearson correlation)
\item \code{"cosine"}
\item \code{"dice"}
\item \code{"euclidean"}
\item \code{"hamming"}
\item \code{"hellinger"}
\item \code{"jaccard"}
\item \code{"jensenshannon"}
\item \code{"kulsinski"}
\item \code{"sqeuclidean"} (squared Euclidean)
\item \code{"manhattan"}
\item \code{"rogerstanimoto"}
\item \code{"russellrao"}
\item \code{"sokalmichener"}
\item \code{"sokalsneath"}
\item \code{"spearmanr"} (1 minus the Spearman rank correlation)
\item \code{"symmetrickl"} (symmetric Kullback-Leibler divergence)
\item \code{"tsss"} (Triangle Area Similarity-Sector Area Similarity or TS-SS
metric)
\item \code{"yule"}
}

For non-sparse data, the following variants are available with
preprocessing: this trades memory for a potential speed up during the
distance calculation. Some minor numerical differences should be expected
compared to the non-preprocessed versions:
\itemize{
\item \code{"cosine-preprocess"}: \code{cosine} with preprocessing.
\item \code{"correlation-preprocess"}: \code{correlation} with preprocessing.
}

For non-sparse binary data passed as a \code{logical} matrix, the following
metrics have specialized variants which should be substantially faster than
the non-binary variants (in other cases the logical data will be treated as
a dense numeric vector of 0s and 1s):
\itemize{
\item \code{"dice"}
\item \code{"hamming"}
\item \code{"jaccard"}
\item \code{"kulsinski"}
\item \code{"matching"}
\item \code{"rogerstanimoto"}
\item \code{"russellrao"}
\item \code{"sokalmichener"}
\item \code{"sokalsneath"}
\item \code{"yule"}
}}

\item{diversify_prob}{the degree of diversification of the search graph
by removing unnecessary edges through occlusion pruning. This should take a
value between \code{0} (no diversification) and \code{1} (remove as many edges as
possible) and is treated as the probability of a neighbor being removed if
it is found to be an "occlusion". If item \code{p} and \code{q}, two members of the
neighbor list of item \code{i}, are closer to each other than they are to \code{i},
then the nearer neighbor \code{p} is said to "occlude" \code{q}. It is likely that
\code{q} will be in the neighbor list of \code{p} so there is no need to retain it in
the neighbor list of \code{i}. You may also set this to \code{NULL} to skip any
occlusion pruning. Note that occlusion pruning is carried out twice, once
to the forward neighbors, and once to the reverse neighbors. Reducing this
value will result in a more dense graph. This is similar to increasing the
"alpha" parameter used by in the DiskAnn pruning method of Subramanya and
co-workers (2014).}

\item{pruning_degree_multiplier}{How strongly to truncate the final neighbor
list for each item. The neighbor list of each item will be truncated to
retain only the closest \code{d} neighbors, where
\code{d = k * pruning_degree_multiplier}, and \code{k} is the
original number of neighbors per item in \code{graph}. Roughly, values
larger than \code{1} will keep all the nearest neighbors of an item, plus
the given fraction of reverse neighbors (if they exist). For example,
setting this to \code{1.5} will keep all the forward neighbors and then
half as many of the reverse neighbors, although exactly which neighbors are
retained is also dependent on any occlusion pruning that occurs. Set this
to \code{NULL} to skip this step.}

\item{n_threads}{Number of threads to use.}

\item{verbose}{If \code{TRUE}, log information to the console.}

\item{obs}{set to \code{"C"} to indicate that the input \code{data} orientation stores
each observation as a column. The default \code{"R"} means that observations are
stored in each row. Storing the data by row is usually more convenient, but
internally your data will be converted to column storage. Passing it
already column-oriented will save some memory and (a small amount of) CPU
usage.}
}
\value{
a search graph for \code{data} based on \code{graph}, represented as a sparse
matrix, suitable for use with \code{\link[=graph_knn_query]{graph_knn_query()}}.
}
\description{
Create a graph using existing nearest neighbor data to balance search
speed and accuracy using the occlusion pruning and truncation strategies
of Harwood and Drummond (2016). The resulting search graph should be more
efficient for querying new data than the original nearest neighbor graph.
}
\details{
An approximate nearest neighbor graph is not very useful for querying via
\code{\link[=graph_knn_query]{graph_knn_query()}}, especially if the query data is initialized randomly:
some items in the data set may not be in the nearest neighbor list of any
other item and can therefore never be returned as a neighbor, no matter how
close they are to the query. Even those which do appear in at least one
neighbor list may not be reachable by expanding an arbitrary starting list if
the neighbor graph contains disconnected components.

Converting the directed graph represented by the neighbor graph to an
undirected graph by adding an edge from item \code{j} to \code{i} if
an edge exists from \code{i} to \code{j} (i.e. creating the mutual neighbor
graph) solves the problems above, but can result in inefficient searches.
Although the out-degree of each item is restricted to the number of neighbors
the in-degree has no such restrictions: a given item could be very "popular"
and in a large number of neighbors lists. Therefore mutualizing the neighbor
graph can result in some items with a large number of neighbors to search.
These usually have very similar neighborhoods so there is nothing to be
gained from searching all of them.

To balance accuracy and search time, the following procedure is carried out:
\enumerate{
\item The graph is "diversified" by occlusion pruning.
\item The reverse graph is formed by reversing the direction of all edges in
the pruned graph.
\item The reverse graph is diversified by occlusion pruning.
\item The pruned forward and pruned reverse graph are merged.
\item The outdegree of each node in the merged graph is truncated.
\item The truncated merged graph is returned as the prepared search graph.
}

Explicit zero distances in the \code{graph} will be converted to a small positive
number to avoid being dropped in the sparse representation. The one exception
is the "self" distance, i.e. any edge in the \code{graph} which links a node to
itself (the diagonal of the sparse distance matrix). These trivial edges
aren't useful for search purposes and are always dropped.
}
\examples{
# 100 reference iris items
iris_ref <- iris[iris$Species \%in\% c("setosa", "versicolor"), ]

# 50 query items
iris_query <- iris[iris$Species == "versicolor", ]

# First, find the approximate 4-nearest neighbor graph for the references:
ref_ann_graph <- nnd_knn(iris_ref, k = 4)

# Create a graph for querying with
ref_search_graph <- prepare_search_graph(iris_ref, ref_ann_graph)

# Using the search graph rather than the ref_ann_graph directly may give
# more accurate or faster results
iris_query_nn <- graph_knn_query(
  query = iris_query, reference = iris_ref,
  reference_graph = ref_search_graph, k = 4, metric = "euclidean",
  verbose = TRUE
)
}
\references{
Harwood, B., & Drummond, T. (2016).
Fanng: Fast approximate nearest neighbour graphs.
In \emph{Proceedings of the IEEE Conference on Computer Vision and Pattern Recognition}
(pp. 5713-5722).

Jayaram Subramanya, S., Devvrit, F., Simhadri, H. V., Krishnawamy, R., & Kadekodi, R. (2019).
Diskann: Fast accurate billion-point nearest neighbor search on a single node.
\emph{Advances in Neural Information Processing Systems}, \emph{32}.
}
\seealso{
\code{\link[=graph_knn_query]{graph_knn_query()}}
}
