#' @rdname process_folder
#' 
#' @useDynLib rplexos
#' @export
process_solution <- function(file, keep.temp = FALSE) {
  # Check that inputs are valid
  assert_that(is.string(file))
  
  # Check that file exists
  if (!file.exists(file)) {
    warning(file, " does not exist and was ignored.", call. = FALSE)
    return(invisible(""))
  }

  # Database name will match that of the zip file
  db.temp <- gsub(".zip", "-temp.db", file)
  db.name <- gsub(".zip", ".db", file)
  
  # Does the file need to be reprocessed? Does it already exist?
  if (file.exists(db.temp)) {
    stop_ifnot_delete(db.temp)
  }
  if (file.exists(db.name)) {
    stop_ifnot_delete(db.name)
  }
  
  # Read list of files in the zip file
  zip.content <- unzip(file, list = TRUE)
  
  # Check that zip file has valid XML, Log and BIN files
  xml.pos <- grep("^Model.*xml$", zip.content$Name)
  bin.pos <- grep("^t_data_[0-4].BIN$", zip.content$Name)
  log.pos <- grep("^Model.*Log.txt$", zip.content$Name)
  if ((length(xml.pos) == 0) | (length(bin.pos) == 0) | (length(log.pos) == 0)) {
    warning(file, " is not a PLEXOS solution file and was ignored.", call. = FALSE)
    return(invisible(""))
  }
  
  # Read content from the XML file
  xml.content <- read_file_in_zip(file, xml.pos)
  
  # Check that XML is a valid PLEXOS file
  plexos.check <- grep("SolutionDataset", xml.content)
  if (length(plexos.check) == 0) {
    warning(file, " is not a PLEXOS database and was ignored.", call. = FALSE)
    return(invisible(""))
  }
  
  # Create an empty database and add the XML information
  message("  - Solution: '", file, "'")
  
  # Open connection to SQLite for R
  dbt <- src_sqlite(db.temp, create = TRUE)
  
  # Add basic XML structure and delete cached XML file
  new_database(dbt, xml.content)
  rm(xml.content)
  
  # Add a few tables that will be useful later on
  add_extra_tables(dbt)
  
  # Create SQLite database to store final results
  dbf <- src_sqlite(db.name, create = TRUE)
  
  # Store time stamps
  sql <- "CREATE TABLE data_time (interval INT PRIMARY KEY, time TEXT)"
  dbGetQuery(dbf$con, sql)
  sql <- "CREATE VIEW time AS
          SELECT interval, datetime(time) time
          FROM data_time"
  dbGetQuery(dbf$con, sql)
  
  # Turn PRAGMA OFF
  dbGetQuery(dbf$con, "PRAGMA synchronous = OFF")
  dbGetQuery(dbf$con, "PRAGMA journal_mode = MEMORY")
  dbGetQuery(dbf$con, "PRAGMA temp_store = MEMORY")
  
  # Attach final database to temporary database
  dbGetQuery(dbt$con, sprintf("ATTACH '%s' AS new", db.name))
  
  # Add config table
  dbGetQuery(dbt$con, "CREATE TABLE new.config AS SELECT * FROM t_config")
  sql <- sprintf("INSERT INTO new.config VALUES ('rplexos', '%s')", packageVersion("rplexos"))
  dbGetQuery(dbt$con, sql)
  
  # Add time data
  sql <- "INSERT INTO new.data_time
          SELECT DISTINCT interval_id, time
          FROM temp_period_0"
  dbGetQuery(dbt$con, sql)
  
  # Collate information to key (first period data, then summary data)
  sql <- "CREATE TABLE new.key (key INT PRIMARY KEY,
                                table_name TEXT,
                                collection TEXT,
                                property TEXT,
                                unit TEXT,
                                name TEXT,
                                parent TEXT,
                                category TEXT,
                                region TEXT,
                                zone TEXT,
                                class TEXT,
                                class_group TEXT,
                                phase_id INT,
                                period_type_id INT,
                                timeslice TEXT,
                                band INT,
                                sample TEXT)"
  dbGetQuery(dbt$con, sql)
  
  sql <- "INSERT INTO new.key
          SELECT *
          FROM temp_key"
  dbGetQuery(dbt$con, sql)
  
  # Detach database
  dbGetQuery(dbt$con, "DETACH new");
  
  # Define columns from the key table that go into the views
  view.k <- paste0("k.", c("key", "collection", "property", "unit", "name", "parent", "category",
                            "region", "zone", "phase_id", "period_type_id", "timeslice",
                            "band", "sample"))
  view.k2 <- paste(view.k, collapse = ", ")
  
  # For each summary time, create a table and a view
  times <- c("day", "week", "month", "year")
  for (i in times) {
    sql <- sprintf("CREATE TABLE data_%s (key integer, time real, value double)", i);
    dbGetQuery(dbf$con, sql)
    
    sql <- sprintf("CREATE VIEW %s AS
                    SELECT %s, datetime(d.time) AS time, d.value 
                    FROM data_%s d NATURAL LEFT JOIN key k ", i, view.k2, i);
    dbGetQuery(dbf$con, sql)
  }
  
  # Create interval data tables and views
  sql <- "SELECT DISTINCT table_name
          FROM key
          WHERE period_type_id = 0"
  props <- dbGetQuery(dbf$con, sql)

  for (p in props$table_name) {
    sql <- sprintf("CREATE TABLE '%s' (key INT, time_from INT, time_to INT, value DOUBLE)", p)
    dbGetQuery(dbf$con, sql)
    
    view.name <- gsub("data_interval_", "", p)
    sql <- sprintf("CREATE VIEW %s AS
                    SELECT %s, t1.time time_from, t2.time time_to, d.value 
                    FROM %s d
                    NATURAL JOIN key k 
                    JOIN time t1
                      ON t1.interval = d.time_from
                    JOIN time t2
                      ON t2.interval = d.time_to
                    WHERE k.table_name = '%s'", view.name, view.k2, p, p);
    dbGetQuery(dbf$con, sql)
  }
  
  # Create table for list of properties
  sql <- "CREATE TABLE property AS
          SELECT DISTINCT class_group,
                          class,
                          collection,
                          property,
                          unit,
                          phase_id,
                          period_type_id AS is_summary,
                          table_name,
                          COUNT(DISTINCT band) AS count_band,
                          COUNT(DISTINCT sample) AS count_sample,
                          COUNT(DISTINCT timeslice) AS count_timeslice
          FROM key
          GROUP BY class_group, class, collection, property, unit, phase_id, period_type_id, table_name
          ORDER BY phase_id, period_type_id, class_group, class, collection, property"
  dbGetQuery(dbf$con, sql)
  
  # Add binary data
  for (period in 0:4) {
    # Check if binary file exists, otherwise, skip this period
    bin.name <- sprintf("t_data_%s.BIN", period)
    if(!bin.name %in% zip.content$Name)
      next
    bin.con <- unz(file, bin.name, open = "rb")
    
    # Check if length in t_key_index is correct
    correct.length <- correct_length(dbt, period)
    
    if (period > 0) {
      # Read t_key_index entries for current period
      sql <- sprintf("SELECT tki.key_id, nk.phase_id, tki.period_offset, tki.length
                      FROM t_key_index tki
                      JOIN temp_key nk
                      ON tki.key_id = nk.[key]
                      WHERE tki.period_type_id = %s
                      ORDER BY tki.position", period)
      t.key <- dbGetQuery(dbt$con, sql) %>%
        mutate(phase_id = as.integer(phase_id))
      
      # Fix length if necessary
      if (!correct.length) {
        t.key <- t.key %>% mutate(length = length - period_offset)
      }
      
      # Query time
      t.time <- tbl(dbt, sprintf("temp_period_%s", period)) %>%
        arrange(phase_id, period_id) %>%
        select(phase_id, period_id, time) %>%
        collect()
      
      # Read all the binary data for summary periods
      tdata <- expand_tkey(t.key)
      value.data <- readBin(bin.con, double(),
                            n = nrow(tdata),
                            size = 8,
                            endian = "little")
      
      # Check the size of data (this happens when there is a problem)
      if (length(value.data) < nrow(tdata)) {
        stop("Problem reading binary data for ", times[period], " results (reached end of file).\n",
             "  ", nrow(tdata), " values requested, ", length(value.data), " returned.\n",
             "  This is likely a bug in rplexos. Please report it.", call. = FALSE)
      }
        
      # Copy data
      tdata$value <- value.data
      tdata2 <- tdata %>%
                inner_join(t.time, by = c("phase_id", "period_id")) %>%
                select(key, time, value)
      
      # Execute query in one big binding statement
      sql <- sprintf("INSERT INTO data_%s VALUES(?, ?, ?)", times[period])
      if (packageVersion("RSQLite") >= 1) {
        dbBegin(dbf$con)
      } else {
        dbBeginTransaction(dbf$con)
      }
      dbGetPreparedQuery(dbf$con, sql, bind.data = tdata2)
      dbCommit(dbf$con)
    } else {
      # Read time data
      time0 <- tbl(dbt, "temp_period_0") %>%
        select(phase_id, period_id, interval_id) %>%
        collect
      
      # Read t_key_index entries for period data
      sql <- "SELECT nk.key, nk.phase_id, nk.table_name, tki.period_offset, tki.length
              FROM t_key_index tki
              JOIN temp_key nk
              ON tki.key_id = nk.[key]
              WHERE tki.period_type_id = 0
              ORDER BY tki.position"
      tki <- dbSendQuery(dbt$con, sql)
      
      # Data insert in one transaction
      if (packageVersion("RSQLite") >= 1) {
        dbBegin(dbf$con)
      } else {
        dbBeginTransaction(dbf$con)
      }
      
      # Read one row from the query
      trow <- fetch(tki, 1)
      
      while (nrow(trow) > 0) {
        # Fix length if necessary
        if (!correct.length)
          trow$length = trow$length - trow$period_offset
        
        # Query data
        tdata <- data.frame(key = as.integer(trow$key),
                            phase_id = as.integer(trow$phase_id),
                            period_id    = 1:trow$length + trow$period_offset)
        value.data <- readBin(bin.con, double(),
                              n = nrow(tdata),
                              size = 8,
                              endian = "little")
        
        # Check the size of data (this happens when there is a problem)
        if (length(value.data) < nrow(tdata)) {
          stop("Problem reading binary data for interval results (reached end of file)\n",
               "  ", nrow(tdata), " values requested, ", length(value.data), " returned.\n",
               "  This is likely a bug in rplexos. Please report it.", call. = FALSE)
        }
        
        # Copy data
        tdata$value <- value.data
        
        # Eliminate repeats
        tdata2 <- tdata %>%
          inner_join(time0, by = c("phase_id", "period_id")) %>%
          arrange(interval_id)
        default.interval.to.id <- max(tdata2$interval_id)
        tdata3 <- tdata2 %>%
          filter(value != lag(value, default = Inf)) %>%
          mutate(interval_to_id = lead(interval_id - 1, default = default.interval.to.id)) %>%
          select(key, time_from = interval_id, time_to = interval_to_id, value)
        
        # Add data to SQLite
        dbGetPreparedQuery(dbf$con,
                           sprintf("INSERT INTO %s (key, time_from, time_to, value)
                                    VALUES(?, ?, ?, ?)", trow$table_name),
                           bind.data = data.frame(tdata3))
       
        # Next row in the query
        trow <- fetch(tki, 1)
      }

      # Finish transaction
      dbClearResult(tki)
      dbCommit(dbf$con)
    }
    
    # Close binary file connection
    close(bin.con)
  }
  
  # Read Log file into memory
  log.content <- read_file_in_zip(file, log.pos)
  log.result <- plexos_log_parser(log.content)
  for(i in names(log.result)) {
    dbWriteTable(dbf$con, i, log.result[[i]] %>% as.data.frame, row.names = FALSE)
  }
  
  # Close database connections
  dbDisconnect(dbt$con)
  dbDisconnect(dbf$con)
  
  # Delete temporary database
  if (!keep.temp) {
    stop_ifnot_delete(db.temp)
  }
  
  # Return the name of the database that was created
  invisible(db.name)
}

# Read a file in a zip file onto memory
read_file_in_zip <- function(zip.file, position) {
  zip.content <- unzip(zip.file, list = TRUE)
  read.file <- zip.content[position, ]
  read.con <- unz(zip.file, read.file$Name, open = "r")
  read.content <- readChar(read.con, read.file$Length)
  close(read.con)
  read.content
}

# Populate new database with XML contents
new_database <- function(db, xml) {
  # Read XML and convert to a list
  xml.list <- process_xml(xml)
  
  # Turn PRAGMA OFF
  dbGetQuery(db$con, "PRAGMA synchronous = OFF");
  dbGetQuery(db$con, "PRAGMA journal_mode = MEMORY");
  dbGetQuery(db$con, "PRAGMA temp_store = MEMORY");
  
  # Create data tables
  for (i in 0:4) {
    sql <- sprintf("CREATE TABLE t_data_%s (key_id integer, period_id integer, value double)", i)
    dbGetQuery(db$con, sql)
  }
  
  # Create phase tables
  for (i in 0:4) {
    sql <- sprintf("CREATE TABLE t_phase_%s (interval_id integer, period_id integer)", i)
    dbGetQuery(db$con, sql)
  }
  
  # Create t_key_index table
  dbGetQuery(db$con, "CREATE TABLE t_key_index (key_id integer, period_type_id integer, position long, length integer, period_offset integer)");
  
  # Write tables from XML file
  for (t in names(xml.list))
    dbWriteTable(db$con, t, xml.list[[t]], append = TRUE, row.names = FALSE)
  
  0
}

# Add a few tables that will be useful later on
add_extra_tables <- function(db) {
  # View with class and class_group
  sql <- "CREATE VIEW temp_class AS
          SELECT tc.class_id class_id,
                 tc.name class,
                 tcg.name class_group
          FROM t_class tc
          LEFT JOIN t_class_group tcg
            ON tc.class_group_id = tcg.class_group_id"
  dbGetQuery(db$con, sql)
  
  # View with object, category, class, class_group
  sql <- "CREATE VIEW temp_object AS
          SELECT o.object_id object_id,
                 o.name name,
                 cat.name category,
                 c.class_group class_group,
                 c.class class
          FROM t_object o
          JOIN temp_class c
            ON o.class_id = c.class_id
          JOIN t_category cat
            ON o.category_id = cat.category_id"
  dbGetQuery(db$con, sql)
  
  # Create a new table making long version of property
  sql <- "CREATE TABLE temp_property AS
          SELECT p.property_id property_id,
                 '0' period_type_id,
                 p.name property,
                 c.name collection,
                 u.value unit
          FROM t_property p
          JOIN t_collection c
            ON c.collection_id = p.collection_id
         JOIN t_unit u
           ON u.unit_id = p.unit_id"
  dbGetQuery(db$con, sql)
  sql <- "INSERT INTO temp_property
          SELECT p.property_id property_id,
                 '1' period_type_id,
                 p.summary_name property,
                 c.name collection,
                 u.value unit
          FROM t_property p
          JOIN t_collection c
            ON c.collection_id = p.collection_id
         JOIN t_unit u
           ON u.unit_id = p.summary_unit_id"
  dbGetQuery(db$con, sql)
  
  # View with memberships, collection, parent and child objects
  sql <- "CREATE VIEW temp_membership AS
          SELECT m.membership_id membership_id,
                 m.parent_object_id parent_object_id,
                 m.child_object_id child_object_id,
                 c.name collection,
                 p.name parent_name,
                 p.class parent_class,
                 p.class_group parent_group,
                 p.category parent_category,
                 ch.name child_name,
                 ch.class child_class,
                 ch.class_group child_group,
                 ch.category child_category
          FROM t_membership m
          JOIN t_collection c
            ON c.collection_id = m.collection_id
          JOIN temp_object p
            ON p.object_id = m.parent_object_id
          JOIN temp_object ch
            ON ch.object_id = m.child_object_id"
  dbGetQuery(db$con, sql)
  
  # View to list zones
  sql <- "CREATE VIEW temp_zones AS
          SELECT child_object_id,
                 parent_name region,
                 parent_category zone
          FROM temp_membership
          WHERE collection = 'Generators' 
                AND
                parent_class = 'Region'"
  dbGetQuery(db$con, sql)
  
  # Read key data and transform it
  sql <- "SELECT k.key_id key,
                 m.child_name child_name,
                 m.parent_name parent_name,
                 m.child_class child_class,
                 m.parent_class parent_class,
                 m.child_group child_group,
                 m.parent_group parent_group,
                 m.child_category child_category,
                 m.parent_category parent_category,
                 m.collection child_collection,
                 p.property property,
                 p.unit unit,
                 ts.name timeslice,
                 k.band_id band,
                 k.sample_id sample,
                 k.period_type_id period_type_id,
                 k.phase_id phase_id,
                 z.region region,
                 z.zone zone
          FROM t_key k
          JOIN temp_membership m
            ON m.membership_id = k.membership_id
          JOIN t_timeslice ts
            ON ts.timeslice_id = k.timeslice_id
          JOIN temp_property p
            ON p.property_id = k.property_id
               AND
               p.period_type_id = k.period_type_id
          LEFT OUTER JOIN temp_zones z
               on z.child_object_id = m.child_object_id"
  key <- dbGetQuery(db$con, sql) %>%
    mutate(zone = ifelse(is.na(zone), "", zone),
           region = ifelse(is.na(region), "", region))
  
  # Add collection (to match PLEXOS GUI) and table name
  key2 <- key %>%
    mutate(collection = ifelse(parent_class == "System",
                               child_class,
                               paste0(parent_class, ".", child_collection)),
           table_name = paste0("data_interval_", clean_string(collection), "_", clean_string(property)))
  
  # Change sample to identify stats
  key3 <- key2 %>%
    mutate(sample = gsub("^0$",  "Mean",  sample)) %>%
    mutate(sample = gsub("^-1$", "StDev", sample)) %>%
    mutate(sample = gsub("^-2$", "Min",   sample)) %>%
    mutate(sample = gsub("^-3$", "Max",   sample))
  
  # Write results into a new table
  key3 %>%
    select(key, table_name, collection, property, unit, name = child_name,
           parent = parent_name, category = child_category, region, zone,
           class = child_class, class_group = child_group, phase_id, period_type_id,
           timeslice, band, sample) %>%
    dbWriteTable(db$con, "temp_key", ., row.names = FALSE)
  
  # Create tables to hold interval, day, week, month, and yearly timestamps
  for (i in 0:4) {
    sql <- sprintf("CREATE TABLE temp_period_%s (phase_id INT, period_id INT, interval_id INT, time real)", i)
    dbGetQuery(db$con, sql)
  }
  
  # For each phase add corresponding values to the time tables
  column.times <- c("day_id", "week_id", "month_id", "fiscal_year_id")
  for (phase in 1:4) {
    # Join t_period_0 and t_phase
    sql <- sprintf("CREATE VIEW temp_phase_%s AS
                    SELECT p.*, ph.period_id, julianday(p.year || '-' || substr(0 || p.month_of_year, -2)
                    || '-' || substr(0 || p.day_of_month, -2) || 'T' || substr(p.datetime, -8)) AS correct_time
                    FROM t_period_0 p
                    INNER JOIN t_phase_%s ph
                    ON p.interval_id = ph.interval_id", phase, phase)
    dbGetQuery(db$con, sql)
    
    # Fix time stamps in t_period_0 (interval)
    sql <- sprintf("INSERT INTO temp_period_0
                    SELECT %s, period_id, interval_id, correct_time time
                    FROM temp_phase_%s", phase, phase)
    dbGetQuery(db$con, sql)
    
    # Fix time stamps in t_period_X (summary data)
    for (i in 1:length(column.times)) {
      sql <- sprintf("INSERT INTO temp_period_%s
                      SELECT %s, %s, min(interval_id), min(correct_time) time
                      FROM temp_phase_%s
                      GROUP BY %s", i, phase, column.times[i], phase, column.times[i])
      dbGetQuery(db$con, sql)
    }
  }
  
  0
}

# Does t_key_index have the correct length?
correct_length <- function(db, p) {
  res <- tbl(db, "t_key_index") %>%
    filter(period_type_id == p) %>%
    summarize(JustLength           = max(position / 8 + length),
              SumLength            = sum(length),
              SumLengthMinusOffset = sum(length - period_offset)) %>%
    collect
  
  if (res$JustLength == res$SumLength) {
    return(TRUE)
  } else if (res$JustLength == res$SumLengthMinusOffset) {
    return(FALSE)
  }
  
  warning("Problem with lenght of 't_key_index' for period ", p, "\n",
          "in file '", db$path, "'",
          call. = FALSE, immediate. = TRUE)
  
  TRUE
}
