% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map_plot.saeTrafo.R
\name{map_plot}
\alias{map_plot}
\title{Visualizes regional disaggregated estimates on a map}
\usage{
map_plot(
  object,
  MSE = FALSE,
  CV = FALSE,
  map_obj = NULL,
  map_dom_id = NULL,
  map_tab = NULL,
  color = c("white", "red4"),
  scale_points = NULL,
  guide = "colourbar",
  return_data = FALSE
)
}
\arguments{
\item{object}{an object of type \code{saeTrafo}, containing the estimates to
be visualized.}

\item{MSE}{optional logical. If \code{TRUE}, the MSE is also visualized.
Defaults to \code{FALSE}.}

\item{CV}{optional logical. If \code{TRUE}, the CV is also visualized.
Defaults to \code{FALSE}.}

\item{map_obj}{an \code{SpatialPolygonsDataFrame} object as defined by the
\pkg{sp} package on which the data should be visualized.}

\item{map_dom_id}{a character string containing the name of a variable in
\code{map_obj} that indicates the domains.}

\item{map_tab}{a \code{data.frame} object with two columns that match the
domain variable from the census data set (first column) with the domain
variable in the map_obj (second column). This should only be used if the IDs
in both objects differ.}

\item{color}{a \code{vector} of length 2 defining the lowest and highest
color in the plots.}

\item{scale_points}{a numeric vector of length two. This
scale will be used for every plot.}

\item{guide}{character passed to \code{scale_fill_gradient} from
\pkg{ggplot2}. Possible values are "none", "colourbar", and "legend".
Defaults to "colourbar".}

\item{return_data}{if set to \code{TRUE}, a fortified data frame including
the map data as well as the chosen indicators is returned. Customized maps
can easily be obtained from this data frame via the package \pkg{ggplot2}.
Defaults to \code{FALSE}.}
}
\value{
Creates the plots demanded and, if selected, a fortified data.frame
containing the mapdata and chosen indicators.
}
\description{
Function \code{map_plot} creates spatial visualizations of the estimates
obtained by small area estimation methods.
}
\examples{

\donttest{
# Examples for creating maps to visualize the saeTrafo estimates

# Load Data
data("eusilcA_smp")
data("pop_area_size")
data("pop_mean")
data("pop_cov")

# Nested error regression model
NER_model <- NER_Trafo(fixed = eqIncome ~ gender + eqsize + cash +
                       self_empl + unempl_ben + age_ben + surv_ben +
                       sick_ben + dis_ben + rent + fam_allow + house_allow +
                       cap_inv + tax_adj,
                       smp_domains = "district",
                       pop_area_size = pop_area_size,
                       pop_mean = pop_mean, pop_cov = pop_cov,
                       smp_data = eusilcA_smp, MSE = TRUE)

# Load shape file
load_shapeaustria()

# Example 1: Map plots with uncertainty plots (for MSE and CV)
map_plot(NER_model, MSE = TRUE, CV = TRUE, map_obj = shape_austria_dis,
         map_dom_id = "PB")

# Example 2: Personalize map plot for point estimates
map_plot(NER_model, map_obj = shape_austria_dis, map_dom_id = "PB",
         color = c("white", "darkblue"),
         scale_points = c(0, max(NER_model$ind$Mean)))

# Example 3: More options to personalize map plot by using return_data = TRUE
# and ggplot2
require(ggplot2)
library(ggplot2)
data_plot <- map_plot(NER_model, map_obj = shape_austria_dis, map_dom_id = "PB",
                      return_data = TRUE)
ggplot(data_plot, aes(long = NULL, lat = NULL,
                      group = "PB", fill = Mean))+
       geom_sf(color = "black") +
       theme_void() +
       ggtitle("Personalized map") +
       scale_fill_gradient2(low = "red", mid = "white", high = "darkgreen",
                            midpoint = 20000)

# Example 4: Create a suitable mapping table to use numerical identifiers of
# the shape file

# First find the right order
dom_ord <- match(shape_austria_dis$PB, NER_model$ind$Domain)

#Create the mapping table based on the order obtained above
map_tab <- data.frame(pop_data_id = NER_model$ind$Domain[dom_ord],
                      shape_id = shape_austria_dis$BKZ)

# Create map plot for mean indicator - point and CV estimates but no MSE
# using the numerical domain identifiers of the shape file
map_plot(object = NER_model, MSE = FALSE, CV = TRUE,
         map_obj = shape_austria_dis,
         map_dom_id = "BKZ", map_tab = map_tab)
}

}
\seealso{
\code{\link[sf]{sf}}, \code{\link{NER_Trafo}},
\code{\link{saeTrafoObject}}
}
