% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/priors.R
\name{sdmTMBpriors}
\alias{sdmTMBpriors}
\alias{normal}
\alias{halfnormal}
\alias{mvnormal}
\alias{pc_matern}
\title{Prior distributions}
\usage{
sdmTMBpriors(
  matern_s = pc_matern(range_gt = NA, sigma_lt = NA),
  matern_st = pc_matern(range_gt = NA, sigma_lt = NA),
  phi = halfnormal(NA, NA),
  ar1_rho = normal(NA, NA),
  tweedie_p = normal(NA, NA),
  b = normal(NA, NA),
  sigma_G = halfnormal(NA, NA)
)

normal(location = 0, scale = 1)

halfnormal(location = 0, scale = 1)

mvnormal(location = 0, scale = diag(length(location)))

pc_matern(range_gt, sigma_lt, range_prob = 0.05, sigma_prob = 0.05)
}
\arguments{
\item{matern_s}{A PC (Penalized Complexity) prior (\code{pc_matern()}) on the
spatial random field Matérn parameters.}

\item{matern_st}{Same as \code{matern_s} but for the spatiotemporal random field.
Note that you will likely want to set \code{share_fields = FALSE} if you choose
to set both a spatial and spatiotemporal Matérn PC prior since they both
include a prior on the spatial range parameter.}

\item{phi}{A \code{halfnormal()} prior for the dispersion parameter in the
observation distribution.}

\item{ar1_rho}{A \code{normal()} prior for the AR1 random field parameter. Note
the parameter has support \verb{-1 < ar1_rho < 1}.}

\item{tweedie_p}{A \code{normal()} prior for the Tweedie power parameter. Note the
parameter has support \verb{1 < tweedie_p < 2} so choose a mean appropriately.}

\item{b}{\code{normal()} priors for the main population-level 'beta' effects.}

\item{sigma_G}{\code{halfnormal()} priors for the random intercept SDs.}

\item{location}{Location parameter(s).}

\item{scale}{Scale parameter. For \code{normal()}/\code{halfnormal()}: standard
deviation(s). For \code{mvnormal()}: variance-covariance matrix.}

\item{range_gt}{A value one expects the spatial or spatiotemporal range is
\strong{g}reater \strong{t}han with \code{1 - range_prob} probability.}

\item{sigma_lt}{A value one expects the spatial or spatiotemporal marginal
standard deviation (\code{sigma_O} or \code{sigma_E} internally) is \strong{l}ess \strong{t}han
with \code{1 - sigma_prob} probability.}

\item{range_prob}{Probability. See description for \code{range_gt}.}

\item{sigma_prob}{Probability. See description for \code{sigma_lt}.}
}
\value{
A named list with values for the specified priors.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Optional priors/penalties on model parameters. This results in penalized
likelihood within TMB or can be used as priors if the model is passed to
\pkg{tmbstan} (see the Bayesian vignette).

\strong{Note that Jacobian adjustments are only made if \code{bayesian = TRUE}} when the
\code{\link[=sdmTMB]{sdmTMB()}} model is fit. I.e., the final model will be fit with \pkg{tmbstan}
and priors are specified then \code{bayesian} should be set to \code{TRUE}. Otherwise,
leave \code{bayesian = FALSE}.

\code{pc_matern()} is the Penalized Complexity prior for the Matern
covariance function.
}
\details{
Meant to be passed to the \code{priors} argument in \code{\link[=sdmTMB]{sdmTMB()}}.

\code{normal()} and \code{halfnormal()} define normal and half-normal priors that, at
this point, must have a location (mean) parameter of 0. \code{halfnormal()} is the
same as \code{normal()} but can be used to make the syntax clearer. It is intended
to be used for parameters that have support \verb{> 0}.

See \url{https://arxiv.org/abs/1503.00256} for a description of the
PC prior for Gaussian random fields. Quoting the discussion (and substituting
the argument names in \code{pc_matern()}):
"In the simulation study we observe good coverage of the equal-tailed 95\%
credible intervals when the prior satisfies \code{P(sigma > sigma_lt) = 0.05} and
\code{P(range < range_gt) = 0.05}, where \code{sigma_lt} is between 2.5 to 40 times
the true marginal standard deviation and \code{range_gt} is between 1/10 and 1/2.5
of the true range."

Keep in mind that the range is dependent on the units and scale of the
coordinate system. In practice, you may choose to try fitting the model
without a PC prior and then constraining the model from there. A better
option would be to simulate from a model with a given range and sigma to
choose reasonable values for the system or base the prior on knowledge from a
model fit to a similar system but with more spatial information in the data.
}
\examples{
normal(0, 1)
halfnormal(0, 1)
mvnormal(c(0, 0))
pc_matern(range_gt = 5, sigma_lt = 1)
plot_pc_matern(range_gt = 5, sigma_lt = 1)

\donttest{
d <- subset(pcod, year > 2011)
pcod_spde <- make_mesh(d, c("X", "Y"), cutoff = 30)

# - no priors on population-level effects (`b`)
# - halfnormal(0, 10) prior on dispersion parameter `phi`
# - Matern PC priors on spatial `matern_s` and spatiotemporal
#   `matern_st` random field parameters
m <- sdmTMB(density ~ s(depth, k = 3),
  data = d, mesh = pcod_spde, family = tweedie(),
  share_range = FALSE, time = "year",
  priors = sdmTMBpriors(
    phi = halfnormal(0, 10),
    matern_s = pc_matern(range_gt = 5, sigma_lt = 1),
    matern_st = pc_matern(range_gt = 5, sigma_lt = 1)
  )
)

# - no prior on intercept
# - normal(0, 1) prior on depth coefficient
# - no prior on the dispersion parameter `phi`
# - Matern PC prior
m <- sdmTMB(density ~ depth_scaled,
  data = d, mesh = pcod_spde, family = tweedie(),
  spatiotemporal = "off",
  priors = sdmTMBpriors(
    b = normal(c(NA, 0), c(NA, 1)),
    matern_s = pc_matern(range_gt = 5, sigma_lt = 1)
  )
)

# You get a prior, you get a prior, you get a prior!
# (except on the annual means; see the `NA`s)
m <- sdmTMB(density ~ 0 + depth_scaled + depth_scaled2 + as.factor(year),
  data = d, time = "year", mesh = pcod_spde, family = tweedie(link = "log"),
  share_range = FALSE, spatiotemporal = "AR1",
  priors = sdmTMBpriors(
    b = normal(c(0, 0, NA, NA, NA), c(2, 2, NA, NA, NA)),
    phi = halfnormal(0, 10),
    tweedie_p = normal(1.5, 2),
    ar1_rho = normal(0, 1),
    matern_s = pc_matern(range_gt = 5, sigma_lt = 1),
    matern_st = pc_matern(range_gt = 5, sigma_lt = 1))
)
}
}
\references{
Fuglstad, G.-A., Simpson, D., Lindgren, F., and Rue, H. (2016) Constructing
Priors that Penalize the Complexity of Gaussian Random Fields.
arXiv:1503.00256

Simpson, D., Rue, H., Martins, T., Riebler, A., and Sørbye, S. (2015)
Penalising model component complexity: A principled, practical approach to
constructing priors. arXiv:1403.4630
}
\seealso{
\code{\link[=plot_pc_matern]{plot_pc_matern()}}
}
