% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/slim-select.R
\name{slimSelectInput}
\alias{slimSelectInput}
\title{Slim Select Input}
\usage{
slimSelectInput(
  inputId,
  label,
  choices,
  selected = NULL,
  multiple = FALSE,
  search = TRUE,
  placeholder = NULL,
  allowDeselect = NULL,
  closeOnSelect = !multiple,
  keepOrder = NULL,
  alwaysOpen = NULL,
  contentPosition = NULL,
  ...,
  inline = FALSE,
  width = NULL
)
}
\arguments{
\item{inputId}{The \code{input} slot that will be used to access the value.}

\item{label}{Display label for the control, or \code{NULL} for no label.}

\item{choices}{List of values to select from.
You can use:
\itemize{
\item \code{vector} a simple vector.
\item \verb{named list} / \verb{named vector} in the same way as with \code{\link[shiny:selectInput]{shiny::selectInput()}}
\item cuxtom choices prepared with \code{\link[=prepare_slim_choices]{prepare_slim_choices()}}.
}}

\item{selected}{The initially selected value (or multiple values if \code{multiple = TRUE}). If not specified then defaults to the first value for
single-select lists and no values for multiple select lists.}

\item{multiple}{Is selection of multiple items allowed?}

\item{search}{Enable search feature.}

\item{placeholder}{Placeholder text.}

\item{allowDeselect}{This will allow you to deselect a value on a single/multiple select dropdown.}

\item{closeOnSelect}{A boolean value in which determines whether or not to close the content area upon selecting a value.}

\item{keepOrder}{If \code{TRUE} will maintain the order in which options are selected.}

\item{alwaysOpen}{If \code{TRUE} keep the select open at all times.}

\item{contentPosition}{Will set the css position to either relative or absolute.}

\item{...}{Other settings passed to Slim Select JAvaScript method.}

\item{inline}{Display the widget inline.}

\item{width}{The width of the input, e.g. \code{'400px'}, or \code{'100\%'};
see \code{\link[shiny:validateCssUnit]{validateCssUnit()}}.}
}
\value{
A \code{shiny.tag} object that can be used in a UI definition.
}
\description{
An advanced select dropdown,
based on \href{https://github.com/brianvoe/slim-select}{slim-select} JavaScript library.
}
\examples{

library(shiny)
library(shinyWidgets)
library(htmltools)

state_data <- data.frame(
  name = state.name,
  abb = state.abb,
  region = state.region,
  division = state.division
)

ui <- fluidPage(
  tags$h2("Slim Select examples"),
  fluidRow(
    column(
      width = 4,

      slimSelectInput(
        inputId = "slim1",
        label = "Single slim select:",
        choices = month.name,
        width = "100\%"
      ),
      verbatimTextOutput("res1"),

      slimSelectInput(
        inputId = "slim4",
        label = "Allow deselect in single select:",
        choices = month.name,
        placeholder = "Select something:",
        allowDeselect = TRUE,
        width = "100\%"
      ),
      verbatimTextOutput("res4")

    ),
    column(
      width = 4,

      slimSelectInput(
        inputId = "slim2",
        label = "Multiple slim select:",
        choices = month.name,
        multiple = TRUE,
        placeholder = "Select a month",
        width = "100\%"
      ),
      verbatimTextOutput("res2"),

      slimSelectInput(
        inputId = "slim5",
        label = "Keep order:",
        choices = month.name,
        multiple = TRUE,
        keepOrder = TRUE,
        width = "100\%"
      ),
      verbatimTextOutput("res5")

    ),
    column(
      width = 4,

      slimSelectInput(
        inputId = "slim3",
        label = "Use prepare_slim_choices:",
        choices = prepare_slim_choices(
          state_data,
          label = name,
          value = abb,
          .by = region,
          selectAll = TRUE,
          closable = "close"
        ),
        multiple = TRUE,
        width = "100\%"
      ),
      verbatimTextOutput("res3"),

      slimSelectInput(
        inputId = "slim6",
        label = "Always open:",
        choices = month.name,
        multiple = TRUE,
        alwaysOpen = TRUE,
        # contentPosition = "relative",
        # contentLocation = "slim6-placeholder",
        width = "100\%"
      ) |> htmltools::tagAppendAttributes(style = css(height = "350px")),
      verbatimTextOutput("res6")

    )
  )
)

server <- function(input, output, session) {

  output$res1 <- renderPrint(input$slim1)

  output$res2 <- renderPrint(input$slim2)

  output$res3 <- renderPrint(input$slim3)

  output$res4 <- renderPrint(input$slim4)

  output$res5 <- renderPrint(input$slim5)

  output$res6 <- renderPrint(input$slim6)

}

if (interactive())
  shinyApp(ui, server)
}
