% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/helpers.R
\name{trimmed.weighted.mean}
\alias{trimmed.weighted.mean}
\title{Weighted trimmed mean}
\usage{
trimmed.weighted.mean(x, trim = 0, w = NULL, na.rm = FALSE, ...)
}
\arguments{
\item{x}{Numeric vector of data values.}

\item{trim}{Single number in \eqn{[0,\,0.5]}. Fraction of the total weight to cut
from each tail.}

\item{w}{Numeric vector of non-negative weights of the same length as `x`.
If `NULL` (default), equal weights are used.}

\item{na.rm}{Logical: should `NA` values in `x` or `w` be removed?}

\item{...}{Further arguments passed to [`weighted.mean()`] (for compatibility).}
}
\value{
A single numeric value: the trimmed weighted mean of `x`. Returns `NA_real_`
if no non-`NA` observations remain after optional `na.rm` handling.
}
\description{
Compute a weighted trimmed mean, i.e. a mean that assigns non-negative weights
to the observations and (2)  discards an equal share of total weight from
each tail of the distribution before averaging.
}
\details{
For example, `trim = 0.10` removes 10% of the weight from the left tail and 10%
from the right (20% in total), then takes the weighted mean of what is left.
Setting `trim = 0.5` returns the weighted median.


The algorithm follows these steps:
\enumerate{
  \item Sort the data by `x` and accumulate the corresponding weights.
  \item Identify the lower and upper cut-points that mark the central
        share of the total weight.
  \item Drop observations whose cumulative weight lies entirely
        outside the cut-points and proportionally down-weight the two (at most)
        remaining outermost observations.
  \item Return the weighted mean of the retained mass.  If `trim == 0.5`,
        only the 50% quantile remains, so the function returns the weighted median.
}
}
\examples{
set.seed(1)
z <- rt(100, df = 3)
w <- pmin(1, 1 / abs(z)^2)  # Far-away observations tails get lower weight

mean(z, trim = 0.20)  # Ordinary trimmed mean
trimmed.weighted.mean(z, trim = 0.20)  # Same

weighted.mean(z, w)   # Ordinary weighted mean (no trimming)
trimmed.weighted.mean(z, w = w)  # Same

trimmed.weighted.mean(z, trim = 0.20, w = w)  # Weighted trimmed mean
trimmed.weighted.mean(z, trim = 0.5,  w = w)  # Weighted median
}
\seealso{
[`mean()`] for the unweighted trimmed mean, [`weighted.mean()`] for the untrimmed weighted mean.
}
