% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fetchSDA_spatial.R
\name{fetchSDA_spatial}
\alias{fetchSDA_spatial}
\title{Get Spatial Data from Soil Data Access by \code{mukey}, \code{nationalmusym} or \code{areasymbol}}
\usage{
fetchSDA_spatial(
  x,
  by.col = "mukey",
  method = "feature",
  geom.src = "mupolygon",
  db = "SSURGO",
  add.fields = NULL,
  chunk.size = 10,
  verbose = TRUE
)
}
\arguments{
\item{x}{A vector of map unit keys (\code{mukey}) or national map unit symbols (\code{nmusym}) for \code{mupolygon} geometry OR legend keys (\code{lkey}) or soil survey area symbols (\code{areasymbol}) for \code{sapolygon} geometry}

\item{by.col}{Column name containing map unit identifier \code{"mukey"}, \code{"nmusym"}/\code{"nationalmusym"} for \code{geom.src} \code{mupolygon} OR \code{"areasymbol"} for \code{geom.src} \code{sapolygon}; default is determined by \code{is.numeric(x)} \code{TRUE} for \code{mukey} or \code{lkey} and \code{nationalmusym} or \code{areasymbol} otherwise.}

\item{method}{geometry result type: \code{"feature"} returns polygons, \code{"bbox"} returns the bounding box of each polygon (via \code{STEnvelope()}), and \code{"point"} returns a single point (via \code{STPointOnSurface()}) within each polygon.}

\item{geom.src}{Either \code{mupolygon} (map unit polygons) or \code{sapolygon} (soil survey area boundary polygons)}

\item{db}{Default: SSURGO. When \code{geom.src} is \code{mupolygon}, use STATSGO polygon geometry instead of SSURGO by setting \code{db = "STATSGO"}}

\item{add.fields}{Column names from \code{mapunit} or \code{legend} table to add to result. Must specify parent table name as the prefix before column name e.g. \code{mapunit.muname}.}

\item{chunk.size}{Number of values of \code{x} to process per query. Necessary for large results. Default: \code{10}}

\item{verbose}{Print messages?}
}
\value{
A \code{Spatial*DataFrame} corresponding to SDA spatial data for all symbols requested. Default result contains geometry with attribute table containing unique feature ID, symbol and area symbol plus additional fields in result specified with \code{add.fields}.
}
\description{
This method facilitates queries to Soil Data Access (SDA) mapunit and survey area geometry. Queries are generated based on map unit key (\code{mukey}) and national map unit symbol (\code{nationalmusym}) for \code{mupolygon} (SSURGO) or \code{gsmmupolygon} (STATSGO) geometry OR legend key (\code{lkey}) and area symbols (\code{areasymbol}) for \code{sapolygon} (Soil Survey Area; SSA) geometry).

A Soil Data Access query returns geometry and key identifying information about the map unit or area of interest. Additional columns from the map unit or legend table can be included; see \code{add.fields} argument.
}
\details{
This function automatically "chunks" the input vector (using \code{makeChunks()}) of map unit identifiers to minimize the likelihood of exceeding the SDA data request size. The number of chunks varies with the \code{chunk.size} setting and the length of your input vector. If you are working with many map units and/or large extents, you may need to decrease this number in order to have more chunks.

Querying regions with complex mapping may require smaller \code{chunk.size}. Numerically adjacent IDs in the input vector may share common qualities (say, all from same soil survey area or region) which could cause specific chunks to perform "poorly" (slow or error) no matter what the chunk size is. Shuffling the order of the inputs using \code{sample()} may help to eliminate problems related to this, depending on how you obtained your set of MUKEY/nationalmusym to query. One could feasibly use \code{muacres} as a heuristic to adjust for total acreage within chunks.

Note that STATSGO data are fetched where \code{CLIPAREASYMBOL = 'US'} to avoid duplicating state and national subsets of the geometry.
}
\examples{
\donttest{
if(requireNamespace("curl") &
   curl::has_internet()) {

   # get spatial data for a single mukey
    single.mukey <- fetchSDA_spatial(x = "2924882")

    # demonstrate fetching full extent (multi-mukey) of national musym
    full.extent.nmusym <- fetchSDA_spatial(x = "2x8l5", by = "nmusym")

    # compare extent of nmusym to single mukey within it
    if(require(sp)) {
     plot(full.extent.nmusym, col = "RED",border=0)
     plot(single.mukey, add = TRUE, col = "BLUE", border=0)
    }

    # demo adding a field (`muname`) to attribute table of result
    head(fetchSDA_spatial(x = "2x8l5", by="nmusym", add.fields="muname"))
}
}
}
\author{
Andrew G. Brown, Dylan E. Beaudette
}
