% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/edge_dist.R
\name{edge_dist}
\alias{edge_dist}
\title{Distance based edge lists}
\usage{
edge_dist(
  DT = NULL,
  threshold,
  id = NULL,
  coords = NULL,
  timegroup,
  splitBy = NULL,
  returnDist = FALSE,
  fillNA = TRUE
)
}
\arguments{
\item{DT}{input data.table}

\item{threshold}{distance for grouping points, in the units of the
coordinates}

\item{id}{Character string of ID column name}

\item{coords}{Character vector of X coordinate and Y coordinate column names}

\item{timegroup}{timegroup field in the DT upon which the grouping will be
calculated}

\item{splitBy}{(optional) character string or vector of grouping column
name(s) upon which the grouping will be calculated}

\item{returnDist}{boolean indicating if the distance between individuals
should be returned. If FALSE (default), only ID1, ID2 columns (and
timegroup, splitBy columns if provided) are returned. If TRUE, another
column "distance" is returned indicating the distance between ID1 and ID2.}

\item{fillNA}{boolean indicating if NAs should be returned for individuals
that were not within the threshold distance of any other. If TRUE, NAs are
returned. If FALSE, only edges between individuals within the threshold
distance are returned.}
}
\value{
\code{edge_dist} returns a \code{data.table} with columns ID1, ID2,
timegroup (if supplied) and any columns provided in splitBy. If
'returnDist' is TRUE, column 'distance' is returned indicating the distance
between ID1 and ID2.

The ID1 and ID2 columns represent the edges defined by the spatial (and
temporal with \code{group_times}) thresholds.
}
\description{
\code{edge_dist} returns edge lists defined by a spatial distance within the
user defined threshold. The function accepts a \code{data.table} with
relocation data, individual identifiers and a threshold argument. The
threshold argument is used to specify the criteria for distance between
points which defines a group. Relocation data should be in two columns
representing the X and Y coordinates.
}
\details{
The \code{DT} must be a \code{data.table}. If your data is a
\code{data.frame}, you can convert it by reference using
\code{\link[data.table:setDT]{data.table::setDT}}.

The \code{id}, \code{coords} (and optional \code{timegroup} and
\code{splitBy}) arguments expect the names of a column in \code{DT} which
correspond to the individual identifier, X and Y coordinates, timegroup
(generated by \code{group_times}) and additional grouping columns.

If provided, the \code{threshold} must be provided in the units of the coordinates and must be larger than 0.
If the \code{threshold} is NULL, the distance to all other individuals will be returned. The coordinates must be planar
coordinates (e.g.: UTM). In the case of UTM, a \code{threshold} = 50 would
indicate a 50m distance threshold.

The \code{timegroup} argument is optional, but recommended to pair with
\code{\link{group_times}}. The intended framework is to group rows temporally
with \code{\link{group_times}} then spatially with \code{edge_dist} (or
grouping functions).

The \code{splitBy} argument offers further control over grouping. If within
your \code{DT}, you have multiple populations, subgroups or other distinct
parts, you can provide the name of the column which identifies them to
\code{splitBy}. \code{edge_dist} will only consider rows within each
\code{splitBy} subgroup.
}
\examples{
# Load data.table
library(data.table)

# Read example data
DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))

# Cast the character column to POSIXct
DT[, datetime := as.POSIXct(datetime, tz = 'UTC')]

# Temporal grouping
group_times(DT, datetime = 'datetime', threshold = '20 minutes')

# Edge list generation
edges <- edge_dist(
    DT,
    threshold = 100,
    id = 'ID',
    coords = c('X', 'Y'),
    timegroup = 'timegroup',
    returnDist = TRUE,
    fillNA = TRUE
  )
}
\seealso{
Other Edge-list generation: 
\code{\link{edge_nn}()}
}
\concept{Edge-list generation}
