% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unif_test.R
\name{unif_test}
\alias{unif_test}
\title{Circular and (hyper)spherical uniformity tests}
\usage{
unif_test(data, type = "all", p_value = "asymp", alpha = c(0.1, 0.05,
  0.01), M = 10000, stats_MC = NULL, crit_val = NULL,
  data_sorted = FALSE, Rayleigh_m = 1, cov_a = 2 * pi, Rothman_t = 1/3,
  Cressie_t = 1/3, Pycke_q = 0.5, Riesz_s = 1, CCF09_dirs = NULL,
  K_CCF09 = 25, CJ12_reg = 3, CJ12_beta = 0, K_max = 10000, ...)
}
\arguments{
\item{data}{sample to perform the test. A matrix of size \code{c(n, p)}
containing a sample of size \code{n} of directions (in Cartesian
coordinates) on \eqn{S^{p-1}}. Alternatively if \code{p = 2}, a matrix of
size \code{c(n, 1)} containing the \code{n} angles on \eqn{[0, 2\pi)} of the
circular sample on \eqn{S^{1}}. Other objects accepted are an array of size
\code{c(n, p, 1)} with directions (in Cartesian coordinates), or a vector of
size \code{n} or an array of size \code{c(n, 1, 1)} with angular data.
Must not contain \code{NA}'s.}

\item{type}{type of test to be applied. A character vector containing any of
the following types of tests, depending on the dimension \eqn{p}:
\itemize{
  \item Circular data: any of the names available at object
  \code{\link{avail_cir_tests}}.
  \item (Hyper)spherical data: any of the names available at object
  \code{\link{avail_sph_tests}}.
}
If \code{type = "all"} (default), then \code{type} is set as
\code{avail_cir_tests} or \code{avail_sph_tests}, depending on the value of
\eqn{p}.}

\item{p_value}{type of \eqn{p}-value computation. Either \code{"MC"} for
employing the approximation by Monte Carlo of the exact null distribution,
\code{"asymp"} (default) for the use of the asymptotic/exact null
distribution (if available), or \code{"crit_val"} for approximation by means
of the table of critical values \code{crit_val}.}

\item{alpha}{vector with significance levels. Defaults to
\code{c(0.10, 0.05, 0.01)}.}

\item{M}{number of Monte Carlo replications for approximating the null
distribution when \code{approx = "MC"}. Also, number of Monte Carlo samples
for approximating the asymptotic distributions based on weighted sums of chi
squared random variables. Defaults to \code{1e4}.}

\item{stats_MC}{a data frame of size \code{c(M, length(type))}, with column
names containing the character vector \code{type}, that results from
extracting \code{$stats_MC} from a call to \code{\link{unif_stat_MC}}. If
provided, the computation of Monte Carlo statistics when \code{approx = "MC"}
is skipped. \code{stats_MC} is checked internally to see if it is sorted.
Internally computed if \code{NULL} (default).}

\item{crit_val}{table with critical values for the tests, to be used if
\code{p_value = "crit_val"}. A data frame, with column names containing the
character vector \code{type} and rows corresponding to the significance
levels \code{alpha}, that results from extracting \code{$crit_val_MC} from
a call to \code{\link{unif_stat_MC}}. Internally computed if
\code{NULL} (default).}

\item{data_sorted}{is the circular data sorted? If \code{TRUE}, certain
statistics are faster to compute. Defaults to \code{FALSE}.}

\item{Rayleigh_m}{integer \eqn{m} for the \eqn{m}-modal Rayleigh test.
Defaults to \code{m = 1} (the standard Rayleigh test).}

\item{cov_a}{\eqn{a_n = a / n} parameter used in the length of the arcs
of the coverage-based tests. Must be positive. Defaults to \code{2 * pi}.}

\item{Rothman_t}{\eqn{t} parameter for the Rothman test, a real in
\eqn{(0, 1)}. Defaults to \code{1 / 3}.}

\item{Cressie_t}{\eqn{t} parameter for the Cressie test, a real in
\eqn{(0, 1)}. Defaults to \code{1 / 3}.}

\item{Pycke_q}{\eqn{q} parameter for the Pycke "\eqn{q}-test", a real in
\eqn{(0, 1)}. Defaults to \code{1 / 2}.}

\item{Riesz_s}{\eqn{s} parameter for the \eqn{s}-Riesz test, a real in
\eqn{(0, 2)}. Defaults to \code{1}.}

\item{CCF09_dirs}{a matrix of size \code{c(n_proj, p)} containing
\code{n_proj} random directions (in Cartesian coordinates) on \eqn{S^{p-1}}
to perform the CCF09 test. If \code{NULL} (default), a sample of size
\code{n_proj = 50} directions is computed internally.}

\item{K_CCF09}{integer giving the truncation of the series present in the
asymptotic distribution of the Kolmogorov-Smirnov statistic. Defaults to
\code{5e2}.}

\item{CJ12_reg}{type of asymptotic regime for CJ12 test, either \code{1}
(sub-exponential regime), \code{2} (exponential), or \code{3}
(super-exponential; default).}

\item{CJ12_beta}{\eqn{\beta} parameter in the exponential regime of CJ12 test,
a positive real.}

\item{K_max}{integer giving the truncation of the series that compute the
asymptotic p-value of a Sobolev test. Defaults to \code{1e4}.}

\item{...}{If \code{p_value = "MC"} or \code{p_value = "crit_val"}, optional
performance parameters to be passed to \code{\link{unif_stat_MC}}:
\code{chunks}, \code{cores}, and \code{seed}.}
}
\value{
If only a \bold{single test} is performed, a list with class
\code{htest} containing the following components:
\itemize{
  \item \code{statistic}: the value of the test statistic.
  \item \code{p.value}: the p-value of the test. If
  \code{p_value = "crit_val"}, an \code{NA}.
  \item \code{alternative}: a character string describing the alternative
  hypothesis.
  \item \code{method}: a character string indicating what type of test was
  performed.
  \item \code{data.name}: a character string giving the name of the data.
  \item \code{reject}: the rejection decision for the levels of significance
  \code{alpha}.
  \item \code{crit_val}: a vector with the critical values for the
  significance levels \code{alpha} used with \code{p_value = "MC"} or
  \code{p_value = "asymp"}.
}
If \bold{several tests} are performed, a \code{type}-named list with
entries for each test given by the above list.
}
\description{
Implementation of several uniformity tests on the (hyper)sphere
\eqn{S^{p-1}:=\{{\bf x}\in R^p:||{\bf x}||=1\}}{
S^{p-1}:=\{x\in R^p:||x||=1\}}, \eqn{p\ge 2}, with calibration either in
terms of their asymptotic/exact distributions, if available, or Monte Carlo.

\code{unif_test} receives a sample of directions
\eqn{{\bf X}_1,\ldots,{\bf X}_n\in S^{p-1}}{X_1,\ldots,X_n\in S^{p-1}} in
\emph{Cartesian coordinates}, except for the circular case (\eqn{p=2}) in
which the sample can be represented in terms of \emph{angles}
\eqn{\Theta_1,\ldots,\Theta_n\in [0, 2\pi)}.

\code{unif_test} allows to perform several tests within a single call,
facilitating thus the exploration of a dataset by applying several tests.
}
\details{
All the tests reject for large values of the test statistic, so the critical
values for the significance levels \code{alpha} correspond to the
\code{alpha}-upper quantiles of the null distribution of the test statistic.

When \code{p_value = "asymp"}, tests that do not have an implemented or
known asymptotic are omitted, and a warning is generated.

When \code{p_value = "MC"}, it is possible to have a progress bar indicating
the Monte Carlo simulation progress if \code{unif_test} is wrapped with
\code{\link[progressr:with_progress]{progressr::with_progress}} or if
\code{progressr::handlers(global = TRUE)} is invoked (once) by the user.
See the examples below. The progress bar is updated with the number of
finished chunks.

All the statistics are continuous random variables except the
Hodges--Ajne statistic (\code{"Hodges_Ajne"}), the Cressie statistic
(\code{"Cressie"}), and the number of (different) uncovered spacings
(\code{"Num_uncover"}). These three statistics are discrete random variables.

The Monte Carlo calibration for the CCF09 test is made conditionally
on the choice of \cr\code{CCF09_dirs}. That is, all the Monte
Carlo statistics share the same random directions.

Descriptions and references for most of the tests are available
in García-Portugués and Verdebout (2018).
}
\examples{
## Asymptotic distribution

# Circular data
n <- 10
samp_cir <- r_unif_cir(n = n)

# Matrix
unif_test(data = samp_cir, type = "Ajne", p_value = "asymp")

# Vector
unif_test(data = samp_cir[, 1], type = "Ajne", p_value = "asymp")

# Array
unif_test(data = array(samp_cir, dim = c(n, 1, 1)), type = "Ajne",
          p_value = "asymp")
\donttest{
# Several tests
unif_test(data = samp_cir, type = avail_cir_tests, p_value = "asymp")
}
# Spherical data
n <- 10
samp_sph <- r_unif_sph(n = n, p = 3)

# Array
unif_test(data = samp_sph, type = "Bingham", p_value = "asymp")

# Matrix
unif_test(data = samp_sph[, , 1], type = "Bingham", p_value = "asymp")
\donttest{
# Several tests
unif_test(data = samp_sph, type = avail_sph_tests, p_value = "asymp")

## Monte Carlo

# Circular data
unif_test(data = samp_cir, type = "Ajne", p_value = "MC")
unif_test(data = samp_cir, type = avail_cir_tests, p_value = "MC")

# Spherical data
unif_test(data = samp_sph, type = "Bingham", p_value = "MC")
unif_test(data = samp_sph, type = avail_sph_tests, p_value = "MC")

# Caching stats_MC
stats_MC_cir <- unif_stat_MC(n = nrow(samp_cir), p = 2)$stats_MC
stats_MC_sph <- unif_stat_MC(n = nrow(samp_sph), p = 3)$stats_MC
unif_test(data = samp_cir, type = avail_cir_tests,
          p_value = "MC", stats_MC = stats_MC_cir)
unif_test(data = samp_sph, type = avail_sph_tests, p_value = "MC",
          stats_MC = stats_MC_sph)

## Critical values

# Circular data
unif_test(data = samp_cir, type = avail_cir_tests, p_value = "crit_val")

# Spherical data
unif_test(data = samp_sph, type = avail_sph_tests, p_value = "crit_val")

# Caching crit_val
crit_val_cir <- unif_stat_MC(n = n, p = 2)$crit_val_MC
crit_val_sph <- unif_stat_MC(n = n, p = 3)$crit_val_MC
unif_test(data = samp_cir, type = avail_cir_tests,
          p_value = "crit_val", crit_val = crit_val_cir)
unif_test(data = samp_sph, type = avail_sph_tests, p_value = "crit_val",
          crit_val = crit_val_sph)

## Specific arguments

# Rothman
unif_test(data = samp_cir, type = "Rothman", Rothman_t = 0.5)

# CCF09
unif_test(data = samp_sph, type = "CCF09", p_value = "MC",
          CCF09_dirs = samp_sph[1:2, , 1])
unif_test(data = samp_sph, type = "CCF09", p_value = "MC",
          CCF09_dirs = samp_sph[3:4, , 1])

## Using a progress bar when p_value = "MC"

# Define a progress bar
require(progress)
require(progressr)
handlers(handler_progress(
  format = ":spin [:bar] :percent Total: :elapsedfull End \u2248 :eta",
  clear = FALSE))

# Call unif_test() within with_progress()
with_progress(
  unif_test(data = samp_sph, type = avail_sph_tests, p_value = "MC",
            chunks = 10, M = 1e3)
)

# With several cores
with_progress(
  unif_test(data = samp_sph, type = avail_sph_tests, p_value = "MC",
            cores = 2, chunks = 10, M = 1e3)
)

# Instead of using with_progress() each time, it is more practical to run
# handlers(global = TRUE)
# once to activate progress bars in your R session
}
}
\references{
García-Portugués, E. and Verdebout, T. (2018) An overview of uniformity
tests on the hypersphere. \emph{arXiv:1804.00286}.
\url{https://arxiv.org/abs/1804.00286}.
}
