% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimMSSD.R
\name{optimMSSD}
\alias{optimMSSD}
\alias{objMSSD}
\alias{MSSD}
\alias{objMSSD}
\title{Optimization of sample configurations for spatial interpolation (I)}
\usage{
optimMSSD(points, candi, schedule = scheduleSPSANN(), plotit = FALSE,
  track = FALSE, boundary, progress = "txt", verbose = FALSE)

objMSSD(points, candi)
}
\arguments{
\item{points}{Integer value, integer vector, data frame or matrix, or list.
\itemize{
\item Integer value. The number of points. These points will be randomly sampled from \code{candi} to form
the starting sample configuration.
\item Integer vector. The row indexes of \code{candi} that correspond to the points that form the starting
sample configuration. The length of the vector defines the number of points.
\item Data frame or matrix. An object with three columns in the following order: \code{[, "id"]}, the 
row indexes of \code{candi} that correspond to each point, \code{[, "x"]}, the projected x-coordinates, and
\code{[, "y"]}, the projected y-coordinates.
\item List. An object with two named sub-arguments: \code{fixed}, a data frame or matrix with the projected
x- and y-coordinates of the existing sample configuration -- kept fixed during the optimization --, and 
\code{free}, an integer value defining the number of points that should be added to the existing sample
configuration -- free to move during the optimization.
}}

\item{candi}{Data frame or matrix with the candidate locations for the jittered points. \code{candi} must 
have two columns in the following order: \code{[, "x"]}, the projected x-coordinates, and \code{[, "y"]}, 
the projected y-coordinates.}

\item{schedule}{List with 11 named sub-arguments defining the control parameters of the cooling schedule. 
See \code{\link[spsann]{scheduleSPSANN}}.}

\item{plotit}{(Optional) Logical for plotting the optimization results, including a) the progress of the
objective function, and b) the starting (gray circles) and current sample configuration (black dots), and 
the maximum jitter in the x- and y-coordinates. The plots are updated at each 10 jitters. When adding 
points to an existing sample configuration, fixed points are indicated using black crosses. Defaults to 
\code{plotit = FALSE}.}

\item{track}{(Optional) Logical value. Should the evolution of the energy state be recorded and returned 
along with the result? If \code{track = FALSE} (the default), only the starting and ending energy states 
are returned along with the results.}

\item{boundary}{(Optional) SpatialPolygon defining the boundary of the spatial domain. If missing and 
\code{plotit = TRUE}, \code{boundary} is estimated from \code{candi}.}

\item{progress}{(Optional) Type of progress bar that should be used, with options \code{"txt"}, for a text
progress bar in the R console, \code{"tk"}, to put up a Tk progress bar widget, and \code{NULL} to omit the
progress bar. A Tk progress bar widget is useful when using parallel processors. Defaults to 
\code{progress = "txt"}.}

\item{verbose}{(Optional) Logical for printing messages about the progress of the optimization. Defaults to 
\code{verbose = FALSE}.}
}
\value{
\code{optimMSSD} returns an object of class \code{OptimizedSampleConfiguration}: the optimized sample
configuration with details about the optimization.

\code{objMSSD} returns a numeric value: the energy state of the sample configuration -- the objective
function value.
}
\description{
Optimize a sample configuration for spatial interpolation. The criterion used is the mean squared shortest 
distance (\bold{MSSD}) between sample points and prediction points.
}
\details{
Details about the mechanism used to generate a new sample configuration out of the current sample 
configuration by randomly perturbing the coordinates of a sample point are available in the help page of
\code{\link[spsann]{spJitter}}.

\subsection{Spatial coverage sampling}{
Spatial coverage sampling is based on the knowledge that the kriging variance depends upon the distance 
between sample points. As such, the better the spread of the sample points in the spatial domain, the 
smaller the kriging variance. This is similar to using a regular grid of sample points. However, 
a regular grid usually is suboptimal for irregularly shaped areas.
}
}
\note{
The distance between two points is computed as the Euclidean distance between them. This computation 
assumes that the optimization is operating in the two-dimensional Euclidean space, i.e. the coordinates of
the sample points and candidate locations should not be provided as latitude/longitude. \pkg{spsann} has no 
mechanism to check if the coordinates are projected: the user is responsible for making sure that this
requirement is attained.

This function was derived with modifications from the method known as \emph{spatial coverage sampling} 
originally proposed by Brus, de Gruijter and van Groenigen (2006), and implemented in the R-package 
\pkg{\link[spcosa]{spcosa}} by Dennis Walvoort, Dick Brus and Jaap de Gruijter.
}
\examples{
require(sp)
data(meuse.grid)
candi <- meuse.grid[, 1:2]
schedule <- scheduleSPSANN(chains = 1, initial.temperature = 5000000,
                           x.max = 1540, y.max = 2060, x.min = 0, 
                           y.min = 0, cellsize = 40)
set.seed(2001)
res <- optimMSSD(points = 10, candi = candi, schedule = schedule)
objSPSANN(res) - objMSSD(candi = candi, points = res)

}
\references{
Brus, D. J.; de Gruijter, J. J.; van Groenigen, J.-W. Designing spatial coverage samples using the k-means
clustering algorithm. In: P. Lagacherie,A. M.; Voltz, M. (Eds.) \emph{Digital soil mapping -- an 
introductory perspective}. Elsevier, v. 31, p. 183-192, 2006.

de Gruijter, J. J.; Brus, D.; Bierkens, M.; Knotters, M. \emph{Sampling for natural resource monitoring}.
Berlin: Springer, p. 332, 2006.

Walvoort, D. J. J.; Brus, D. J.; de Gruijter, J. J. An R package for spatial coverage sampling and random 
sampling from compact geographical strata by k-means. \emph{Computers and Geosciences}. v. 36, p. 
1261-1267, 2010.
}
\seealso{
\code{\link[raster]{distanceFromPoints}}, \code{\link[spcosa]{stratify}}.
}
\author{
Alessandro Samuel-Rosa \email{alessandrosamuelrosa@gmail.com}
}
\concept{
simulated annealing

spatial interpolation
}
\keyword{iteration}
\keyword{optimize}
\keyword{spatial}
