% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wrappers.R
\name{svlm}
\alias{svlm}
\title{Markov Chain Monte Carlo (MCMC) Sampling for the Stochastic Volatility (SV)
Model}
\usage{
svlm(
  formula,
  data,
  draws = 10000,
  burnin = 1000,
  heavytails = FALSE,
  asymmetry = FALSE,
  priorspec = NULL,
  thin = 1,
  keeptime = "all",
  quiet = FALSE,
  startpara = NULL,
  startlatent = NULL,
  parallel = c("no", "multicore", "snow"),
  n_cpus = 1L,
  cl = NULL,
  n_chains = 1L,
  print_progress = "automatic",
  expert = NULL,
  ...
)
}
\arguments{
\item{formula}{an object of class \code{"formula"}, as in \code{\link{lm}}.}

\item{data}{an optional data frame, list or environment (or object
coercible by \code{as.data.frame} to a data frame) containing the
variables in the model.  If not found in \code{data}, the
variables are taken from \code{environment(formula)}, typically
the environment from which \code{svlm} is called.}

\item{draws}{single number greater or equal to 1, indicating the number of
draws after burn-in (see below). Will be automatically coerced to integer.
The default value is 10000.}

\item{burnin}{single number greater or equal to 0, indicating the number of
draws discarded as burn-in. Will be automatically coerced to integer. The
default value is 1000.}

\item{heavytails}{if \code{TRUE}, then the residuals of the linear model
will follow a t-distribution conditional on the latent volatility process.
This model is usually called SV-t. If \code{priorspec} is given, then
\code{heavytails} is ignored.}

\item{asymmetry}{if \code{TRUE}, then the residuals of the linear model
will follow an SV process with leverage. If \code{priorspec} is given, then
\code{heavytails} is ignored.}

\item{priorspec}{using the smart constructor \code{\link{specify_priors}},
one can set the details of the prior distribution.}

\item{thin}{single number greater or equal to 1, coercible to integer.
Every \code{thinpara}th parameter and latent draw is kept and returned. The default
value is 1, corresponding to no thinning of the parameter draws i.e. every
draw is stored.}

\item{keeptime}{Either 'all' (the default) or 'last'. Indicates which latent
volatility draws should be stored.}

\item{quiet}{logical value indicating whether the progress bar and other
informative output during sampling should be omitted. The default value is
\code{FALSE}, implying verbose output.}

\item{startpara}{\emph{optional} named list, containing the starting values
for the parameter draws. If supplied, \code{startpara} may contain
elements named \code{mu}, \code{phi}, \code{sigma}, \code{nu}, \code{rho},
\code{beta}, and \code{latent0}.
The default value is equal to the prior mean.
In case of parallel execution with \code{cl} provided, \code{startpara} can be a list of
named lists that initialize the parallel chains.}

\item{startlatent}{\emph{optional} vector of length \code{length(y)},
containing the starting values for the latent log-volatility draws. The
default value is \code{rep(-10, length(y))}.
In case of parallel execution with \code{cl} provided, \code{startlatent} can be a list of
named lists that initialize the parallel chains.}

\item{parallel}{\emph{optional} one of \code{"no"} (default), \code{"multicore"}, or \code{"snow"},
indicating what type of parallellism is to be applied. Option
\code{"multicore"} is not available on Windows.}

\item{n_cpus}{\emph{optional} positive integer, the number of CPUs to be used in case of
parallel computations. Defaults to \code{1L}. Ignored if parameter
\code{cl} is supplied and \code{parallel != "snow"}.}

\item{cl}{\emph{optional} so-called SNOW cluster object as implemented in package
\code{parallel}. Ignored unless \code{parallel == "snow"}.}

\item{n_chains}{\emph{optional} positive integer specifying the number of independent MCMC chains}

\item{print_progress}{\emph{optional} one of \code{"automatic"}, \code{"progressbar"},
or \code{"iteration"}, controls the output. Ignored if \code{quiet} is \code{TRUE}.}

\item{expert}{\emph{optional} named list of expert parameters. For most
applications, the default values probably work best. Interested users are
referred to the literature provided in the References section. If
\code{expert} is provided, it may contain the following named elements:
\describe{
\item{interweave}{Logical value. If \code{TRUE} (the default),
then ancillarity-sufficiency interweaving strategy (ASIS) is applied
to improve on the sampling efficiency for the parameters.
Otherwise one parameterization is used.}
\item{correct_model_misspecification}{Logical value. If \code{FALSE}
(the default), then auxiliary mixture sampling is used to sample the latent
states. If \code{TRUE}, extra computations are made to correct for model
misspecification either ex-post by reweighting or on-line using a
Metropolis-Hastings step.}
}}

\item{\dots}{Any extra arguments will be forwarded to
\code{\link{updatesummary}}, controlling the type of statistics calculated
for the posterior draws.}
}
\value{
The value returned is a list object of class \code{svdraws} holding
\item{para}{\code{mcmc.list} object containing the \emph{parameter} draws from
the posterior distribution.}
\item{latent}{\code{mcmc.list} object containing the
\emph{latent instantaneous log-volatility} draws from the posterior
distribution.}
\item{latent0}{\code{mcmc.list} object containing the \emph{latent
initial log-volatility} draws from the posterior distribution.}
\item{tau}{\code{mcmc.list} object containing the \emph{latent variance inflation
factors} for the sampler with conditional t-innovations \emph{(optional)}.}
\item{beta}{\code{mcmc.list} object containing the \emph{regression coefficient}
draws from the posterior distribution \emph{(optional)}.}
\item{y}{the left hand side of the observation equation, usually
the argument \code{y}. In case of an AR(\code{k}) specification, the
first \code{k} elements are removed.}
\item{runtime}{\code{proc_time} object containing the
run time of the sampler.}
\item{priors}{a \code{priorspec} object containing the parameter
values of the prior distributions for \code{mu},
\code{phi}, \code{sigma}, \code{nu}, \code{rho}, and
\code{beta}s, and the variance of specification for \code{latent0}.}
\item{thinning}{\code{list} containing the thinning
parameters, i.e. the arguments \code{thinpara}, \code{thinlatent} and
\code{keeptime}.}
\item{summary}{\code{list} containing a collection of
summary statistics of the posterior draws for \code{para}, \code{latent},
and \code{latent0}.}
\item{meanmodel}{\code{character} containing information about how \code{designmatrix}
was employed.}
\item{svlm}{a flag for the use of \code{svlm}}
\item{model_terms}{helper object that represents the formula}
\item{formula}{argument \code{formula}}
\item{xlevels}{helper object that is needed to interpret the formula}

To display the output, use \code{print}, \code{summary} and \code{plot}. The
\code{print} method simply prints the posterior draws (which is very likely
a lot of output); the \code{summary} method displays the summary statistics
currently stored in the object; the \code{plot} method
\code{\link{plot.svdraws}} gives a graphical overview of the posterior
distribution by calling \code{\link{volplot}}, \code{\link{traceplot}} and
\code{\link{densplot}} and displaying the results on a single page.
}
\description{
\code{svlm} is a wrapper around \code{\link{svsample}} with a formula interface.
The name derives from SV and \code{\link{lm}} because a linear model with SV residuals is fitted.
The function simulates from the joint posterior distribution of the regression coefficients and the SV
parameters \code{mu}, \code{phi}, \code{sigma} (and potentially \code{nu} and \code{rho}),
along with the latent log-volatilities \code{h_0,...,h_n} and returns the
MCMC draws.
}
\details{
For details concerning the algorithm please see the paper by Kastner and
Frühwirth-Schnatter (2014) and Hosszejni and Kastner (2019).
}
\examples{
# Simulate data
n <- 50L
dat <- data.frame(x = runif(n, 3, 4),
                  z = runif(n, -1, -0.5))
designmatrix <- matrix(c(dat$x, dat$x^2, log10(dat$x),
                         dat$z), ncol = 4)
betas <- matrix(c(-1, 1, 2, 0), ncol = 1)
y <- designmatrix \%*\% betas + svsim(n)$y
dat$y <- y
# Formula interface
res <- svlm(y ~ 0 + x + I(x^2) + log10(x) + z, data = dat)
# Prediction
predn <- 10L
preddat <- data.frame(x = runif(predn, 3, 4),
                      z = runif(predn, -1, -0.5))
pred <- predict(res, newdata = preddat, steps = predn)
}
\references{
Kastner, G. and Frühwirth-Schnatter, S. (2014).
Ancillarity-sufficiency interweaving strategy (ASIS) for boosting MCMC
estimation of stochastic volatility models. \emph{Computational Statistics &
Data Analysis}, \bold{76}, 408--423,
\doi{10.1016/j.csda.2013.01.002}.

Hosszejni, D. and Kastner, G. (2019).
Approaches Toward the Bayesian Estimation of the Stochastic Volatility Model with Leverage.
\emph{Springer Proceedings in Mathematics & Statistics}, \bold{296}, 75--83,
\doi{10.1007/978-3-030-30611-3_8}.
}
\seealso{
\code{\link{svsample}}, \code{\link{svsim}}, \code{\link{specify_priors}}
}
\keyword{models}
\keyword{ts}
