% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as_period.R
\name{as_period}
\alias{as_period}
\title{Change \code{tbl_time} periodicity}
\usage{
as_period(.tbl_time, period = "yearly", start_date = NULL, side = "start",
  include_endpoints = FALSE, ...)
}
\arguments{
\item{.tbl_time}{A \code{tbl_time} object.}

\item{period}{A character specification used for time-based grouping. The
general format to use is \code{"frequency period"} where frequency is a number
like 1 or 2, and period is an interval like weekly or yearly. There must be
a space between the two.

Note that you can pass the specification in a flexible way:
\itemize{
\item 1 Year: \code{'1 year'} / \code{'1 Y'} / \code{'1 yearly'} / \code{'yearly'}
}

This shorthand is available for year, quarter, month, day, hour, minute,
and second periodicities. Note that "m" is the 1 letter specification used
for "month" and "M" is used for "minute". Otherwise case does not matter.}

\item{start_date}{Optional argument used to specify the start date for the
first group. The default is to start at the closest period boundary
below the minimum date in the supplied index.}

\item{side}{Whether to return the date at the beginning or the end of the
new period. By default, the \code{"start"} of the period. Use \code{"end"} to change
to the end of the period.}

\item{include_endpoints}{Whether to include the first or last data point in
addition to the transformed data.}

\item{...}{Not currently used.}
}
\description{
Convert a \code{tbl_time} object from daily to monthly,
from minute data to hourly, and more. This allows the user to easily
aggregate data to a less granular level by taking the value from either
the beginning or end of the period.
}
\details{
This function respects \code{\link[dplyr:group_by]{dplyr::group_by()}} groups.

Currently periods finer than second data are not supported.

The \code{side} argument is useful when you want to return data at, say, the
end of a quarter, or the end of a month.

\code{include_endpoints} can be useful when calculating a change over time.
In addition to changing to monthly dates, you often need the first data point
as a baseline for the first calculation.
}
\examples{

# Basic usage ---------------------------------------------------------------

# FB stock prices
data(FB)
FB <- as_tbl_time(FB, date)

# Aggregate FB to yearly data
as_period(FB, "yearly")

# Aggregate FB to every 2 years
as_period(FB, "2 years")

# Aggregate FB to yearly data, but use the last data point available
# in that period
as_period(FB, "yearly", side = "end")

# Aggregate FB to yearly data, end of period, and include the first
# endpoint
as_period(FB, "yearly", side = "end", include_endpoints = TRUE)

# Aggregate to weekly. Notice that it only uses the earliest day available
# in the data set at that periodicity. It will not set the date of the first
# row to 2013-01-01 because that date did not exist in the original data set.
as_period(FB, "weekly")

# Aggregate to every other week
as_period(FB, "2 weeks")

# FB is daily data, aggregate to minute?
# Not allowed for Date class indices, an error is thrown
# as_period(FB, "minute")

# Grouped usage -------------------------------------------------------------

# FANG contains Facebook, Amazon, Netflix and Google stock prices
data(FANG)
FANG <- as_tbl_time(FANG, date)

FANG <- dplyr::group_by(FANG, symbol)

# Respects groups
as_period(FANG, "yearly")

# Every 6 months, respecting groups
as_period(FANG, "6 months")

# Using start_date ----------------------------------------------------------

# FB stock prices
data(FB)
FB <- as_tbl_time(FB, date)

# The Facebook series starts at 2013-01-02 so the 'every 2 day' counter
# starts at that date as well. Groups become (2013-01-02, 2013-01-03),
# (2013-01-04, 2013-01-05) and so on.
as_period(FB, "2 day")

# Specifying the `start_date = "2013-01-01"` might be preferable.
# Groups become (2013-01-01, 2013-01-02), (2013-01-03, 2013-01-04) and so on.
as_period(FB, "2 day", start_date = "2013-01-01")

# Manually calculating returns at different periods -------------------------

data(FB)

# Annual Returns
# Convert to end of year periodicity, but include the endpoints to use as
# a reference for the first return calculation. Then calculate returns.
FB \%>\%
  as_tbl_time(date) \%>\%
  as_period("1 y", side = "end", include_endpoints = TRUE) \%>\%
  dplyr::mutate(yearly_return = adjusted / dplyr::lag(adjusted) - 1)

}
