% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gt_cluster_pca.R
\name{gt_cluster_pca}
\alias{gt_cluster_pca}
\title{Run K-clustering on principal components}
\usage{
gt_cluster_pca(
  x = NULL,
  n_pca = NULL,
  k_clusters = c(1, round(nrow(x$u)/10)),
  method = c("kmeans", "ward"),
  n_iter = 1e+05,
  n_start = 10,
  quiet = FALSE
)
}
\arguments{
\item{x}{a \code{gt_pca} object returned by one of the \verb{gt_pca_*} functions.}

\item{n_pca}{number of principal components to be fed to the LDA.}

\item{k_clusters}{number of clusters to explore, either a single value, or a
vector of length 2 giving the minimum and maximum (e.g. 1:5). If left NULL,
it will use 1 to the number of pca components divided by 10 (a reasonable
guess).}

\item{method}{either 'kmeans' or 'ward'}

\item{n_iter}{number of iterations for kmeans (only used if
\code{method="kmeans"})}

\item{n_start}{number of starting points for kmeans (only used if
\code{method="kmeans"})}

\item{quiet}{boolean on whether to silence outputting information to the
screen (defaults to FALSE)}
}
\value{
a \code{gt_cluster_pca} object, which is a subclass of \code{gt_pca} with an
additional element 'cluster', a list with elements:
\itemize{
\item 'method' the clustering method (either kmeans or ward)
\item 'n_pca' number of principal components used for clustering
\item 'k' the k values explored by the function
\item 'WSS' within sum of squares for each k
\item 'AIC' the AIC for each k
\item 'BIC' the BIC for each k
\item 'groups' a list, with each element giving the group assignments
for a given k
}
}
\description{
This function implements the clustering procedure used in Discriminant
Analysis of Principal Components (DAPC, Jombart et al. 2010). This procedure
consists in running successive K-means with an increasing number of clusters
(k), after transforming data using a principal component analysis (PCA). For
each model, several statistical measures of goodness of fit are computed,
which allows to choose the optimal k using the function
\code{\link[=gt_cluster_pca_best_k]{gt_cluster_pca_best_k()}}. See details for a description of how to select the
optimal k and vignette("adegenet-dapc") for a tutorial.
}
\examples{
# Create a gen_tibble of lobster genotypes
bed_file <-
  system.file("extdata", "lobster", "lobster.bed", package = "tidypopgen")
lobsters <- gen_tibble(bed_file,
  backingfile = tempfile("lobsters"),
  quiet = TRUE
)

# Remove monomorphic loci and impute
lobsters <- lobsters \%>\% select_loci_if(loci_maf(genotypes) > 0)
lobsters <- gt_impute_simple(lobsters, method = "mode")

# Create PCA object
pca <- gt_pca_partialSVD(lobsters)

# Run clustering on the first 10 PCs
gt_cluster_pca(
  x = pca,
  n_pca = 10,
  k_clusters = c(1, 5),
  method = "kmeans",
  n_iter = 1e5,
  n_start = 10,
  quiet = FALSE
)

# Alternatively, use method "ward"
gt_cluster_pca(
  x = pca,
  n_pca = 10,
  k_clusters = c(1, 5),
  method = "ward",
  quiet = FALSE
)

}
\references{
Jombart T, Devillard S and Balloux F (2010) Discriminant analysis
of principal components: a new method for the analysis of genetically
structured populations. BMC Genetics 11:94. doi:10.1186/1471-2156-11-94
}
