% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vpcstats.R
\name{binless}
\alias{binless}
\alias{binless.tidyvpcobj}
\title{Perform binless Visual Predictive Check (VPC)}
\usage{
binless(o, ...)

\method{binless}{tidyvpcobj}(
  o,
  optimize = TRUE,
  optimization.interval = c(0, 7),
  loess.ypc = FALSE,
  lambda = NULL,
  span = NULL,
  sp = NULL,
  ...
)
}
\arguments{
\item{o}{A \code{tidyvpcobj}.}

\item{...}{Other arguments to include will be ignored.}

\item{optimize}{Logical indicating whether smoothing parameters should be optimized using AIC.}

\item{optimization.interval}{Numeric vector of length 2 specifying the min/max range of smoothing parameter for optimization. Only applicable if \code{optimize = TRUE}.}

\item{loess.ypc}{Logical indicating LOESS prediction corrected VPC. Must first use \code{\link{predcorrect}}, if specifying \code{loess.ypc = TRUE}. Only applicable to continuous VPC.}

\item{lambda}{Numeric vector of length 3 specifying lambda values for each quantile. If stratified, specify a \code{data.frame} with given strata represented the column name, and value specified as a numeric vector of length 3.
See below examples. Only applicable to continuous VPC with \code{optimize = FALSE}.}

\item{span}{Numeric between 0,1 specifying smoothing parameter for LOESS prediction correction. Only applicable for continuous VPC with \code{loess.ypc = TRUE} and \code{optimize = FALSE}.}

\item{sp}{List of smoothing parameters applied to \code{mgcv::gam()}. Elements of list must be in the same order as unique values of DV. If one or more stratification variables present, the order of sp
should be specified as unique combination of strata + DV, in ascending order. See below examples. Only applicable for categorical VPC, if \code{optimize = FALSE}.}
}
\value{
For continuous VPC, updates \code{tidyvpcobj} with additive quantile regression fits for observed and simulated data for quantiles specified in the \code{qpred} argument of \code{vpcstats()}.
  If the \code{optimize = TRUE} argument is specified, the resulting \code{tidyvpcobj} will contain optimized lambda values according to AIC.  For prediction
  corrected VPC (pcVPC), specifying \code{loess.ypc = TRUE} will return optimized span value for LOESS smoothing. For categorical VPC, 
  updates \code{tidyvpcobj} with fits obtained by \code{gam(family="binomial")} for observed and simulated data for each category of DV (in each stratum if \code{stratify} defined).
  If \code{optimize = TRUE} argument is specified, the resulting \code{tidyvpcobj} wil contain optimized \code{sp} values according to AIC.
}
\description{
Use this function in place of traditional binning methods to derive VPC. For continuous
VPC, this is obtained using additive quantile regression (\code{quantreg::rqss()}) and LOESS for pcVPC. While for categorical
VPC, this is obtained using a generalized additive model (\code{gam(family = "binomial")}).
}
\examples{
\donttest{

require(magrittr)
require(data.table)

obs_data <- obs_data[MDV == 0]
sim_data <- sim_data[MDV == 0]


 vpc <- observed(obs_data, y = DV, x = TIME) \%>\%
      simulated(sim_data, y = DV) \%>\%
      binless() \%>\%
      vpcstats()
      
 # Binless example with LOESS prediction correction
 
 obs_data$PRED <- sim_data[REP == 1, PRED]
 
 vpc <- observed(obs_data, y = DV, x = TIME) \%>\%
      simulated(sim_data, y = DV) \%>\%
      predcorrect(pred = PRED) \%>\%
      binless(optimize = TRUE, loess.ypc = TRUE) \%>\%
      vpcstats()
      
# Binless example with user specified lambda values stratified on 
# "GENDER" with 2 levels ("M", "F"), 10\%, 50\%, 90\% quantiles.
 
 lambda_strat <- data.table(
 GENDER_M = c(3,5,2),
 GENDER_F = c(1,3,4)
 )
 
 vpc <- observed(obs_data, y = DV, x = TIME) \%>\%
      simulated(sim_data, y = DV) \%>\%
      stratify(~ GENDER) \%>\%
      binless(optimize = FALSE, lambda = lambda_strat) \%>\%
      vpcstats(qpred = c(0.1, 0.5, 0.9))
      
 # Binless example for categorical DV with optimized smoothing
 vpc <- observed(obs_cat_data, x = agemonths, yobs = zlencat) \%>\%
       simulated(sim_cat_data, ysim = DV) \%>\%
       stratify(~ Country_ID_code) \%>\%
       binless() \%>\%
       vpcstats(vpc.type = "cat", quantile.type = 6)
       
 # Binless example for categorical DV with user specified sp values
 user_sp <- list(
 Country1_prob0 = 100,
 Country1_prob1 = 3,
 Country1_prob2 = 4,
 Country2_prob0 = 90,
 Country2_prob1 = 3,
 Country2_prob2 = 4,
 Country3_prob0 = 55,
 Country3_prob1 = 3,
 Country3_prob2 = 200)
 
 vpc <- observed(obs_cat_data, x = agemonths, yobs = zlencat) \%>\%
        simulated(sim_cat_data, ysim = DV) \%>\%
        stratify(~ Country_ID_code) \%>\%
        binless(optimize = FALSE, sp = user_sp) \%>\%
        vpcstats(vpc.type = "categorical", conf.level = 0.9, quantile.type = 6)
}

}
\seealso{
\code{\link{observed}} \code{\link{simulated}} \code{\link{censoring}} \code{\link{predcorrect}} \code{\link{stratify}} \code{\link{binning}} \code{\link{vpcstats}}
}
