% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/relative_mortality.R
\name{rmm}
\alias{rmm}
\title{Relative Mortality Metric (RMM) Calculation}
\usage{
rmm(
  data,
  Ps_col,
  outcome_col,
  n_samples = 1000,
  Divisor1 = 5,
  Divisor2 = 5,
  Threshold_1 = 0.9,
  Threshold_2 = 0.99,
  pivot = FALSE
)
}
\arguments{
\item{data}{A data frame or tibble containing the data.}

\item{Ps_col}{The name of the column containing the survival probabilities
(Ps). Should be numeric (values between 0 and 100).}

\item{outcome_col}{The name of the column containing the outcome data. It
should be binary, with values indicating patient survival. A value of \code{1}
should represent "alive" (survived), while \code{0} should represent "dead" (did
not survive). Ensure the column contains only these two possible values.}

\item{n_samples}{A numeric value indicating the number of bootstrap samples
to take from the data source.}

\item{Divisor1}{A divisor used for binning the survival probabilities
(default is 5).}

\item{Divisor2}{A second divisor used for binning the survival probabilities
(default is 5).}

\item{Threshold_1}{The first threshold for dividing the survival
probabilities (default is 0.9).}

\item{Threshold_2}{The second threshold for dividing the survival
probabilities (default is 0.99).}

\item{pivot}{A logical indicating whether to return the results in a long
format (pivot = TRUE) or wide format (pivot = FALSE, default).}
}
\value{
A tibble containing the Relative Mortality Metric (RMM) and related
statistics:
\itemize{
\item \code{population_RMM_LL}: The lower bound of the 95\% confidence interval for the population RMM.
\item \code{population_RMM}: The final calculated Relative Mortality Metric for the population
existing in \code{data}.
\item \code{population_RMM_UL}: The upper bound of the 95\% confidence interval for the population RMM.
\item \code{population_CI}: The confidence interval width for the population RMM.
\item \code{bootstrap_RMM_LL}: The lower bound of the 95\% confidence interval for the bootstrap RMM.
\item \code{bootstrap_RMM}: The average RMM value calculated for the bootstrap sample.
\item \code{bootstrap_RMM_UL}: The upper bound of the 95\% confidence interval for the bootstrap RMM.
\item \code{bootstrap_CI}: The width of the 95\% confidence interval for the bootstrap RMM.
\item If \code{pivot = TRUE}, the results will be in long format with two columns: \code{stat}
and \code{value}, where each row corresponds to one of the calculated
statistics.
\item If \code{pivot = FALSE} (default), the results will be returned in wide format,
with each statistic as a separate column.
}
}
\description{
Calculates the Relative Mortality Metric (RMM) from Napoli et al. (2017)
based on patient survival probabilities (Ps) and actual outcomes. The
function groups patients into bins based on their survival probability scores
(Ps) and computes a weighted mortality metric along with confidence
intervals. For more information on the methods used in this function, see as
well Schroeder et al. (2019), and Kassar et al. (2016).

The Relative Mortality Metric (RMM) quantifies the performance of a center in
comparison to the anticipated mortality based on the TRISS national
benchmark. The RMM measures the difference between observed and expected
mortality, with a range from -1 to 1.
\itemize{
\item An RMM of 0 indicates that the observed mortality aligns with the expected
national benchmark across all acuity levels.
\item An RMM greater than 0 indicates better-than-expected performance, where
the center is outperforming the national benchmark.
\item An RMM less than 0 indicates under-performance, where the center’s observed
mortality is higher than the expected benchmark.
}

This metric helps assess how a center's mortality compares to the national
standards, guiding quality improvement efforts.  \code{rmm()} utilizes bootstrap
sampling to calculate the confidence intervals via the standard error method.

Due to the use of bootstrap sampling within the function, users should set
the random number seed before running \code{rmm()}.
}
\examples{
# Generate example data with high negative skewness
set.seed(10232015)

# Parameters
n_patients <- 1000  # Total number of patients

# Skewed towards higher values
Ps <- plogis(rnorm(n_patients, mean = 2, sd = 1.5))

# Simulate survival outcomes based on Ps
survival_outcomes <- rbinom(n_patients,
                            size = 1,
                            prob = Ps
                            )

# Create data frame
data <- data.frame(Ps = Ps, survival = survival_outcomes) |>
dplyr::mutate(death = dplyr::if_else(survival == 1, 0, 1))

# Example usage of the `rmm` function
rmm(data = data, Ps_col = Ps,
    outcome_col = survival,
    Divisor1 = 5,
    Divisor2 = 5,
    n_samples = 5
    )

# pivot!
rmm(data = data, Ps_col = Ps,
    outcome_col = survival,
    Divisor1 = 5,
    Divisor2 = 5,
    n_samples = 5,
    pivot = TRUE
    )

}
\author{
Nicolas Foss, Ed.D, MS, original paper and code in MATLAB by Nicholas
J. Napoli, Ph.D., MS
}
