% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ppdist.R
\name{ppdist}
\alias{ppdist}
\alias{ttdist}
\title{Compute Distance Between Two Point Patterns}
\usage{
ppdist(
  dmat,
  penalty = 1,
  type = c("tt", "rtt", "TT", "RTT"),
  ret_matching = FALSE,
  p = 1,
  precision = NULL
)
}
\arguments{
\item{dmat}{a matrix specifying in its \eqn{(i,j)}-th entry the distance from the
i-th point of \eqn{\xi}{xi} to the j-th point of \eqn{\eta}{eta}.}

\item{penalty}{a positive number. The penalty for adding/deleting points.}

\item{type}{either \code{"tt"}/\code{"TT"} for the transport-transform metric
or \code{"rtt"}/\code{"RTT"} for the relative transport-transform metric.}

\item{ret_matching}{logical. Shall the optimal point matching be returned?}

\item{p}{a number \eqn{>0}. The matching is chosen such that the \code{p}-th
order sum (\eqn{\ell_p}{l_p}-norm) is minimized.}

\item{precision}{a small positive integer value. The precisions of the computations, which
are currently performed in integers. After correcting for the penalty, \code{dmat^p}
is divided by its largest entry, multiplied by \code{10^precision} and rounded to
compute the optimal matching. The default value \code{NULL} chooses maximal
integer precision possible, which is \code{precision = 9} on almost all systems.}
}
\value{
The corresponding distance between the point patterns if \code{ret_matching}
        is \code{FALSE}. 
        
        Otherwise a list with components \code{dist} containing
        this distance and two vectors \code{target1, target2} of integers, where
        \code{target}\eqn{i} specifies the indices of the points in the other pattern
        that the points of the \eqn{i}-th pattern are matched to and \code{NA} every
        time a point is deleted.
        
        There may be a minus in front of an index, where
        \code{-j} indicates that the corresponding pairing with point \code{j}
        would be over a distance of more than \eqn{2^{1/p} \cdot 
        \code{penalty}}{2^{1/p} * penalty}. This is 
        equivalent to saying that the corresponding point of the first pattern
        is deleted and the \eqn{j}-th point of the second pattern is added. 
        
        Note that having more than one minus implies that the matching is non-unique.
}
\description{
Based on an arbitrary matrix of "distances" between the points of two point patterns
\eqn{\xi}{xi} and \eqn{\eta}{eta}, this function computes versions of the transport-transform
distance between \eqn{\xi}{xi} and \eqn{\eta}{eta}.
}
\details{
The transport-transform (TT) distance gives the minimal total cost for \dQuote{morphing}
         the pattern \eqn{\xi}{xi} into the pattern \eqn{\eta}{eta} by way of shifting points (at costs
         specified in \code{dmat}) and adding or deleting points (each at cost \code{penalty}).
         The total cost is determined as 
         \deqn{\biggl( \sum_{j=1}^n c_j^p \biggr)^{1/p},}{(sum_{j=1}^n c_j^p)^{1/p},}
         where \eqn{c_j} denotes the cost for the \eqn{j}th individual operation and \eqn{n} is
         the cardinality of the larger point pattern.  

         The relative transport-transform (RTT) metric is exactly the same, but the sum in the
         total cost is divided by the larger cardinality:
         \deqn{\biggl( \frac{1}{n}\sum_{j=1}^n c_j^p \biggr)^{1/p}.}{(1/n * sum_{j=1}^n c_j^p)^{1/p}.}

         The TT- and RTT-metrics form an umbrella concept that    
         includes the OSPA and Spike Time metrics frequently used in the literature.
         See Müller, Schuhmacher and Mateu (2019) for details.
}
\examples{
  # small example
  # -------------
  set.seed(181230)
  xi <- spatstat.core::rpoispp(20)
  eta <- spatstat.core::rpoispp(20)
  dmat <- spatstat.geom::crossdist(xi,eta)
  res <- ppdist(dmat, penalty=1,  type="rtt", ret_matching=TRUE, p=1)
  plotmatch(xi, eta, dmat, res, penalty=1, p=1)
  res$dist

  # for comparison: ospa-distance computation from spatstat:
  res_ospa <- spatstat.geom::pppdist(xi,eta,"spa")
  res_ospa$distance  # exactly the same as above because nothing gets cut off 


  # same example, but with a much smaller penalty for adding/deleting points
  # --------------------------------------------------------------- 
  res <- ppdist(dmat, penalty=0.1,  type="rtt", ret_matching=TRUE, p=1)
  plotmatch(xi, eta, dmat, res, penalty=0.1, p=1)
    # dashed lines indicate points that are deleted and re-added at new position
    # grey segments on dashed lines indicate the cost of deletion plus re-addition
  res$dist
  
  # for comparison: ospa-distance computation from spatstat
  # (if things do get cut off, we have to ensure that the cutoff distances
  # are the same, thus cutoff = 2^(1/p) * penalty):
  res_ospa <- spatstat.geom::pppdist(xi,eta,"spa",cutoff=0.2)
  res_ospa$distance  # NOT the same as above
  res_ospa$distance - abs(xi$n-eta$n) * 0.1 / max(xi$n,eta$n)  # the same as above
  
  
  # a larger example
  # --------------------------------------------------------------- 
  set.seed(190203)
  xi <- spatstat.core::rpoispp(2000)
  eta <- spatstat.core::rpoispp(2000)
  dmat <- spatstat.geom::crossdist(xi,eta)
  res <- ppdist(dmat, penalty = 0.1,  type = "rtt", ret_matching = TRUE, p = 1)
  res$dist
  # takes about 2-3 seconds

}
\references{
Raoul Müller, Dominic Schuhmacher and Jorge Mateu (2020).\cr
            Metrics and Barycenters for Point Pattern Data.\cr
            Statistics and Computing 30, 953-972.\cr
            \doi{10.1007/s11222-020-09932-y}
}
\author{
Dominic Schuhmacher \email{schuhmacher@math.uni-goettingen.de}
}
