#' Family object for the unifed distribution
#'
#' @param link a specification for the model link function.  This can
#'     be a name/expression, a literal character string, a length-one
#'     character vector or an object of class ‘"link-glm"’ (such as
#'     generated by ‘make.link’) provided it is not specified via one
#'     of the accepted names. The \code{unifed} family accepts the
#'     links (as names) 'canonical', 'logit', 'probit', 'cloglog' and
#'     'cauchit'.
#' 
#' @param ... Optional \code{tol} and \code{maxit} arguments for
#'     \code{\link{unifed.unit.deviance}}.
#'
#'
#' @details
#'  The link 'canonical' is not part of the standard names accepted by
#'  \code{make.link()} from the stats package. It corresponds to the
#'  canonical link function for the unifed distribution, which is the
#'  inverse of the derivative of its cumulant generator. There is no
#'  explicit formula for it. The function
#'  \code{\link{unifed.kappa.prime.inverse}()} implements it using the
#'  Newthon-Raphson method.
#' 
#' 
#' @return \code{unifed} returns a family object for using the unifed
#'     distribution with the \code{glm} function.
#' 
#' @seealso \code{Gamma} \code{\link{unifed.kappa.prime.inverse}}
#'
#' @references{
#'
#' Jørgensen, Bent (1992). The Theory of Exponential Dispersion Models and Analysis  of  Deviance.
#' Instituto de Matemática Pura e Aplicada, (IMPA), Brazil.
#' 
#' }
#' 
#' @importFrom stats make.link runif
#' @export
unifed <- function(link="logit",...){

    linktemp <- substitute(link)
    if (!is.character(linktemp)) 
        linktemp <- deparse(linktemp)
    okLinks <- c("logit", "probit", "cloglog", "cauchit","canonical")
    if (linktemp %in% okLinks) 
        stats <- make.link(linktemp)
    else if (is.character(link)) {
        stats <- make.link(link)
        linktemp <- link
    }
    else {
        if (inherits(link, "link-glm")) {
            stats <- link
            if (!is.null(stats$name)) 
                linktemp <- stats$name
        }
        else {
            stop(gettextf("link \"%s\" not available for unifed family; available links are %s", 
                          linktemp, paste(sQuote(okLinks), collapse = ", ")), 
                 domain = NA)
        }
    }

    variance <- unifed.varf

    validmu <- function(mu) all(is.finite(mu)) && all(mu > 0 & 
                                                      mu < 1)

    dev.resids <- function(y,mu,wt){        
        wt * mapply(function(x,y){unifed.unit.deviance(x,y,...)},y,mu)
    }

    initialize <- expression({
        if (NCOL(y) == 1) {
            if (is.factor(y)) y <- y != levels(y)[1L]
            n <- rep.int(1, nobs)
            y[weights == 0] <- 0
            if (any(y < 0 | y > 1)) stop("y values must be 0 <= y <= 1")
            mustart <- (weights * y + 0.5)/(weights + 1)
            m <- weights * y
        }  else stop("for the 'unifed' family, y must be a vector with values between 0 and 1.")
    })

    
    aic <- function(y, n, mu, weights, dev){NA}
    
    structure(list(family = "unifed", link = linktemp, linkfun = stats$linkfun, 
                   linkinv = stats$linkinv, variance = variance, dev.resids = dev.resids, aic=aic
                   , mu.eta = stats$mu.eta, initialize = initialize, 
        validmu = validmu, valideta = stats$valideta), 
        class = "family")
    
}



#' @rdname unifed
#' @name quasiunifed
#'
#' 
#'
#' @return The \code{quasiunifed} family differs from the
#'     \code{unifed} only in that the dispersion parameter is not
#'     fixed to one.
#'
#' @references{
#'
#' Wedderburn, R. W. M. (1974). Quasi-likelihood functions, generalized linear models, and the Gauss—Newton method. Biometrika. \bold{61} (3): 439–447.
#'
#' McCullagh, Peter; Nelder, John (1989). Generalized Linear Models (second ed.). London: Chapman and Hall.
#' 
#' }
#' 
#' @export
quasiunifed <- function(link="logit",...){

    linktemp <- substitute(link)
    if (!is.character(linktemp)) 
        linktemp <- deparse(linktemp)
    okLinks <- c("logit", "probit", "cloglog", "cauchit","canonical")
    if (linktemp %in% okLinks) 
        stats <- make.link(linktemp)
    else if (is.character(link)) {
        stats <- make.link(link)
        linktemp <- link
    }
    else {
        if (inherits(link, "link-glm")) {
            stats <- link
            if (!is.null(stats$name)) 
                linktemp <- stats$name
        }
        else {
            stop(gettextf("link \"%s\" not available for unifed family; available links are %s", 
                          linktemp, paste(sQuote(okLinks), collapse = ", ")), 
                 domain = NA)
        }
    }

    variance <- unifed.varf

    validmu <- function(mu) all(is.finite(mu)) && all(mu > 0 & 
                                                      mu < 1)

    dev.resids <- function(y,mu,wt){        
        wt * mapply(function(x,y){unifed.unit.deviance(x,y,...)},y,mu)
    }

    initialize <- expression({
        if (NCOL(y) == 1) {
            if (is.factor(y)) y <- y != levels(y)[1L]
            n <- rep.int(1, nobs)
            y[weights == 0] <- 0
            if (any(y < 0 | y > 1)) stop("y values must be 0 <= y <= 1")
            mustart <- (weights * y + 0.5)/(weights + 1)
            m <- weights * y
        }  else stop("for the 'unifed' family, y must be a vector with values between 0 and 1.")
    })

    
    aic <- function(y, n, mu, weights, dev){NA}
    
    structure(list(family = "quasiunifed", link = linktemp, linkfun = stats$linkfun, 
                   linkinv = stats$linkinv, variance = variance, dev.resids = dev.resids, aic=aic
                   , mu.eta = stats$mu.eta, initialize = initialize, 
        validmu = validmu, valideta = stats$valideta), 
        class = "family")
    
}

#' @rdname unifed
#' @name unifed.canonical.link
#'
#' This function is used inside of \code{unifed()} when the link
#' parameter is set to \code{"canonical"}. It returns the link
#' function, inverse link function, the derivative dmu/deta and a
#' function for domain checking for the unifed distribution canonical
#' link.
#'
#' @return An object of class \code{"link-glm"}.
#'
#' @seealso \code{make.link}
#' 
#' @export
unifed.canonical.link <- function(){
    structure(list(
        linkfun=unifed.kappa.prime.inverse,
        linkinv=unifed.kappa.prime,
        mu.eta=unifed.kappa.double.prime,
        valideta=function(eta) TRUE
    ),
    class="link-glm")    
}


make.link <- function(link){
    if(link=="canonical")
        unifed.canonical.link()
    else
        stats::make.link(link)
}


##  LocalWords:  logit cauchit cloglog probit param unifed cumulant
