#' Create flanking intervals from input intervals.
#'
#' @param x [tbl_interval()]
#' @param genome [tbl_genome()]
#' @param both number of bases on both sizes
#' @param left number of bases on left side
#' @param right number of bases on right side
#' @param fraction define flanks based on fraction of interval length
#' @param strand define `left` and `right` based on strand
#' @param trim adjust coordinates for out-of-bounds intervals
#' @param ... extra arguments (not used)
#'
#' @return [tbl_interval()]
#'
#' @family single set operations
#' @seealso
#'   \url{http://bedtools.readthedocs.org/en/latest/content/tools/flank.html}
#'
#' @examples
#'
#' x <- trbl_interval(
#'   ~chrom, ~start, ~end,
#'   'chr1',      25,      50,
#'   'chr1',      100,     125
#' )
#'
#' genome <- trbl_genome(
#'   ~chrom, ~size,
#'   'chr1', 130
#' )
#'
#' bed_glyph(bed_flank(x, genome, both = 20))
#'
#' x <- trbl_interval(
#'  ~chrom, ~start, ~end, ~name, ~score, ~strand,
#'  "chr1", 500,    1000, '.',   '.',    '+',
#'  "chr1", 1000,   1500, '.',   '.',    '-'
#' )
#'
#' genome <- trbl_genome(
#'   ~chrom, ~size,
#'   "chr1", 5000
#' )
#'
#' bed_flank(x, genome, left = 100)
#'
#' bed_flank(x, genome, right = 100)
#'
#' bed_flank(x, genome, both = 100)
#'
#' bed_flank(x, genome, both = 0.5, fraction = TRUE)
#'
#' @export
bed_flank <- function(x, genome, both = 0, left = 0,
                      right = 0, fraction = FALSE,
                      strand = FALSE, trim = FALSE, ...) {

  if (!is.tbl_interval(x)) x <- tbl_interval(x)
  if (!is.tbl_genome(genome)) genome <- tbl_genome(genome)

  if (!any(c(both, left, right) > 0))
    stop('specify one of both, left, right', call. = FALSE)

  if (strand && !'strand' %in% colnames(x))
    stop('expected `strand` column in `x`', call. = FALSE)

  if (both != 0 && (left != 0 || right != 0))
    stop('ambiguous side spec for bed_flank', call. = FALSE)

  if (both) left <- right <- both

  if (strand) {
    if (fraction) {
      res <- mutate(x, .size = end - start,
               left_start = ifelse(strand == '+',
                                start - round( left * .size ),
                                end),
               left_end = ifelse(strand == '+',
                              start,
                              end + round( left * .size )),
               right_start = ifelse(strand == '+',
                                end,
                                start - round( right * .size )),
               right_end = ifelse(strand == '+',
                              end + round( right * .size ),
                              start))

     res <- select(res, -start, -end, -.size)

    } else {
      res <- mutate(x, left_start = ifelse(strand == '+',
                                start - left,
                                end),
               left_end = ifelse(strand == '+',
                              start,
                              end + left),
               right_start = ifelse(strand == '+',
                                end,
                                start - right),
               right_end = ifelse(strand == '+',
                              end + right,
                              start))
      res <- select(res, -start, -end)
    }

  } else {
    if (fraction) {
      res <- mutate(x, .size = end - start,
               left_start =  start - round( left * .size ),
               left_end = start,
               right_start = end,
               right_end = end + round( right * .size ))
      res <- select(res, -start, -end, -.size)

    } else {
      res <- mutate(x, left_start = start - left,
               left_end = start,
               right_start = end,
               right_end = end + right)
      res <- select(res, -start, -end)
    }
  }

  if (right && !left) {
    res <- mutate(res,
                  start = right_start,
                  end = right_end)
    res <- select(res, chrom, start, end, everything(),
                  -left_start, -left_end, -right_start, -right_end)

  } else if (left && !right) {
    res <- mutate(res,
                  start = left_start,
                  end = left_end)
    res <- select(res, chrom, start, end, everything(),
                  -left_start, -left_end, -right_start, -right_end)

  } else {
    res_left <- select(res, chrom,  left_start,   left_end, everything(), -right_start, -right_end)
    res_right <- select(res, chrom,  right_start,   right_end, everything(), -left_start, -left_end)
    res_left <- rename(res_left, start = left_start, end = left_end)
    res_right<- rename(res_right, start = right_start, end = right_end)
    res <- bind_rows(res_left, res_right)
  }

  res <- bound_intervals(res, genome, trim)
  res <- arrange(res, chrom, start)

  res
}

