% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/match_gps.R
\name{match_gps}
\alias{match_gps}
\title{Match the data based on generalized propensity score}
\usage{
match_gps(
  csmatrix = NULL,
  method = "nnm",
  caliper = 0.2,
  reference = NULL,
  ratio = NULL,
  replace = NULL,
  order = NULL,
  ties = NULL,
  min_controls = NULL,
  max_controls = NULL,
  kmeans_args = NULL,
  kmeans_cluster = 5,
  verbose_output = FALSE,
  ...
)
}
\arguments{
\item{csmatrix}{An object of class \code{gps} and/or \code{csr} representing a data
frame of generalized propensity scores. The first column must be the
treatment variable, with additional attributes describing the calculation
of the common support region and the estimation of generalized propensity
scores. It is crucial that the common support region was calculated using
the \code{csregion()} function to ensure compatibility.}

\item{method}{A single string specifying the matching method to use. The
default is \code{"nnm"}, which applies the k-nearest neighbors matching
algorithm. See the Details section for a full list of available methods.}

\item{caliper}{A numeric value specifying the caliper width, which defines
the allowable range within which observations can be matched. It is
expressed as a percentage of the standard deviation of the
logit-transformed generalized propensity scores. To perform matching
without a caliper, set this parameter to a very large value. For exact
matching, set \code{caliper = 0} and enable the \code{exact} option by setting it to
\code{TRUE}.}

\item{reference}{A single string specifying the exact level of the treatment
variable to be used as the reference in the matching process. All other
treatment levels will be matched to this reference level. Ideally, this
should be the control level. If no natural control is present, avoid
selecting a level with extremely low or high covariate or propensity score
values. Instead, choose a level with covariate or propensity score
distributions that are centrally positioned among all treatment groups to
maximize the number of matches.}

\item{ratio}{A scalar for the number of matches which should be found for
each control observation. The default is one-to-one matching. Only
available for the methods \code{"nnm"} and \code{"pairopt"}.}

\item{replace}{Logical value indicating whether matching should be done with
replacement. If \code{FALSE}, the order of matches generally matters. Matches
are found in the same order as the data is sorted. Specifically, the
matches for the first observation will be found first, followed by those
for the second observation, and so on. Matching without replacement is
generally not recommended as it tends to increase bias. However, in cases
where the dataset is large and there are many potential matches, setting
\code{replace = FALSE} often results in a substantial speedup with negligible or
no bias. Only available for the method \code{"nnm"}}

\item{order}{A string specifying the order in which logit-transformed GPS
values are sorted before matching. The available options are:
\itemize{
\item \code{"desc"} – sorts GPS values from highest to lowest (default).
\item \code{"asc"} – sorts GPS values from lowest to highest.
\item \code{"original"} – preserves the original order of GPS values.
\item \code{"random"} – randomly shuffles GPS values. To generate different random
orders, set a seed using \code{\link[=set.seed]{set.seed()}}.
}}

\item{ties}{A logical flag indicating how tied matches should be handled.
Available only for the \code{"nnm"} method, with a default value of \code{FALSE} (all
tied matches are included in the final dataset, but only unique
observations are retained). For more details, see the \code{ties} argument in
\code{\link[Matching:Matchby]{Matching::Matchby()}}.}

\item{min_controls}{The minimum number of treatment observations that should
be matched to each control observation. Available only for the \code{"fullopt"}
method. For more details, see the \code{min.controls} argument in
\code{\link[optmatch:fullmatch]{optmatch::fullmatch()}}.}

\item{max_controls}{The maximum number of treatment observations that can be
matched to each control observation. Available only for the \code{"fullopt"}
method. For more details, see the \code{max.controls} argument in
\code{\link[optmatch:fullmatch]{optmatch::fullmatch()}}.}

\item{kmeans_args}{A list of arguments to pass to \link[stats:kmeans]{stats::kmeans}. These
arguments must be provided inside a \code{list()} in the paired \code{name = value}
format.}

\item{kmeans_cluster}{An integer specifying the number of clusters to pass to
\link[stats:kmeans]{stats::kmeans}.}

\item{verbose_output}{a logical flag. If \code{TRUE} a more verbose version of the
function is run and the output is printed out to the console.}

\item{...}{Additional arguments to be passed to the matching
function.}
}
\value{
A \code{data.frame} similar to the one provided as the \code{data} argument in
the \code{\link[=estimate_gps]{estimate_gps()}} function, containing the same columns but only the
observations for which a match was found. The returned object includes two
attributes, accessible with the \code{attr()} function:
\itemize{
\item \code{original_data}: A \code{data.frame} with the original data returned by the
\code{\link[=csregion]{csregion()}} or \code{\link[=estimate_gps]{estimate_gps()}} function, after the estimation of the csr
and filtering out observations not within the csr.
\item \code{matching_filter}: A logical vector indicating which rows from
\code{original_data} were included in the final matched dataset.
}
}
\description{
The \code{match_gps()} function performs sample matching based on
generalized propensity scores (GPS). It utilizes the k-means clustering
algorithm to partition the data into clusters and subsequently matches all
treatment groups within these clusters. This approach ensures efficient and
structured comparisons across treatment levels while accounting for the
propensity score distribution.
}
\details{
Propensity score matching can be performed using various matching
algorithms. Lopez and Gutman (2017) do not explicitly specify the matching
algorithm used, but it is assumed they applied the commonly used k-nearest
neighbors matching algorithm, implemented as \code{method = "nnm"}. However,
this algorithm can sometimes be challenging to use, especially when
treatment and control groups have unequal sizes. When \code{replace = FALSE},
the number of matches is strictly limited by the smaller group, and even
with \code{replace = TRUE}, the results may not always be satisfactory. To
address these limitations, we have implemented an additional matching
algorithm to maximize the number of matched observations within a dataset.

The available matching methods are:
\itemize{
\item \code{"nnm"} – classic k-nearest neighbors matching, implemented using
\code{\link[Matching:Matchby]{Matching::Matchby()}}. The tunable parameters in \code{match_gps()} are
\code{caliper}, \code{ratio}, \code{replace}, \code{order}, and \code{ties}. Additional arguments
can be passed to \code{\link[Matching:Matchby]{Matching::Matchby()}} via the \code{...} argument.
\item \code{"fullopt"} – optimal full matching algorithm, implemented with
\code{\link[optmatch:fullmatch]{optmatch::fullmatch()}}. This method calculates a discrepancy matrix to
identify all possible matches, often optimizing the percentage of matched
observations. The available tuning parameters are \code{caliper},
\code{min_controls}, and \code{max_controls}.
\item \code{"pairmatch"} – optimal 1:1 and 1:k matching algorithm, implemented using
\code{\link[optmatch:pairmatch]{optmatch::pairmatch()}}, which is actually a wrapper around
\code{\link[optmatch:fullmatch]{optmatch::fullmatch()}}. Like \code{"fullopt"}, this method calculates a
discrepancy matrix and finds matches that minimize its sum. The available
tuning parameters are \code{caliper} and \code{ratio}.
}
}
\examples{
# Defining the formula used for gps estimation
formula_cancer <- formula(status ~ age + sex)

# Step 1.) Estimation of the generalized propensity scores
gp_scores <- estimate_gps(formula_cancer,
  data = cancer,
  method = "multinom",
  reference = "control",
  verbose_output = TRUE
)

# Step 2.) Defining the common support region
gps_csr <- csregion(gp_scores)

# Step 3.) Matching the gps
matched_cancer <- match_gps(gps_csr,
  caliper = 0.25,
  reference = "control",
  method = "fullopt",
  kmeans_cluster = 2,
  kmeans_args = list(
    iter.max = 200,
    algorithm = "Forgy"
  ),
  verbose_output = TRUE
)

}
\references{
Michael J. Lopez, Roee Gutman "Estimation of Causal Effects with
Multiple Treatments: A Review and New Ideas," Statistical Science, Statist.
Sci. 32(3), 432-454, (August 2017)
}
\seealso{
\code{\link[=estimate_gps]{estimate_gps()}} for the calculation of generalized propensity
scores; \code{\link[MatchIt:matchit]{MatchIt::matchit()}}, \code{\link[optmatch:fullmatch]{optmatch::fullmatch()}} and
\code{\link[optmatch:pairmatch]{optmatch::pairmatch()}} for the documentation of the matching functions;
\code{\link[stats:kmeans]{stats::kmeans()}} for the documentation of the k-Means algorithm.
}
