% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/racusum_crit.R
\name{racusum_crit}
\alias{racusum_crit}
\alias{racusum_crit_mc}
\alias{racusum_crit_sim}
\title{Alarm thresholds of RA-CUSUM charts}
\usage{
racusum_crit_mc(L0, pmix, RA, RQ, scaling = 600, rounding = "p",
  method = "Toep", jmax = 4, verbose = FALSE)

racusum_crit_sim(L0, pmix, RA = 2, RQ = 1, yemp = FALSE, m = 10000,
  nc = 1, hmax = 30, jmax = 4, verbose = FALSE)
}
\arguments{
\item{L0}{Double. Prespecified Average Run Length.}

\item{pmix}{Numeric Matrix. A three column matrix. First column is the risk
score distribution. Second column are the predicted probabilities from the risk model. Third
 column can be either the predicted probabilities from the risk model or average outcome per
 risk score, see examples.}

\item{RA}{Double. Odds ratio of death under the alternative hypotheses. Detecting deterioration
in performance with increased mortality risk by doubling the odds Ratio \code{RA = 2}. Detecting
 improvement in performance with decreased mortality risk by halving the odds ratio of death
 \code{RA = 1/2}. Odds ratio of death under the null hypotheses is \code{1}.
\code{RQ}. Use \code{RQ = 1} to compute the in-control ARL and other values to compute the
out-of-control ARL.}

\item{RQ}{Double. Defines the true performance of a surgeon with the odds ratio ratio of death
\code{RQ}. Use \code{RQ = 1} to compute the in-control ARL and other values to compute the
out-of-control ARL.}

\item{scaling}{Double. The \code{scaling} parameter controls the quality of the approximation,
larger values achieve higher accuracy but increase the computation burden (larger transition
probability matrix).}

\item{rounding}{Character. If \code{rounding = "p"} a paired rounding implementation of
\emph{Knoth et al. (2019)} is used, if \code{rounding = "s"} a simple rounding method of
\emph{Steiner et al. (2000)} is used.}

\item{method}{Character. If \code{method = "Toep"} a combination of Sequential Probability Ratio
Test and Toeplitz matrix structure is used to calculate the ARL. \code{"ToepInv"} computes the
inverted matrix using Toeplitz matrix structure. \code{"BE"} solves a linear equation system
using the classical approach of \emph{Brook and Evans (1972)} to calculate the ARL.}

\item{jmax}{Integer. Number of digits for grid search.}

\item{verbose}{Logical. If \code{FALSE} a quiet calculation of \code{h} is done. If \code{TRUE}
verbose output of the search procedure is included.}

\item{yemp}{Logical. If \code{TRUE}, use emirical outcome values, else use model.}

\item{m}{Integer. Number of simulation runs.}

\item{nc}{Integer. Number of cores used for parallel processing. Value is passed to
\code{\link{parSapply}}.}

\item{hmax}{Integer. Maximum value of \code{h} for the grid search.}
}
\value{
Returns a single value which is the control limit \code{h} for a given In-control ARL.
}
\description{
Compute alarm threshold of risk-adjusted CUSUM charts.
}
\details{
Determines the control limit for given in-control ARL (\code{"L0"}) using
\code{\link{racusum_arl_mc}} by applying a grid search.

Determines the control limit ("\code{h}") for given in-control ARL (\code{"L0"})
applying a grid search using \code{\link{racusum_arl_sim}} and \code{\link{parSapply}}.
}
\examples{
\dontrun{
library(vlad)
library(dplyr)
data("cardiacsurgery", package = "spcadjust")

## Markov Chain
## preprocess data to 30 day mortality and subset phase I (In-control) of surgeons 2
S2I <- cardiacsurgery \%>\% rename(s = Parsonnet) \%>\%
  mutate(y = ifelse(status == 1 & time <= 30, 1, 0),
         phase = factor(ifelse(date < 2*365, "I", "II"))) \%>\%
  filter(phase == "I", surgeon == 2) \%>\% select(s, y)

## estimate risk model, get relative frequences and probabilities
mod1 <- glm(y ~ s, data = S2I, family = "binomial")
fi  <- as.numeric(table(S2I$s) / length(S2I$s))
usi <- sort(unique(S2I$s))
pi1 <- predict(mod1, newdata = data.frame(s = usi), type = "response")

## set up patient mix
pmix  <- data.frame(fi, pi1, pi1)

## control limit for detecting deterioration RA = 2:
racusum_crit_mc(pmix = pmix, L0 = 740, RA = 2, RQ = 1)
## control limit for detecting improvement RA = 1/2:
racusum_crit_mc(pmix = pmix, L0 = 740, RA = 0.5, RQ = 1)

## Monte Carlo simulation
SALL <- cardiacsurgery \%>\% rename(s = Parsonnet) \%>\%
  mutate(y = ifelse(status == 1 & time <= 30, 1, 0),
         phase = factor(ifelse(date < 2*365, "I", "II")))
SI <- subset(SALL, phase == "I")
y <- subset(SALL, select = y)
GLM <- glm(y ~ s, data = SI, family = "binomial")
pi1 <- predict(GLM, type = "response", newdata = data.frame(s = SALL$s))
pmix <- data.frame(y, pi1, pi1)
h <- racusum_crit_sim(pmix = pmix, L0 = 370, RA = 2, nc = 4, verbose = TRUE)
}
}
\references{
Knoth S, Wittenberg P and Gan FF (2019).
Risk-adjusted CUSUM charts under model error.
\emph{Statistics in Medicine}, \strong{38}(12), pp. 2206--2218.

Wittenberg P, Gan FF, Knoth S (2018).
A simple signaling rule for variable life-adjusted display derived from
an equivalent risk-adjusted CUSUM chart.
\emph{Statistics in Medicine}, \strong{37}(16), pp 2455--2473.

Steiner SH, Cook RJ, Farewell VT and Treasure T (2000).
 Monitoring surgical performance using risk-adjusted cumulative sum charts.
 \emph{Biostatistics}, \strong{1}(4), pp. 441--452.

Brook D and Evans DA (1972)
 An approach to the probability distribution of CUSUM run length.
 \emph{Biometrika}, \strong{59}(3), pp. 539--549

Webster RA and Pettitt AN (2007)
Stability of approximations of average run length of risk-adjusted CUSUM schemes using
the Markov approach: comparing two methods of calculating transition probabilities.
 \emph{Communications in Statistics - Simulation and Computation} \strong{36}(3), pp. 471--482
}
\author{
Philipp Wittenberg
}
