% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/walker.R
\name{walker}
\alias{walker}
\title{Fully Bayesian linear regression with time-varying coefficients}
\usage{
walker(formula, data, beta_prior, sigma_prior, init, chains, newdata,
  naive = FALSE, return_x_reg = FALSE, return_y_rep = TRUE, ...)
}
\arguments{
\item{formula}{An object of class \code{\link[stats]{formula}}. See \code{\link[stats]{lm}} for details.}

\item{data}{An optional data.frame or object coercible to such, as in \code{\link[stats]{lm}}.}

\item{beta_prior}{A matrix with \eqn{k} rows and 2 columns, where first columns defines the 
prior means of the Gaussian priors of the corresponding \eqn{k} regression coefficients, 
and the second column defines the the standard deviations of those prior distributions.}

\item{sigma_prior}{A matrix with \eqn{k + 1} rows and two colums with similar structure as 
\code{beta_prior}, with first row corresponding to the prior of the standard deviation of the 
observation level noise, and rest of the rows define the priors for the standard deviations of 
random walk noise terms. The prior distributions for all sigmas are 
Gaussians truncated to positive real axis.}

\item{init}{Initial value specification, see \code{\link[rstan]{sampling}}. 
Note that compared to default in \code{rstan}, here the default is a to sample from the priors.}

\item{chains}{Number of Markov chains. Default is 4.}

\item{newdata}{Optional data.frame containing covariates used for prediction. This argument is 
ignored if argument \code{naive} is \code{TRUE}.}

\item{naive}{If \code{TRUE}, use "standard" approach which samples the joint posterior 
\eqn{p(beta, sigma | y)}. If \code{FALSE} (the default), use marginalisation approach 
where we sample the marginal posterior \eqn{p(sigma | y)} and generate the samples of 
\eqn{p(beta | sigma, y)} using state space modelling techniques 
(namely simulation smoother by Durbin and Koopman (2002)). Both methods give asymptotically 
identical results, but the latter approach is computationally much more efficient.}

\item{return_x_reg}{If \code{TRUE}, does not perform sampling, but instead returns the matrix of 
predictors after processing the \code{formula}.}

\item{return_y_rep}{If \code{TRUE} (default), \code{walker} also returns the samples from the 
posterior predictive distribution \eqn{p(y_rep | y)}. This argument is ignored if 
argument \code{naive} is \code{TRUE}.}

\item{...}{Further arguments to \code{\link[rstan]{sampling}}.}
}
\description{
Function \code{walker} performs Bayesian inference of a linear 
regression model with time-varying regression coefficients, 
i.e. ordinary regression model where instead of constant coefficients the 
coefficients follow random walks. All computations are done using Hamiltonian 
Monte Carlo provided by Stan, using a state space representation of the model 
in order to marginalise over the coefficients for efficient sampling.
}
\examples{
y <- window(log10(UKgas), end = time(UKgas)[100])
trend <- 1:length(y)
cos_t <- cos(2 * pi * trend /4)
sin_t <- sin(2 * pi * trend /4)
dat <- data.frame(y, trend, cos_t, sin_t)
future <- length(y) + 1:8
new_data <- data.frame(trend = future, cos_t = cos(2 * pi * future / 4), 
  sin_t = sin(2 * pi * future / 4))
fit <- walker(y ~ trend + cos_t + sin_t, data = dat, chains = 1, iter = 500, 
  newdata = new_data, beta = cbind(0, rep(10, 4)), sigma = cbind(0, rep(10, 5)))
print(fit, pars = c("sigma_y", "sigma_b"))
mean_fit <- matrix(summary(fit, "beta")$summary[, "mean"], ncol = 4)

# still needs bit manual work..  
ts.plot(cbind(y, rowSums(mean_fit * cbind(1, as.matrix(dat[, -1])))),
  col = 1:2, lwd = 2:1)
intervals <- summary(fit, pars = "y_new")$summary[, c("mean", "2.5\%", "97.5\%")]
ts.plot(log10(UKgas), ts(intervals, start = end(y) + c(0,1), frequency = 4),
  col = c(1, 2, 2, 2), lty = c(1, 1, 2, 2))
 
\dontrun{
## Comparing the approaches, note that with such a small data 
## the differences aren't huge, but try same with n = 500 and/or more terms...
set.seed(123)
n <- 100
beta1 <- cumsum(c(0.5, rnorm(n - 1, 0, sd = 0.05)))
beta2 <- cumsum(c(-1, rnorm(n - 1, 0, sd = 0.15)))
x1 <- rnorm(n, 1)
x2 <- 0.25 * cos(1:n)
ts.plot(cbind(beta1 * x1, beta2 *x2), col = 1:2)
u <- cumsum(rnorm(n))
y <- rnorm(n, u + beta1 * x1 + beta2 * x2)
ts.plot(y)
lines(u + beta1 * x1 + beta2 * x2, col = 2)
kalman_walker <- walker(y ~ x1 + x2, iter = 2000, chains = 1,
  beta_prior = cbind(0, rep(2, 3)), sigma_prior = cbind(0, rep(2, 4)))
print(kalman_walker, pars = c("sigma_y", "sigma_b"))
betas <- extract(kalman_walker, "beta")[[1]]
ts.plot(cbind(u, beta1, beta2, apply(betas, 2, colMeans)), 
  col = 1:3, lty = rep(2:1, each = 3))
sum(get_elapsed_time(kalman_walker))
naive_walker <- walker(y ~ x1 + x2, iter = 2000, chains = 1, 
  beta_prior = cbind(0, rep(2, 3)), sigma_prior = cbind(0, rep(2, 4)), 
  naive = TRUE)
print(naive_walker, pars = c("sigma_y", "sigma_b"))
# check rstan:::throw_sampler_warnings(naive_walker) 
# (does not work automatically for single chain)
sum(get_elapsed_time(naive_walker))

## Larger problem, this takes some time with naive approach

set.seed(123)
n <- 500
beta1 <- cumsum(c(1.5, rnorm(n - 1, 0, sd = 0.05)))
beta2 <- cumsum(c(-1, rnorm(n - 1, 0, sd = 0.5)))
beta3 <- cumsum(c(-1.5, rnorm(n - 1, 0, sd = 0.15)))
beta4 <- 2
x1 <- rnorm(n, 1)
x2 <- 0.25 * cos(1:n)
x3 <- 1:n/10
x4 <- runif(n, 1, 3)
ts.plot(cbind(beta1 * x1, beta2 * x2, beta3 * x3, beta4 * x4), col = 1:4)
u <- cumsum(rnorm(n))
signal <- u + beta1 * x1 + beta2 * x2 + beta3 * x3 + beta4 * x4
y <- rnorm(n, signal)
ts.plot(y)
lines(signal, col = 2)
kalman_walker <- walker(y ~ x1 + x2 + x3 + x4, iter = 2000, chains = 1,
  beta_prior = cbind(0, rep(2, 5)), sigma_prior = cbind(0, rep(2, 6)))
print(kalman_walker, pars = c("sigma_y", "sigma_b"))
betas <- extract(kalman_walker, "beta")[[1]]
ts.plot(cbind(u, beta1, beta2, apply(betas, 2, colMeans)), 
  col = 1:3, lty = rep(2:1, each = 3))
sum(get_elapsed_time(kalman_walker))
# need to increase adapt_delta in order to get rid of divergences
# and max_treedepth to get rid of related warnings
# and still we end up with low BFMI warning after hours of computation
naive_walker <- walker(y ~ x1 + x2 + x3 + x4, iter = 2000, chains = 1, 
  beta_prior = cbind(0, rep(2, 5)), sigma_prior = cbind(0, rep(2, 6)),
  naive = TRUE, control = list(adapt_delta = 0.9, max_treedepth = 15)) 
print(naive_walker, pars = c("sigma_y", "sigma_b"))
# check rstan:::throw_sampler_warnings(naive_walker)
# (does not work automatically for single chain)
sum(get_elapsed_time(naive_walker))
}

}
