% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/module3_ata.R
\name{ata}
\alias{ata}
\alias{ata.constraint}
\alias{ata.get.forms}
\alias{ata.get.items}
\alias{ata.item.enemy}
\alias{ata.item.fix}
\alias{ata.item.maxselect}
\alias{ata.obj.absolute}
\alias{ata.obj.coefficients}
\alias{ata.obj.relative}
\alias{ata.solve}
\alias{plot.ata}
\alias{print.ata}
\title{Automated Test Assembly (ATA)}
\usage{
ata(pool, nform = 1, len = NULL, maxselect = NULL, debug = FALSE)

\method{print}{ata}(x, ...)

\method{plot}{ata}(x, ...)

ata.get.forms(x, forms, collapse)

ata.obj.coefficients(x, coef, compensate)

ata.obj.relative(x, coef, mode, negative = FALSE, flatten = NULL,
  compensate = FALSE, forms = NULL, collapse = FALSE)

ata.obj.absolute(x, coef, target, compensate = FALSE, forms = NULL,
  collapse = FALSE)

ata.constraint(x, coef, min = NA, max = NA, level = NULL, forms = NULL,
  collapse = FALSE)

ata.item.maxselect(x, maxselect, items = NULL)

ata.item.enemy(x, items)

ata.item.fix(x, items, min, max, forms = NULL, collapse = FALSE)

ata.solve(x, ...)

ata.get.items(x, as.list = FALSE)
}
\arguments{
\item{pool}{a data frame of items}

\item{nform}{the number of forms to be assembled}

\item{len}{the test length}

\item{maxselect}{the maximal times of selection of an item}

\item{debug}{\code{TRUE} to turn on debug mode}

\item{x}{the ata object}

\item{...}{further arguments}

\item{forms}{the forms to which objectives are added. \code{NULL} for all forms}

\item{collapse}{\code{TRUE} to collapse all forms into one objective}

\item{coef}{the coefficients added to the LP}

\item{compensate}{\code{TRUE} when objective functions are compensatory to one another}

\item{mode}{the optimzation mode (i.e. 'max' for maximization or 'min' for minimization)}

\item{negative}{\code{TRUE} when the expected value of the objective function is negative}

\item{flatten}{the flatten parameter}

\item{target}{the targeted value of the objective function}

\item{min}{the minimum value of the constraint}

\item{max}{the maximum value of the constraint}

\item{level}{the level value for categorical variable}

\item{items}{a vector of item index}

\item{as.list}{\code{TRUE} to return a list, \code{FALSE} to return a data frame}
}
\value{
\code{ata} returns an \code{ata} object
}
\description{
\code{ata} creates an \code{ata} object

\code{ata.get.forms} is a helper function for converting input forms into actual form indeces in LP

\code{ata.obj.coefficients} is a helper function for processesing the input coefficients

\code{ata.obj.relative} adds relative (maximize/minimize) objectives to LP

\code{ata.obj.absolute} adds absolute objectives to LP

\code{ata.constraint} adds a constraint to LP

\code{ata.item.maxselect} sets the maximal times of selection for items

\code{ata.item.enemy} adds enemy item relationship to LP

\code{ata.fixitem} set a fixed value for items, e.g, 1 for selection and 0 for no selection

\code{ata.solve} solves the LP

\code{ata.get.items} extracts items using assembly results
}
\details{
An \code{ata} object contains an item pool (\code{pool}), a LP object (\code{lp}),
and several constants such as the number of forms (\code{nform}), the size of item pool(\code{nitem}), 
the size of LP object (\code{nlpitem}), etc. It would contain assembly results (\code{result}) 
if LP has been solved. The result is a data frame of binary code with 1 indicating selected 
(items in rows and forms in columns). Use \code{ata.get.items} to extract actual items from results. \cr

For the function \code{ata.obj.relative} and \code{ata.obj.absolute}, 
when \code{coef} is a pool-size numeric vector, coefficients are used directly.
When \code{coef} is a variable name in the pool, values of that variable are used as coefficients.
When \code{coef} is a numeric vector which is unequal to pool size, information at those theta points 
are used as coefficients.\cr
When the expected value of the objective function is negative, set the \code{negative} argument to \code{TRUE}.\cr
The \code{compensate} argument controls whether objective functions are compensatory. 
For example, the ATA job wants to maximize information at -0.5 and 0.5.
When \code{compensate} is \code{TRUE}, the LP assembles a test maximizing the sum of information at -0.5 and 0.5.
When \code{compensate} is \code{FALSE}, the LP assembles a test maximizing information at each point, but not necessarily a maxmized total.\cr
\code{ata.obj.relative} is to maximize or minimize the objectives. There are four scenarios.
(1) For a maximization job with postive expected value, maximize y while sum(x) - y >= 0 and <= F (flatten).
(2) For a maximization job with negative expected value, minimize y while sum(x) + y >= 0 and <= F.
(3) For a minimization job with postive expected value, minimize y while sum(x) + y <= 0 and >= F.
(4) For a minimization job with negative expected value, maximize y while sum(x) - y <= 0 and >= F.

\code{ata.obj.absolute} minimizes y while sum(x) + y >= target and sum(x) - y <= target. \cr

For \code{ata.constraint}, set \code{coef} to a variable name 
in the pool and \code{level} a level value of that variable to 
add a categorical constraint. Set \code{coef} to a variable name and
leave \code{level} to default value (\code{NULL} or \code{NA}) to add
a quantitative constraint. Set \code{coef} to a number or a vector to 
directly add a constraint.\cr

In \code{ata.solve}, the \code{...} are additional \code{lp.control.options}.
The result (\code{x$result}) is a data frame of binary code with 1 indicating selected 
(items in rows and forms in columns). Use \code{ata.get.items} to extract actual items.
}
\examples{
\dontrun{
# ex. 1: 2 forms, 10 items, maximize b parmaters
items <- irt.model(model="3pl")$gen.data(1, 100)$items
items$content <- sample(1:3, nrow(items), replace=TRUE)
items$time <- round(rlnorm(nrow(items), log(60), .2), 0)
x <- ata(items, 2, debug=TRUE)
x <- ata.obj.relative(x, "b", "max")
x <- ata.constraint(x, 1, 10, 10)
x <- ata.item.maxselect(x, 1)
x <- ata.solve(x)
plot(x)
y <- ata.get.items(x, as.list=TRUE)
mean(y[[1]]$b)
mean(y[[2]]$b)
# ex. 2: 2 forms, 10 items, minimize b parmaeters
x <- ata(items, 2, len=10, maxselect=1, debug=TRUE)
x <- ata.obj.relative(x, "b", "min", negative=TRUE)
x <- ata.solve(x)
plot(x)
y <- ata.get.items(x, as.list=TRUE)
mean(y[[1]]$b)
mean(y[[2]]$b)
# ex. 3: 2 forms, 10 items, maximize information at -0.5 and 0.5
# content distribution: 3, 3, 4; response time: avg. 55--65s
x <- ata(items, 2, len=10, maxselect=1, debug=TRUE)
x <- ata.obj.relative(x, c(-0.5, 0.5), "max")
x <- ata.constraint(x, "content", 3, 3, 1)
x <- ata.constraint(x, "content", 3, 3, 2)
x <- ata.constraint(x, "content", 4, 4, 3)
x <- ata.constraint(x, "time", 55*10, 65*10)
x <- ata.solve(x)
plot(x)
y <- ata.get.items(x, TRUE)
freq(y[[1]]$content, 1:3)$n
mean(y[[1]]$time)
freq(y[[2]]$content, 1:3)$n
mean(y[[2]]$time)
# ex. 4: 2 forms, 10 items, mean(b) = 0.5, sd(b) = 1.0, content = (3, 3, 4)
x <- ata(items, 2, len=10, maxselect=1, debug=TRUE)
x <- ata.obj.absolute(x, "b", 0.5 * 10)
x <- ata.obj.absolute(x, (x$pool$b - 0.5)^2, 1.0 * 10)
x <- ata.constraint(x, "content", 3, 3, 1)
x <- ata.constraint(x, "content", 3, 3, 2)
x <- ata.constraint(x, "content", 4, 4, 3)
x <- ata.solve(x)
plot(x)
y <- ata.get.items(x, TRUE)
c(mean(y[[1]]$b), sd(y[[1]]$b))
freq(y[[1]]$content, 1:3)$n
c(mean(y[[2]]$b), sd(y[[2]]$b))
freq(y[[2]]$content, 1:3)$n
# ex. 5: 2 forms, 10 items, flat TIF over [-1, 1]
x <- ata(items, 2, len=10, maxselect=1, debug=TRUE)
x <- ata.obj.relative(x, seq(-1, 1, .5), "max", negative=FALSE, flatten=.1)
x <- ata.solve(x)
y <- ata.get.items(x, TRUE)
plot(irt.model.3pl(items=y[[1]]), stats="information", total=TRUE)
plot(irt.model.3pl(items=y[[2]]), stats="information", total=TRUE)
}
}

