#' Derive the importance rank of all predictors used in the PNN 
#'
#' The function \code{pnn.imp} derives the importance rank of all predictors used in the PNN
#' It essentially is a wrapper around the function \code{pnn.x_imp}.
#'
#' @param net A PNN object generated by pnn.fit() 
#'
#' @return A dataframe with important values of all predictors in the PNN
#'
#' @seealso \code{\link{pnn.x_imp}}
#'
#' @examples
#' data(iris, package = "datasets")
#' Y <- iris[, 5]
#' X <- scale(iris[, 1:4])
#' pnet <- pnn.fit(x = X, y = Y)
#' \donttest{
#' pnn.imp(pnet)
#' }

pnn.imp <- function(net) {
  if (class(net) != "Probabilistic Neural Net") stop("net needs to be a PNN.", call. = F)

  cls <- parallel::makeCluster(min(ncol(net$x), parallel::detectCores() - 1), type = "PSOCK")
  obj <- c("logl", "net", "pnn.fit", "pnn.predone", "pnn.predict", "pnn.x_imp", "dummies")
  parallel::clusterExport(cls, obj,  envir = environment())
  rst1 <- data.frame(idx = seq(ncol(net$x)),
                     Reduce(rbind, 
                       parallel::parLapply(cls, seq(ncol(net$x)), 
                         function(i) pnn.x_imp(net, i))))
  parallel::stopCluster(cls)
  rst2 <- rst1[with(rst1, order(-imp1, -imp2)), ]
  row.names(rst2) <- NULL
  return(rst2)
}
