use core::slice;

use crate::*;
use libR_sys::*;

use super::{device_descriptor::*, Device, Raster, TextMetric};





























pub enum ClippingStrategy {
    Device,
    DeviceAndEngine,
    Engine,
}









#[allow(non_snake_case, unused_variables, clippy::too_many_arguments)]
pub trait DeviceDriver: std::marker::Sized {




    const USE_RASTER: bool = true;





    const USE_CAPTURE: bool = true;




    const USE_LOCATOR: bool = true;



    const USE_PLOT_HISTORY: bool = false;



    const CLIPPING_STRATEGY: ClippingStrategy = ClippingStrategy::DeviceAndEngine;



    const ACCEPT_UTF8_TEXT: bool = true;


    fn activate(&mut self, dd: DevDesc) {}











    fn circle(&mut self, center: (f64, f64), r: f64, gc: R_GE_gcontext, dd: DevDesc) {}


    fn clip(&mut self, from: (f64, f64), to: (f64, f64), dd: DevDesc) {}




    fn close(&mut self, dd: DevDesc) {}


    fn deactivate(&mut self, dd: DevDesc) {}


    fn line(&mut self, from: (f64, f64), to: (f64, f64), gc: R_GE_gcontext, dd: DevDesc) {}












    fn char_metric(&mut self, c: char, gc: R_GE_gcontext, dd: DevDesc) -> TextMetric {
        TextMetric {
            ascent: 0.0,
            descent: 0.0,
            width: 0.0,
        }
    }



    fn mode(&mut self, mode: i32, dd: DevDesc) {}


    fn new_page(&mut self, gc: R_GE_gcontext, dd: DevDesc) {}


    fn polygon<T: IntoIterator<Item = (f64, f64)>>(
        &mut self,
        coords: T,
        gc: R_GE_gcontext,
        dd: DevDesc,
    ) {
    }


    fn polyline<T: IntoIterator<Item = (f64, f64)>>(
        &mut self,
        coords: T,
        gc: R_GE_gcontext,
        dd: DevDesc,
    ) {
    }


    fn rect(&mut self, from: (f64, f64), to: (f64, f64), gc: R_GE_gcontext, dd: DevDesc) {}






    fn path<T: IntoIterator<Item = impl IntoIterator<Item = (f64, f64)>>>(
        &mut self,
        coords: T,
        winding: bool,
        gc: R_GE_gcontext,
        dd: DevDesc,
    ) {
    }







    fn raster<T: AsRef<[u32]>>(
        &mut self,
        raster: Raster<T>,
        pos: (f64, f64),
        size: (f64, f64),
        angle: f64,
        interpolate: bool,
        gc: R_GE_gcontext,
        dd: DevDesc,
    ) {
    }




    fn capture(&mut self, dd: DevDesc) -> Robj {
        ().into()
    }























    fn size(&mut self, dd: DevDesc) -> (f64, f64, f64, f64) {
        (dd.left, dd.right, dd.bottom, dd.top)
    }













    fn text_width(&mut self, text: &str, gc: R_GE_gcontext, dd: DevDesc) -> f64 {
        text.chars()
            .map(|c| self.char_metric(c, gc, dd).width)
            .sum()
    }





    fn text(
        &mut self,
        pos: (f64, f64),
        text: &str,
        angle: f64,
        hadj: f64,
        gc: R_GE_gcontext,
        dd: DevDesc,
    ) {
    }



    fn on_exit(&mut self, dd: DevDesc) {}



    fn new_frame_confirm(&mut self, dd: DevDesc) -> bool {
        true
    }





    fn holdflush(&mut self, dd: DevDesc, level: i32) -> i32 {
        0
    }


    fn locator(&mut self, x: *mut f64, y: *mut f64, dd: DevDesc) -> bool {
        true
    }






    fn eventHelper(&mut self, dd: DevDesc, code: i32) {}


    fn create_device<T: DeviceDriver>(
        self,
        device_descriptor: DeviceDescriptor,
        device_name: &'static str,
    ) -> Device {
        #![allow(non_snake_case)]
        #![allow(unused_variables)]
        use std::os::raw::{c_char, c_int, c_uint};






        unsafe {
            single_threaded(|| {

                R_GE_checkVersionOrDie(R_GE_version as _);


                R_CheckDeviceAvailable();
            });
        }





        unsafe extern "C" fn device_driver_activate<T: DeviceDriver>(arg1: pDevDesc) {








            let data = ((*arg1).deviceSpecific as *mut T).as_mut().unwrap();

            data.activate(*arg1);
        }

        unsafe extern "C" fn device_driver_circle<T: DeviceDriver>(
            x: f64,
            y: f64,
            r: f64,
            gc: pGEcontext,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.circle((x, y), r, *gc, *dd);
        }

        unsafe extern "C" fn device_driver_clip<T: DeviceDriver>(
            x0: f64,
            x1: f64,
            y0: f64,
            y1: f64,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.clip((x0, y0), (x1, y1), *dd);
        }




        unsafe extern "C" fn device_driver_close<T: DeviceDriver>(dd: pDevDesc) {
            let dev_desc = *dd;
            let data_ptr = dev_desc.deviceSpecific as *mut T;

            let mut data = Box::from_raw(data_ptr);

            data.close(dev_desc);
        }

        unsafe extern "C" fn device_driver_deactivate<T: DeviceDriver>(arg1: pDevDesc) {
            let mut data = ((*arg1).deviceSpecific as *mut T).read();
            data.deactivate(*arg1);
        }

        unsafe extern "C" fn device_driver_line<T: DeviceDriver>(
            x1: f64,
            y1: f64,
            x2: f64,
            y2: f64,
            gc: pGEcontext,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.line((x1, y1), (x2, y2), *gc, *dd);
        }

        unsafe extern "C" fn device_driver_char_metric<T: DeviceDriver>(
            c: c_int,
            gc: pGEcontext,
            ascent: *mut f64,
            descent: *mut f64,
            width: *mut f64,
            dd: pDevDesc,
        ) {












            if let Some(c) = std::char::from_u32(c.abs() as _) {
                let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
                let metric_info = data.char_metric(c, *gc, *dd);
                *ascent = metric_info.ascent;
                *descent = metric_info.descent;
                *width = metric_info.width;
            }
        }

        unsafe extern "C" fn device_driver_mode<T: DeviceDriver>(mode: c_int, dd: pDevDesc) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.mode(mode as _, *dd);
        }

        unsafe extern "C" fn device_driver_new_page<T: DeviceDriver>(gc: pGEcontext, dd: pDevDesc) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.new_page(*gc, *dd);
        }

        unsafe extern "C" fn device_driver_polygon<T: DeviceDriver>(
            n: c_int,
            x: *mut f64,
            y: *mut f64,
            gc: pGEcontext,
            dd: pDevDesc,
        ) {
            let x = slice::from_raw_parts(x, n as _).iter();
            let y = slice::from_raw_parts(y, n as _).iter();

            let coords = x.zip(y).map(|(&x, &y)| (x, y));

            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.polygon(coords, *gc, *dd);
        }

        unsafe extern "C" fn device_driver_polyline<T: DeviceDriver>(
            n: c_int,
            x: *mut f64,
            y: *mut f64,
            gc: pGEcontext,
            dd: pDevDesc,
        ) {
            let x = slice::from_raw_parts(x, n as _).iter();
            let y = slice::from_raw_parts(y, n as _).iter();

            let coords = x.zip(y).map(|(&x, &y)| (x, y));

            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.polyline(coords, *gc, *dd);
        }

        unsafe extern "C" fn device_driver_rect<T: DeviceDriver>(
            x0: f64,
            y0: f64,
            x1: f64,
            y1: f64,
            gc: pGEcontext,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.rect((x0, y0), (x1, y1), *gc, *dd);
        }

        unsafe extern "C" fn device_driver_path<T: DeviceDriver>(
            x: *mut f64,
            y: *mut f64,
            npoly: c_int,
            nper: *mut c_int,
            winding: Rboolean,
            gc: pGEcontext,
            dd: pDevDesc,
        ) {
            let nper = slice::from_raw_parts(nper, npoly as _);

            let n = nper.iter().sum::<i32>() as usize;
            let x = slice::from_raw_parts(x, n as _).iter();
            let y = slice::from_raw_parts(y, n as _).iter();

            let mut coords_flat = x.zip(y).map(|(&x, &y)| (x, y));

            let coords = nper.iter().map(|&np| {
                coords_flat
                    .by_ref()
                    .take(np as _)

                    .collect::<Vec<(f64, f64)>>()
            });

            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();



            let winding = winding != Rboolean::FALSE;

            data.path(coords, winding, *gc, *dd);
        }

        unsafe extern "C" fn device_driver_raster<T: DeviceDriver>(
            raster: *mut c_uint,
            w: c_int,
            h: c_int,
            x: f64,
            y: f64,
            width: f64,
            height: f64,
            rot: f64,
            interpolate: Rboolean,
            gc: pGEcontext,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            let raster = slice::from_raw_parts(raster, (w * h) as _);

            data.raster::<&[u32]>(
                Raster {
                    pixels: raster,
                    width: w as _,
                },
                (x, y),
                (width, height),
                rot,


                interpolate != Rboolean::FALSE,
                *gc,
                *dd,
            );
        }

        unsafe extern "C" fn device_driver_capture<T: DeviceDriver>(dd: pDevDesc) -> SEXP {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();

            data.capture(*dd).get()
        }

        unsafe extern "C" fn device_driver_size<T: DeviceDriver>(
            left: *mut f64,
            right: *mut f64,
            bottom: *mut f64,
            top: *mut f64,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            let sizes = data.size(*dd);
            *left = sizes.0;
            *right = sizes.1;
            *bottom = sizes.2;
            *top = sizes.3;
        }

        unsafe extern "C" fn device_driver_text_width<T: DeviceDriver>(
            str: *const c_char,
            gc: pGEcontext,
            dd: pDevDesc,
        ) -> f64 {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            let cstr = std::ffi::CStr::from_ptr(str);


            if let Ok(cstr) = cstr.to_str() {
                data.text_width(cstr, *gc, *dd)
            } else {
                0.0
            }
        }

        unsafe extern "C" fn device_driver_text<T: DeviceDriver>(
            x: f64,
            y: f64,
            str: *const c_char,
            rot: f64,
            hadj: f64,
            gc: pGEcontext,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            let cstr = std::ffi::CStr::from_ptr(str);


            if let Ok(cstr) = cstr.to_str() {
                data.text((x, y), cstr, rot, hadj, *gc, *dd);
            }
        }

        unsafe extern "C" fn device_driver_on_exit<T: DeviceDriver>(dd: pDevDesc) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.on_exit(*dd);
        }

        unsafe extern "C" fn device_driver_new_frame_confirm<T: DeviceDriver>(
            dd: pDevDesc,
        ) -> Rboolean {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.new_frame_confirm(*dd).into()
        }

        unsafe extern "C" fn device_driver_holdflush<T: DeviceDriver>(
            dd: pDevDesc,
            level: c_int,
        ) -> c_int {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.holdflush(*dd, level as _)
        }

        unsafe extern "C" fn device_driver_locator<T: DeviceDriver>(
            x: *mut f64,
            y: *mut f64,
            dd: pDevDesc,
        ) -> Rboolean {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();
            data.locator(x, y, *dd).into()
        }

        unsafe extern "C" fn device_driver_eventHelper<T: DeviceDriver>(dd: pDevDesc, code: c_int) {
            let mut data = ((*dd).deviceSpecific as *mut T).read();
            data.eventHelper(*dd, code);
        }

        #[cfg(use_r_ge_version_14)]
        unsafe extern "C" fn device_driver_setPattern<T: DeviceDriver>(
            pattern: SEXP,
            dd: pDevDesc,
        ) -> SEXP {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();


            R_NilValue
        }

        #[cfg(use_r_ge_version_14)]
        unsafe extern "C" fn device_driver_releasePattern<T: DeviceDriver>(
            ref_: SEXP,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();


        }

        #[cfg(use_r_ge_version_14)]
        unsafe extern "C" fn device_driver_setClipPath<T: DeviceDriver>(
            path: SEXP,
            ref_: SEXP,
            dd: pDevDesc,
        ) -> SEXP {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();


            R_NilValue
        }

        #[cfg(use_r_ge_version_14)]
        unsafe extern "C" fn device_driver_releaseClipPath<T: DeviceDriver>(
            ref_: SEXP,
            dd: pDevDesc,
        ) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();


        }

        #[cfg(use_r_ge_version_14)]
        unsafe extern "C" fn device_driver_setMask<T: DeviceDriver>(
            path: SEXP,
            ref_: SEXP,
            dd: pDevDesc,
        ) -> SEXP {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();


            R_NilValue
        }

        #[cfg(use_r_ge_version_14)]
        unsafe extern "C" fn device_driver_releaseMask<T: DeviceDriver>(ref_: SEXP, dd: pDevDesc) {
            let data = ((*dd).deviceSpecific as *mut T).as_mut().unwrap();


        }













        let deviceSpecific = Box::into_raw(Box::new(self)) as *mut std::os::raw::c_void;























        let p_dev_desc = unsafe { libc::calloc(1, std::mem::size_of::<DevDesc>()) as *mut DevDesc };

        unsafe {
            (*p_dev_desc).left = device_descriptor.left;
            (*p_dev_desc).right = device_descriptor.right;
            (*p_dev_desc).bottom = device_descriptor.bottom;
            (*p_dev_desc).top = device_descriptor.top;


            (*p_dev_desc).clipLeft = device_descriptor.left;
            (*p_dev_desc).clipRight = device_descriptor.right;
            (*p_dev_desc).clipBottom = device_descriptor.bottom;
            (*p_dev_desc).clipTop = device_descriptor.top;




            (*p_dev_desc).xCharOffset = 0.4900;
            (*p_dev_desc).yCharOffset = 0.3333;
            (*p_dev_desc).yLineBias = 0.2;

            (*p_dev_desc).ipr = device_descriptor.ipr;
            (*p_dev_desc).cra = device_descriptor.cra;







            (*p_dev_desc).gamma = 1.0;

            (*p_dev_desc).canClip = match <T>::CLIPPING_STRATEGY {
                ClippingStrategy::Engine => Rboolean::FALSE,
                _ => Rboolean::TRUE,
            };


            (*p_dev_desc).canChangeGamma = Rboolean::FALSE;

            (*p_dev_desc).canHAdj = CanHAdjOption::VariableAdjustment as _;

            (*p_dev_desc).startps = device_descriptor.startps;
            (*p_dev_desc).startcol = device_descriptor.startcol.to_i32();
            (*p_dev_desc).startfill = device_descriptor.startfill.to_i32();
            (*p_dev_desc).startlty = device_descriptor.startlty.to_i32();
            (*p_dev_desc).startfont = device_descriptor.startfont.to_i32();

            (*p_dev_desc).startgamma = 1.0;


            (*p_dev_desc).deviceSpecific = deviceSpecific;

            (*p_dev_desc).displayListOn = <T>::USE_PLOT_HISTORY.into();


            (*p_dev_desc).canGenMouseDown = Rboolean::FALSE;
            (*p_dev_desc).canGenMouseMove = Rboolean::FALSE;
            (*p_dev_desc).canGenMouseUp = Rboolean::FALSE;
            (*p_dev_desc).canGenKeybd = Rboolean::FALSE;
            (*p_dev_desc).canGenIdle = Rboolean::FALSE;







            (*p_dev_desc).gettingEvent = Rboolean::FALSE;

            (*p_dev_desc).activate = Some(device_driver_activate::<T>);
            (*p_dev_desc).circle = Some(device_driver_circle::<T>);
            (*p_dev_desc).clip = match <T>::CLIPPING_STRATEGY {
                ClippingStrategy::Engine => None,
                _ => Some(device_driver_clip::<T>),
            };
            (*p_dev_desc).close = Some(device_driver_close::<T>);
            (*p_dev_desc).deactivate = Some(device_driver_deactivate::<T>);
            (*p_dev_desc).locator = Some(device_driver_locator::<T>); // TOD;
            (*p_dev_desc).line = Some(device_driver_line::<T>);
            (*p_dev_desc).metricInfo = Some(device_driver_char_metric::<T>);
            (*p_dev_desc).mode = Some(device_driver_mode::<T>);
            (*p_dev_desc).newPage = Some(device_driver_new_page::<T>);
            (*p_dev_desc).polygon = Some(device_driver_polygon::<T>);
            (*p_dev_desc).polyline = Some(device_driver_polyline::<T>);
            (*p_dev_desc).rect = Some(device_driver_rect::<T>);
            (*p_dev_desc).path = Some(device_driver_path::<T>);
            (*p_dev_desc).raster = if <T>::USE_RASTER {
                Some(device_driver_raster::<T>)
            } else {
                None
            };
            (*p_dev_desc).cap = if <T>::USE_CAPTURE {
                Some(device_driver_capture::<T>)
            } else {
                None
            };
            (*p_dev_desc).size = Some(device_driver_size::<T>);
            (*p_dev_desc).strWidth = Some(device_driver_text_width::<T>);
            (*p_dev_desc).text = Some(device_driver_text::<T>);
            (*p_dev_desc).onExit = Some(device_driver_on_exit::<T>);



            (*p_dev_desc).getEvent = None;

            (*p_dev_desc).newFrameConfirm = Some(device_driver_new_frame_confirm::<T>);


            (*p_dev_desc).hasTextUTF8 = <T>::ACCEPT_UTF8_TEXT.into();
            (*p_dev_desc).textUTF8 = if <T>::ACCEPT_UTF8_TEXT {
                Some(device_driver_text::<T>)
            } else {
                None
            };
            (*p_dev_desc).strWidthUTF8 = if <T>::ACCEPT_UTF8_TEXT {
                Some(device_driver_text_width::<T>)
            } else {
                None
            };
            (*p_dev_desc).wantSymbolUTF8 = <T>::ACCEPT_UTF8_TEXT.into();









            (*p_dev_desc).useRotatedTextInContour = Rboolean::FALSE;

            (*p_dev_desc).eventEnv = empty_env().get();
            (*p_dev_desc).eventHelper = Some(device_driver_eventHelper::<T>);

            (*p_dev_desc).holdflush = Some(device_driver_holdflush::<T>);


            (*p_dev_desc).haveTransparency = DevCapTransparency::Yes as _;
            (*p_dev_desc).haveTransparentBg = DevCapTransparentBg::Fully as _;




            (*p_dev_desc).haveRaster = if <T>::USE_RASTER {
                DevCapRaster::Yes as _
            } else {
                DevCapRaster::No as _
            };

            (*p_dev_desc).haveCapture = if <T>::USE_CAPTURE {
                DevCapCapture::Yes as _
            } else {
                DevCapCapture::No as _
            };

            (*p_dev_desc).haveLocator = if <T>::USE_LOCATOR {
                DevCapLocator::Yes as _
            } else {
                DevCapLocator::No as _
            };





            #[cfg(use_r_ge_version_14)]
            {
                (*p_dev_desc).setPattern = Some(device_driver_setPattern::<T>);
                (*p_dev_desc).releasePattern = Some(device_driver_releasePattern::<T>);

                (*p_dev_desc).setClipPath = Some(device_driver_setClipPath::<T>);
                (*p_dev_desc).releaseClipPath = Some(device_driver_releaseClipPath::<T>);

                (*p_dev_desc).setMask = Some(device_driver_setMask::<T>);
                (*p_dev_desc).releaseMask = Some(device_driver_releaseMask::<T>);

                (*p_dev_desc).deviceVersion = R_GE_definitions as _;

                (*p_dev_desc).deviceClip = match <T>::CLIPPING_STRATEGY {
                    ClippingStrategy::Device => Rboolean::TRUE,
                    _ => Rboolean::FALSE,
                };
            }

            #[cfg(use_r_ge_version_15)]
            {
                (*p_dev_desc).defineGroup = None;
                (*p_dev_desc).useGroup = None;
                (*p_dev_desc).releaseGroup = None;

                (*p_dev_desc).stroke = None;
                (*p_dev_desc).fill = None;
                (*p_dev_desc).fillStroke = None;

                (*p_dev_desc).capabilities = None;
            }
        } // unsafe ends here

        let device_name = CString::new(device_name).unwrap();

        single_threaded(|| unsafe {
            let device = GEcreateDevDesc(p_dev_desc);




            GEaddDevice2(device, device_name.as_ptr() as *mut i8);
            GEinitDisplayList(device);

            Device { inner: device }
        })
    }
}
