% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcea.R, R/bcea.default.R
\name{bcea}
\alias{bcea}
\alias{bcea.default}
\alias{bcea.rjags}
\alias{bcea.rstan}
\alias{bcea.bugs}
\title{Create Bayesian Cost-Effectiveness Analysis Object}
\usage{
bcea(
  eff,
  cost,
  ref = 1,
  interventions = NULL,
  .comparison = NULL,
  Kmax = 50000,
  k = NULL,
  plot = FALSE,
  ...
)

\method{bcea}{default}(
  eff,
  cost,
  ref = NULL,
  interventions = NULL,
  .comparison = NULL,
  Kmax = 50000,
  k = NULL,
  plot = FALSE,
  ...
)

\method{bcea}{rjags}(eff, ...)

\method{bcea}{rstan}(eff, ...)

\method{bcea}{bugs}(eff, ...)
}
\arguments{
\item{eff}{An object containing \code{nsim} simulations for the variable of
clinical effectiveness for each intervention being considered. In general it
is a matrix with \code{nsim} rows and \code{nint} columns.
This are partially matched with \verb{e' from previous version of }BCEA` for back-compatibility.}

\item{cost}{An object containing \code{nsim} simulations for the variable of
cost for each intervention being considered. In general it is a matrix with
\code{nsim} rows and \code{nint} columns.
This are partially matched with \verb{c' from previous version of }BCEA` for back-compatibility.}

\item{ref}{Defines which intervention (columns of \code{eff} or \code{cost}) is
considered to be the reference strategy. The default value \code{ref = 1}
means that the intervention associated with the first column of \code{eff} or
\code{cost} is the reference and the one(s) associated with the other column(s)
is(are) the comparators.}

\item{interventions}{Defines the labels to be associated with each
intervention. By default and if \code{NULL}, assigns labels in the form
"Intervention1", ... , "InterventionT".}

\item{.comparison}{Selects the comparator, in case of more than two
interventions being analysed. Default as NULL plots all the comparisons
together. Any subset of the possible comparisons can be selected (e.g.,
\code{comparison=c(1,3)} or \code{comparison = 2}).}

\item{Kmax}{Maximum value of the willingness to pay to be considered.
Default value is \code{k = 50000}. The willingness to pay is then approximated
on a discrete grid in the interval \verb{[0, Kmax]}. The grid is equal to
\code{k} if the parameter is given, or composed of \code{501} elements if
\code{k = NULL} (the default).}

\item{k}{A(n optional) vector for the values of the willingness
to pay grid. Should be of length > 1 otherwise plots will be empty.
If not specified then BCEA will construct a grid of 501 values
from 0 to \code{Kmax}. This option is useful when performing intensive
computations (e.g. for the EVPPI). This was changed from \code{wtp} in previous versions
for consistency with other functions and so will be deprecated in the future.}

\item{plot}{A logical value indicating whether the function should produce
the summary plot or not.}

\item{...}{Additional arguments}
}
\value{
An object of the class \code{bcea} containing the following elements:
\itemize{
\item \strong{n_sim}: Number of simulations produced by the Bayesian model.
\item \strong{n_comparators}: Number of interventions being analyzed.
\item \strong{n_comparisons}: Number of possible pairwise comparisons.
\item \strong{delta.e}: For each possible comparison, the differential in the effectiveness measure.
\item \strong{delta.c}: For each possible comparison, the differential in the cost measure.
\item \strong{ICER}: The value of the Incremental Cost-Effectiveness Ratio.
\item \strong{Kmax}: The maximum value assumed for the willingness to pay threshold.
\item \strong{k}: The vector of values for the grid approximation of the willingness to pay.
\item \strong{ceac}: The value for the Cost-Effectiveness Acceptability Curve, as a function of the willingness to pay.
\item \strong{ib}: The distribution of the Incremental Benefit, for a given willingness to pay.
\item \strong{eib}: The value for the Expected Incremental Benefit, as a function of the willingness to pay.
\item \strong{kstar}: The grid approximation of the break-even point(s).
\item \strong{best}: A vector containing the numeric label of the intervention that is the most cost-effective for each value of the willingness to pay in the selected grid approximation.
\item \strong{U}: An array including the value of the expected utility for each simulation from the Bayesian model, for each value of the grid approximation of the willingness to pay, and for each intervention being considered.
\item \strong{vi}: An array including the value of information for each simulation from the Bayesian model and for each value of the grid approximation of the willingness to pay.
\item \strong{Ustar}: An array including the maximum "known-distribution" utility for each simulation from the Bayesian model and for each value of the grid approximation of the willingness to pay.
\item \strong{ol}: An array including the opportunity loss for each simulation from the Bayesian model and for each value of the grid approximation of the willingness to pay.
\item \strong{evi}: The vector of values for the Expected Value of Information, as a function of the willingness to pay.
\item \strong{interventions}: A vector of labels for all the interventions considered.
\item \strong{ref}: The numeric index associated with the intervention used as reference in the analysis.
\item \strong{comp}: The numeric index(es) associated with the intervention(s) used as comparator(s) in the analysis.
\item \strong{step}: The step size used to form the grid approximation to the willingness to pay.
\item \strong{e}: The \code{eff} matrix used to generate the object (see Arguments).
\item \strong{c}: The \code{cost} matrix used to generate the object (see Arguments).
}
}
\description{
Cost-effectiveness analysis based on the results of a simulation model for a
variable of clinical benefits (e) and of costs (c). Produces results to be
post-processed to give the health economic analysis. The output is stored in
an object of the class "bcea".
}
\examples{
# See Baio (2013), Baio (2011) for a detailed description of the 
# Bayesian model and economic problem

# Load the processed results of the MCMC simulation model
data(Vaccine)

# Runs the health economic evaluation using BCEA
m <- bcea(
      e=eff,
      c=cost,               # defines the variables of 
                            #  effectiveness and cost
      ref=2,                # selects the 2nd row of (e, c) 
                            #  as containing the reference intervention
      interventions=treats, # defines the labels to be associated 
                            #  with each intervention
      Kmax=50000,           # maximum value possible for the willingness 
                            #  to pay threshold; implies that k is chosen 
                            #  in a grid from the interval (0, Kmax)
      plot=TRUE             # plots the results
)

# Creates a summary table
summary(
      m,         # uses the results of the economic evaluation 
                 #  (a "bcea" object)
      wtp=25000	# selects the particular value for k 
)

\donttest{

# Plots the cost-effectiveness plane using base graphics
ceplane.plot(
      m,             # plots the Cost-Effectiveness plane
      comparison=1,  # if more than 2 interventions, selects the
                     #  pairwise comparison 
      wtp=25000,     # selects the relevant willingness to pay 
                     #  (default: 25,000) 
      graph="base"   # selects base graphics (default)
)

# Plots the cost-effectiveness plane using ggplot2
if (requireNamespace("ggplot2")) {
ceplane.plot(
      m,             # plots the Cost-Effectiveness plane
      comparison=1,  # if more than 2 interventions, selects the
                     #  pairwise comparison 
      wtp=25000,     # selects the relevant willingness to pay 
                     #  (default: 25,000) 
      graph="ggplot2"# selects ggplot2 as the graphical engine
)

# Some more options
ceplane.plot(
      m,
      graph="ggplot2",
      pos="top",
      size=5,
      ICER_size=1.5,
      label.pos=FALSE,
      opt.theme=ggplot2::theme(text=ggplot2::element_text(size=8))
)
}

# Plots the contour and scatterplot of the bivariate 
# distribution of (Delta_e,Delta_c)
contour(
      m,            # uses the results of the economic evaluation 
                    #  (a "bcea" object)
      comparison=1, # if more than 2 interventions, selects the 
                    #  pairwise comparison 
      nlevels=4,    # selects the number of levels to be 
                    #  plotted (default=4)
      levels=NULL,  # specifies the actual levels to be plotted 
                    #  (default=NULL, so that R will decide)
      scale=0.5,    # scales the bandwidths for both x- and 
                    #  y-axis (default=0.5)
      graph="base"  # uses base graphics to produce the plot
)

# Plots the contour and scatterplot of the bivariate 
#   distribution of (Delta_e,Delta_c)
contour2(
      m,          # uses the results of the economic evaluation 
                  #  (a "bcea" object)
      wtp=25000,  # selects the willingness-to-pay threshold
)

# Using ggplot2
if (requireNamespace("ggplot2")) {
contour2(
      m,              # uses the results of the economic evaluation 
                      #  (a "bcea" object)
      graph="ggplot2",# selects the graphical engine
      wtp=25000,      # selects the willingness-to-pay threshold
      label.pos=FALSE # alternative position for the wtp label
)
}

# Plots the Expected Incremental Benefit for the "bcea" object m
eib.plot(m)

# Plots the distribution of the Incremental Benefit
ib.plot(
    m,            # uses the results of the economic evaluation 
                  #  (a "bcea" object)
    comparison=1, # if more than 2 interventions, selects the 
                  #  pairwise comparison 
    wtp=25000,    # selects the relevant willingness 
                  #  to pay (default: 25,000)
    graph="base"  # uses base graphics
)

# Produces a plot of the CEAC against a grid of values for the 
# willingness to pay threshold
ceac.plot(m)

# Plots the Expected Value of Information for the "bcea" object m
evi.plot(m)
}

}
\references{
\insertRef{Baio2013}{BCEA}

\insertRef{Baio2011}{BCEA}
}
\author{
Gianluca Baio, Andrea Berardi, Nathan Green
}
\keyword{manip}
