% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wms_all.R
\name{catr_wms_get_layer}
\alias{catr_wms_get_layer}
\title{WMS INSPIRE: Download map images}
\usage{
catr_wms_get_layer(
  x,
  srs,
  what = "building",
  styles = "default",
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  crop = FALSE,
  ...
)
}
\arguments{
\item{x}{See \strong{Details}. It could be:
\itemize{
\item A numeric vector of length 4 with the coordinates that defines
the bounding box: \code{c(xmin, ymin, xmax, ymax)}
\item A \code{sf/sfc} object, as provided by the \strong{sf} package.
}}

\item{srs}{SRS/CRS to use on the query. To check the admitted values check
\link{catr_srs_values}, specifically the \code{wfs_service} column. See \strong{Details}.}

\item{what}{Layer to be extracted. Possible values are \code{"building"},
\code{"parcel"}, \code{"zoning"}, \code{"address"}. See \strong{Details}.}

\item{styles}{Style of the WMS layer. See \strong{Details}.}

\item{update_cache}{A logical whether to update cache. Default is \code{FALSE}.
When set to \code{TRUE} it would force a fresh download of the source file.}

\item{cache_dir}{A path to a cache directory. On missing value the function
would store the cached files on a temporary dir (See \code{\link[base:tempfile]{base::tempdir()}}).}

\item{verbose}{Logical, displays information. Useful for debugging,
default is \code{FALSE}.}

\item{crop}{\code{TRUE} if results should be cropped to the specified \code{x} extent,
\code{FALSE} otherwise. If \code{x} is an \code{sf} object with one \code{POINT}, crop is set
to \code{FALSE}.}

\item{...}{
  Arguments passed on to \code{\link[mapSpain:esp_getTiles]{mapSpain::esp_getTiles}}
  \describe{
    \item{\code{res}}{Resolution (in pixels) of the final tile. Only valid for WMS.}
    \item{\code{bbox_expand}}{A numeric value that indicates the expansion percentage
of the bounding box of \code{x}.}
    \item{\code{transparent}}{Logical. Provides transparent background, if supported.
Depends on the selected provider on \code{type}.}
    \item{\code{mask}}{\code{TRUE} if the result should be masked to \code{x}.}
  }}
}
\value{
A \code{SpatRaster} is returned, with 3 (RGB) or 4 (RGBA) layers. See
\code{\link[terra:rast]{terra::rast()}}.
}
\description{
Get geotagged images from the Spanish Cadastre. This function is a wrapper of
\code{\link[mapSpain:esp_getTiles]{mapSpain::esp_getTiles()}}.
}
\details{
When \code{x} is a numeric vector, make sure that the \code{srs} matches the
coordinate values. When \code{x} is a \code{sf} object, the value \code{srs} is ignored.

The query is performed using \href{https://epsg.io/3857}{EPSG:3857} (Web Mercator)
and the spatial object is projected back to the SRS of the initial object. In
case that the tile looks deformed, try either providing \code{bbox} on EPSG:3857
or projecting your \code{sf} object to \code{sf::st_crs(3857)}.
}
\section{Layers}{
The parameter \code{what} defines the layer to be extracted. The equivalence with
the
\href{https://www.catastro.minhap.es/webinspire/documentos/inspire-WMS.pdf}{API Docs}
equivalence is:
\itemize{
\item "parcel": CP.CadastralParcel
\item "zoning": CP.CadastralZoning
\item "building": BU.Building
\item "buildingpart": BU.BuildingPart
\item "address": AD.Address
\item "admboundary": AU.AdministrativeBoundary
\item "admunit": AU.AdministrativeUnit
}
}

\section{Styles}{
The WMS service provide different styles on each layer (\code{what} parameter).
Some of the styles available are:
\itemize{
\item "parcel": styles : \code{"BoundariesOnly"}, \code{"ReferencePointOnly"},
\code{"ELFCadastre"}.
\item "zoning": styles : \code{"BoundariesOnly"}, \code{"ELFCadastre"}.
\item "building" and "buildingpart": \code{"ELFCadastre"}
\item "address": \code{"Number.ELFCadastre"}
\item "admboundary" y "admunit": \code{"ELFCadastre"}
}

Check the
\href{https://www.catastro.minhap.es/webinspire/documentos/inspire-WMS.pdf}{API Docs}
for more information.
}

\examples{
\dontshow{if (tolower(Sys.info()[["sysname"]]) != "linux") (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{

# With a bbox

pict <- catr_wms_get_layer(c(222500, 4019500, 223700, 4020700),
  srs = 25830,
  what = "parcel"
)

library(mapSpain)
library(ggplot2)
library(tidyterra)

ggplot() +
  geom_spatraster_rgb(data = pict)


# With a spatial object

parcels <- catr_wfs_get_parcels_neigh_parcel("3662303TF3136B")

# Transform
parcels <- sf::st_transform(parcels, 3857)

# Use styles

parcels_img <- catr_wms_get_layer(parcels,
  what = "buildingpart",
  bbox_expand = 0.3,
  styles = "ELFCadastre"
)


ggplot() +
  geom_sf(data = parcels, fill = "blue", alpha = 0.5) +
  geom_spatraster_rgb(data = parcels_img)
}
\dontshow{\}) # examplesIf}
}
\references{
\href{https://www.catastro.minhap.es/webinspire/documentos/inspire-WMS.pdf}{API Documentation}

\href{https://www.catastro.minhap.es/webinspire/index.html}{INSPIRE Services for Cadastral Cartography}
}
\seealso{
\code{\link[mapSpain:esp_getTiles]{mapSpain::esp_getTiles()}}, \code{\link[tidyterra:geom_spatraster_rgb]{tidyterra::geom_spatraster_rgb()}},
\code{\link[terra:plotRGB]{terra::plotRGB()}}.

INSPIRE API functions:
\code{\link{catr_atom_get_address_db_all}()},
\code{\link{catr_atom_get_address}()},
\code{\link{catr_atom_get_buildings_db_all}()},
\code{\link{catr_atom_get_buildings}()},
\code{\link{catr_atom_get_parcels_db_all}()},
\code{\link{catr_atom_get_parcels}()},
\code{\link{catr_wfs_get_address_bbox}()},
\code{\link{catr_wfs_get_buildings_bbox}()},
\code{\link{catr_wfs_get_parcels_bbox}()}

Other spatial: 
\code{\link{catr_atom_get_address}()},
\code{\link{catr_atom_get_buildings}()},
\code{\link{catr_atom_get_parcels}()},
\code{\link{catr_wfs_get_address_bbox}()},
\code{\link{catr_wfs_get_buildings_bbox}()},
\code{\link{catr_wfs_get_parcels_bbox}()}
}
\concept{INSPIRE}
\concept{WMS}
\concept{spatial}
